<?php
declare(strict_types=1);

use Mpdf\Mpdf;

require_once dirname(__DIR__, 2) . '/vendor/autoload.php';
require_once dirname(__DIR__, 2) . '/config/config.php';
require_once dirname(__DIR__, 2) . '/config/db.php';

header('Content-Type: application/pdf');

function fail(int $code, string $msg){ http_response_code($code); echo $msg; exit; }

try {
  $embarqueId = isset($_GET['embarque_id']) ? (int)$_GET['embarque_id'] : 0;
  if ($embarqueId <= 0) fail(400, 'embarque_id requerido');

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // Header del embarque
  $st = $pdo->prepare("SELECT e.*, ee.code AS estado, d.code AS deposito_code FROM so_embarque e JOIN so_embarque_estado ee ON ee.id=e.estado_id JOIN wh_deposito d ON d.id=e.deposito_id WHERE e.id=? LIMIT 1");
  $st->execute([$embarqueId]);
  $e = $st->fetch(PDO::FETCH_ASSOC);
  if (!$e) fail(404, 'No existe el embarque');

  // PREs asociados
  $stp = $pdo->prepare("SELECT pre.id, pre.codigo FROM so_embarque_pre ep JOIN so_preembarque pre ON pre.id=ep.preembarque_id WHERE ep.embarque_id=? ORDER BY pre.id");
  $stp->execute([$embarqueId]);
  $pres = $stp->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Cargar filas (movimientos planificados de picking) por cada PRE
  $sqlRows = "
    SELECT sp.id AS pick_id,
           d.id AS pedido_dest_id,
           d.destinatario_id,
           dest.razon_social AS destinatario,
           i.producto_id,
           pr.denominacion AS producto,
           l.codigo AS lote,
           l.fecha_vencimiento AS vto,
           pos.code AS from_pos,
           pos2.code AS to_pos,
           sp.pallet_id,
           sp.uv_cajas,
           sp.uc_unidades
      FROM so_pre_pick sp
      JOIN so_pedido_dest_item i ON i.id=sp.pedido_dest_item_id
      JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
      JOIN para_destinatarios dest ON dest.id=d.destinatario_id
      JOIN para_productos pr ON pr.id=i.producto_id
      LEFT JOIN wh_lote l ON l.id=sp.lote_id
      LEFT JOIN wh_posicion pos ON pos.id=sp.from_pos_id
      LEFT JOIN wh_posicion pos2 ON pos2.id=sp.to_pos_id
     WHERE sp.preembarque_id=?
     ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC, sp.from_pos_id ASC, pr.denominacion ASC
  ";

  $rows = [];
  $stR = $pdo->prepare($sqlRows);
  foreach ($pres as $p) {
    $stR->execute([(int)$p['id']]);
    $part = $stR->fetchAll(PDO::FETCH_ASSOC) ?: [];
    foreach ($part as $r) { $rows[] = $r; }
  }

  // Configurar mPDF (A4 horizontal)
  $mpdf = new Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 35,
    'margin_bottom' => 15,
  ]);

  // Encabezado repetible similar a la planilla de báscula
  $header = '<div style="border-bottom:2px solid #000; font-size:11px;">
    <table width="100%" cellpadding="0" cellspacing="0">
      <tr>
        <td style="font-weight:bold; font-size:16px;">Planilla de Salida</td>
        <td align="right">Fecha impresión: '.date('Y-m-d H:i').'</td>
      </tr>
      <tr>
        <td>
          Embarque: <b>'.htmlspecialchars((string)$e['codigo']).'</b>
          &nbsp;|&nbsp; Depósito: <b>'.htmlspecialchars((string)$e['deposito_code']).'</b>
          &nbsp;|&nbsp; Estado: <b>'.htmlspecialchars((string)$e['estado']).'</b>
        </td>
        <td align="right">Página {PAGENO}/{nbpg}</td>
      </tr>
      <tr>
        <td>
          Móvil: <b>'.htmlspecialchars((string)$e['movil_id'] ?? '').'</b>
          &nbsp;|&nbsp; Chofer: <b>'.htmlspecialchars((string)$e['chofer_id'] ?? '').'</b>
        </td>
        <td align="right">Hora: _____________</td>
      </tr>
      <tr>
        <td colspan="2">
          Llegada: '.htmlspecialchars((string)($e['llegada_at'] ?? '')).'
          &nbsp;|&nbsp; Inicio carga: '.htmlspecialchars((string)($e['carga_inicio_at'] ?? '')).'
          &nbsp;|&nbsp; Fin carga: '.htmlspecialchars((string)($e['carga_fin_at'] ?? '')).'
          &nbsp;|&nbsp; Salida: '.htmlspecialchars((string)($e['salida_at'] ?? '')).'
          &nbsp;|&nbsp; Ticket Portería: '.htmlspecialchars((string)($e['ticket_porteria'] ?? '')).'
          &nbsp;|&nbsp; Ticket Báscula: '.htmlspecialchars((string)($e['ticket_bascula'] ?? '')).'
        </td>
      </tr>
    </table>
  </div>';
  $mpdf->SetHTMLHeader($header);

  $css = '<style>
    body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
    table.tbl { width: 100%; border-collapse: collapse; }
    table.tbl th, table.tbl td { border: 1px solid #000; padding: 4px 5px; }
    table.tbl th { background: #eee; font-weight: bold; }
    .right { text-align: right; }
    .signatures { width:100%; margin-top: 16px; }
    .sig { width:32%; text-align:center; }
    .sig .line { border-top:1px solid #000; margin-top:32px; }
    .section-title { font-weight: bold; margin-top: 14px; margin-bottom: 6px; }
  </style>';

  $html = $css;
  // Tabla principal (sin columnas Desde / → PREP)
  $html .= '<table class="tbl"><thead><tr>
    <th style="width:28px;">#</th>
    <th>Destinatario</th>
    <th>Producto</th>
    <th style="width:110px;">Lote</th>
    <th style="width:80px;">Vto</th>
    <th style="width:60px;" class="right">UV</th>
    <th style="width:60px;" class="right">UC</th>
  </tr></thead><tbody>';

  $i=0; $totUv=0; $totUc=0;
  foreach ($rows as $r) {
    $i++;
    $uv = (int)($r['uv_cajas'] ?? 0); $uc = (int)($r['uc_unidades'] ?? 0);
    $totUv += $uv; $totUc += $uc;
    $html .= '<tr>'
      .'<td>'. $i .'</td>'
      .'<td>'. htmlspecialchars((string)($r['destinatario'] ?? '')) .'</td>'
      .'<td>'. htmlspecialchars((string)($r['producto'] ?? '')) .'</td>'
      .'<td>'. htmlspecialchars((string)($r['lote'] ?? '')) .'</td>'
      .'<td>'. htmlspecialchars((string)($r['vto'] ?? '')) .'</td>'
      .'<td class="right">'. $uv .'</td>'
    .'<td class="right">'. $uc .'</td>'
      .'</tr>';
  }

  // Totales
  $html .= '<tr>'
    .'<td colspan="5" class="right" style="font-weight:bold;">Totales</td>'
    .'<td class="right" style="font-weight:bold;">'. $totUv .'</td>'
    .'<td class="right" style="font-weight:bold;">'. $totUc .'</td>'
    .'</tr>';

  $html .= '</tbody></table>';

  // Subtotales por destino y por producto + facturas
  $byDest = [];
  $byProd = [];
  foreach ($rows as $r) {
    $destId = (int)($r['destinatario_id'] ?? 0);
    $dest   = (string)($r['destinatario'] ?? '');
    $prod   = (string)($r['producto'] ?? '');
    $uv = (int)($r['uv_cajas'] ?? 0); $uc = (int)($r['uc_unidades'] ?? 0);
    if (!isset($byDest[$destId])) $byDest[$destId] = ['dest'=>$dest,'uv'=>0,'uc'=>0,'pedido_dest_ids'=>[]];
    $byDest[$destId]['uv'] += $uv; $byDest[$destId]['uc'] += $uc;
    $pdid = (int)($r['pedido_dest_id'] ?? 0);
    if ($pdid) $byDest[$destId]['pedido_dest_ids'][$pdid] = true;
    if (!isset($byProd[$prod])) $byProd[$prod] = ['uv'=>0,'uc'=>0];
    $byProd[$prod]['uv'] += $uv; $byProd[$prod]['uc'] += $uc;
  }

  // Helpers de introspección
  $tableExists = function(PDO $pdo, string $table): bool {
    $q = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $q->execute([$table]); return ((int)$q->fetchColumn()) > 0;
  };
  $columnExists = function(PDO $pdo, string $table, string $col): bool {
    $q = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
    $q->execute([$table,$col]); return ((int)$q->fetchColumn()) > 0;
  };

  // Obtener facturas por pedido_dest (best-effort)
  $invoicesByPedidoDest = [];
  $pedidoDestAll = [];
  foreach ($byDest as $d) { foreach (array_keys($d['pedido_dest_ids']) as $pid) { if ($pid) $pedidoDestAll[$pid] = true; } }
  $pedidoDestIds = array_keys($pedidoDestAll);
  if (!empty($pedidoDestIds)) {
    if ($tableExists($pdo,'so_pedido_dest_doc') && $columnExists($pdo,'so_pedido_dest_doc','pedido_dest_id')) {
      $place = implode(',', array_fill(0, count($pedidoDestIds), '?'));
      $sql = "SELECT pedido_dest_id, COALESCE(doc_tipo,'') AS doc_tipo, COALESCE(doc_numero,'') AS doc_numero FROM so_pedido_dest_doc WHERE pedido_dest_id IN ($place)";
      $stInv = $pdo->prepare($sql); $stInv->execute($pedidoDestIds);
      while ($row = $stInv->fetch(PDO::FETCH_ASSOC)) {
        $pid = (int)$row['pedido_dest_id'];
        if (!isset($invoicesByPedidoDest[$pid])) $invoicesByPedidoDest[$pid] = [];
        $num = trim(($row['doc_tipo']?($row['doc_tipo'].': '):'').($row['doc_numero'] ?? ''));
        if ($num !== '') $invoicesByPedidoDest[$pid][] = $num;
      }
    } else if ($tableExists($pdo,'so_pedido_dest')) {
      $hasFactura = $columnExists($pdo,'so_pedido_dest','factura');
      $hasDocNum  = $columnExists($pdo,'so_pedido_dest','doc_numero');
      if ($hasFactura || $hasDocNum) {
        $place = implode(',', array_fill(0, count($pedidoDestIds), '?'));
        $cols = ['id']; if ($hasFactura) $cols[] = 'factura'; if ($hasDocNum) $cols[] = 'doc_numero';
        $sql = 'SELECT '.implode(',', $cols).' FROM so_pedido_dest WHERE id IN ('.$place.')';
        $stInv = $pdo->prepare($sql); $stInv->execute($pedidoDestIds);
        while ($row = $stInv->fetch(PDO::FETCH_ASSOC)) {
          $pid = (int)$row['id'];
          $inv = [];
          if ($hasFactura && !empty($row['factura'])) $inv[] = 'Factura: '.$row['factura'];
          if ($hasDocNum && !empty($row['doc_numero'])) $inv[] = 'Doc: '.$row['doc_numero'];
          if (!empty($inv)) $invoicesByPedidoDest[$pid] = $inv;
        }
      }
    }
  }

  if (!empty($byDest)) {
    $html .= '<div class="section-title">Resumen por Destinatario</div>';
    $html .= '<table class="tbl"><thead><tr><th>Destinatario</th><th class="right">UV</th><th class="right">UC</th><th>Facturas/Docs</th></tr></thead><tbody>';
    foreach ($byDest as $dId => $agg) {
      $pids = array_keys($agg['pedido_dest_ids']);
      $docs = [];
      foreach ($pids as $pid) { if (isset($invoicesByPedidoDest[$pid])) { $docs = array_merge($docs, $invoicesByPedidoDest[$pid]); } }
      $docTxt = htmlspecialchars(implode(' | ', array_unique($docs)));
      $html .= '<tr>'
        .'<td>'. htmlspecialchars($agg['dest']) .'</td>'
        .'<td class="right">'.$agg['uv'].'</td>'
        .'<td class="right">'.$agg['uc'].'</td>'
        .'<td>'.$docTxt.'</td>'
        .'</tr>';
    }
    $html .= '</tbody></table>';
  }

  if (!empty($byProd)) {
    $html .= '<div class="section-title">Resumen por Producto</div>';
    $html .= '<table class="tbl"><thead><tr><th>Producto</th><th class="right">UV</th><th class="right">UC</th></tr></thead><tbody>';
    foreach ($byProd as $prod => $agg) {
      $html .= '<tr>'
        .'<td>'. htmlspecialchars($prod) .'</td>'
        .'<td class="right">'.$agg['uv'].'</td>'
        .'<td class="right">'.$agg['uc'].'</td>'
        .'</tr>';
    }
    $html .= '</tbody></table>';
  }


  // Firmas
  $html .= '<table class="signatures" cellspacing="0" cellpadding="0"><tr>'
    .'<td class="sig"><div class="line"></div>Montacarguista</td>'
    .'<td class="sig"><div class="line"></div>Supervisor</td>'
    .'<td class="sig"><div class="line"></div>Chofer</td>'
    .'</tr></table>';

  $mpdf->WriteHTML($html);

  $fname = 'planilla_salida_'. preg_replace('/[^A-Za-z0-9_\-]/','',(string)$e['codigo']) .'_'. date('Ymd_His') .'.pdf';
  // Mostrar inline
  $mpdf->Output($fname, 'I');

} catch (Throwable $ex) {
  fail(500, 'No se pudo generar el PDF: '.$ex->getMessage());
}
