<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function respond(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function normalize_datetime(?string $value): ?string {
    if ($value === null) {
        return null;
    }
    $value = trim($value);
    if ($value === '') {
        return null;
    }
    if (strpos($value, 'T') !== false) {
        $value = str_replace('T', ' ', $value);
    }
    if (strlen($value) === 16) {
        $value .= ':00';
    }
    return $value;
}

function normalize_date(?string $value): ?string {
    if ($value === null) {
        return null;
    }
    $value = trim($value);
    if ($value === '') {
        return null;
    }
    if (strlen($value) >= 10) {
        return substr($value, 0, 10);
    }
    return $value;
}

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    if ($method === 'GET') {
        $embarqueId = isset($_GET['embarque_id']) ? (int)$_GET['embarque_id'] : 0;
        if ($embarqueId <= 0) {
            respond(['ok' => false, 'error' => 'embarque_id requerido'], 422);
        }
        $stmt = $pdo->prepare('SELECT id, factura, condicion, fecha_factura, monto, fecha_rendicion FROM so_embarque_rendiciones WHERE embarque_id = :emb ORDER BY id ASC');
        $stmt->execute(['emb' => $embarqueId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $rows = array_map(static function (array $row): array {
            return [
                'id' => (int)($row['id'] ?? 0),
                'factura' => (string)($row['factura'] ?? ''),
                'condicion' => $row['condicion'] !== null ? (string)$row['condicion'] : '',
                'fecha_factura' => $row['fecha_factura'] !== null ? (string)$row['fecha_factura'] : null,
                'monto' => $row['monto'] !== null ? (float)$row['monto'] : 0.0,
                'fecha_rendicion' => $row['fecha_rendicion'] !== null ? (string)$row['fecha_rendicion'] : null,
            ];
        }, $rows);
        respond(['ok' => true, 'data' => $rows]);
    }

    if ($method === 'POST') {
        $body = json_decode(file_get_contents('php://input') ?: '[]', true) ?: [];
        $embarqueId = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
        if ($embarqueId <= 0) {
            respond(['ok' => false, 'error' => 'embarque_id requerido'], 422);
        }
        $rowId = isset($body['row_id']) ? (int)$body['row_id'] : 0;
        $data = isset($body['data']) && is_array($body['data']) ? $body['data'] : [];

        $factura = trim((string)($data['factura'] ?? ''));
        if ($factura === '') {
            respond(['ok' => false, 'error' => 'La factura es obligatoria'], 422);
        }
        if (strlen($factura) > 100) {
            $factura = substr($factura, 0, 100);
        }

        $condicion = trim((string)($data['condicion'] ?? ''));
        if ($condicion === '') {
            $condicion = null;
        } elseif (strlen($condicion) > 100) {
            $condicion = substr($condicion, 0, 100);
        }

        $fechaFactura = normalize_date($data['fecha_factura'] ?? null);
        $monto = isset($data['monto']) ? (float)$data['monto'] : 0.0;
        if (!is_finite($monto)) {
            $monto = 0.0;
        }
        if ($monto < 0) {
            $monto = 0.0;
        }

        $fechaRendicion = normalize_datetime($data['fecha_rendicion'] ?? null);
        if ($fechaRendicion === null) {
            $fechaRendicion = date('Y-m-d H:i:s');
        }

        if ($rowId > 0) {
            $verify = $pdo->prepare('SELECT id FROM so_embarque_rendiciones WHERE id = :id AND embarque_id = :emb LIMIT 1');
            $verify->execute(['id' => $rowId, 'emb' => $embarqueId]);
            if (!$verify->fetch(PDO::FETCH_ASSOC)) {
                respond(['ok' => false, 'error' => 'Registro no encontrado'], 404);
            }
            $upd = $pdo->prepare('UPDATE so_embarque_rendiciones
                                  SET factura = :factura,
                                      condicion = :condicion,
                                      fecha_factura = :fecha_factura,
                                      monto = :monto,
                                      fecha_rendicion = :fecha_rendicion
                                  WHERE id = :id AND embarque_id = :emb');
            $upd->execute([
                'factura' => $factura,
                'condicion' => $condicion,
                'fecha_factura' => $fechaFactura,
                'monto' => $monto,
                'fecha_rendicion' => $fechaRendicion,
                'id' => $rowId,
                'emb' => $embarqueId,
            ]);
            respond(['ok' => true, 'id' => $rowId]);
        }

        $ins = $pdo->prepare('INSERT INTO so_embarque_rendiciones (embarque_id, factura, condicion, fecha_factura, monto, fecha_rendicion)
                               VALUES (:emb, :factura, :condicion, :fecha_factura, :monto, :fecha_rendicion)');
        $ins->execute([
            'emb' => $embarqueId,
            'factura' => $factura,
            'condicion' => $condicion,
            'fecha_factura' => $fechaFactura,
            'monto' => $monto,
            'fecha_rendicion' => $fechaRendicion,
        ]);
        $newId = (int)$pdo->lastInsertId();
        respond(['ok' => true, 'id' => $newId], 201);
    }

    if ($method === 'DELETE') {
        $body = json_decode(file_get_contents('php://input') ?: '[]', true) ?: [];
        $embarqueId = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
        $rowId = isset($body['row_id']) ? (int)$body['row_id'] : 0;
        if ($embarqueId <= 0 || $rowId <= 0) {
            respond(['ok' => false, 'error' => 'embarque_id y row_id requeridos'], 422);
        }
        $del = $pdo->prepare('DELETE FROM so_embarque_rendiciones WHERE id = :id AND embarque_id = :emb LIMIT 1');
        $del->execute(['id' => $rowId, 'emb' => $embarqueId]);
        if ($del->rowCount() === 0) {
            respond(['ok' => false, 'error' => 'Registro no encontrado'], 404);
        }
        respond(['ok' => true]);
    }

    respond(['ok' => false, 'error' => 'Método no permitido'], 405);
} catch (Throwable $e) {
    respond(['ok' => false, 'error' => 'Error interno', 'message' => $e->getMessage()], 500);
}
