<?php
declare(strict_types=1);

use Mpdf\Mpdf;

require_once dirname(__DIR__, 2) . '/vendor/autoload.php';
require_once dirname(__DIR__, 2) . '/config/config.php';
require_once dirname(__DIR__, 2) . '/config/db.php';

header('Content-Type: application/pdf');

function fail(int $code, string $msg){ http_response_code($code); echo $msg; exit; }

function is_hhmm(?string $s): bool { if ($s===null) return false; return (bool)preg_match('/^\d{1,2}:\d{2}$/', trim($s)); }
function parse_hhmm(?string $s): int { if (!is_hhmm($s)) return 0; [$h,$m] = array_map('intval', explode(':', trim((string)$s))); return $h*60+$m; }
function fmt_hhmm(int $mins): string { $mins = max(0,$mins); $h = intdiv($mins,60); $m = $mins%60; return sprintf('%02d:%02d',$h,$m); }
function diff_hhmm(?string $later, ?string $earlier): ?string { if (!is_hhmm($later) || !is_hhmm($earlier)) return null; $d = parse_hhmm($later) - parse_hhmm($earlier); if ($d < 0) $d = 0; return fmt_hhmm($d); }

try {
  $embarqueId = isset($_GET['embarque_id']) ? (int)$_GET['embarque_id'] : 0;
  if ($embarqueId <= 0) fail(400, 'embarque_id requerido');

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // Header del embarque
  $st = $pdo->prepare("SELECT e.*, ee.code AS estado, d.code AS deposito_code FROM so_embarque e JOIN so_embarque_estado ee ON ee.id=e.estado_id JOIN wh_deposito d ON d.id=e.deposito_id WHERE e.id=? LIMIT 1");
  $st->execute([$embarqueId]);
  $e = $st->fetch(PDO::FETCH_ASSOC);
  if (!$e) fail(404, 'No existe el embarque');

  // PREs asociados
  $stp = $pdo->prepare("SELECT pre.id, pre.codigo FROM so_embarque_pre ep JOIN so_preembarque pre ON pre.id=ep.preembarque_id WHERE ep.embarque_id=? ORDER BY pre.id");
  $stp->execute([$embarqueId]);
  $pres = $stp->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Cargar filas base (destinatarios, facturas, etc.) — similar a tracking API
  $sqlRows = "
    SELECT d.id AS pedido_dest_id,
           d.destinatario_id,
           dest.razon_social AS destinatario
      FROM so_pre_pick sp
      JOIN so_pedido_dest_item i ON i.id=sp.pedido_dest_item_id
      JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
      JOIN para_destinatarios dest ON dest.id=d.destinatario_id
     WHERE sp.preembarque_id=?
     GROUP BY d.id, d.destinatario_id, dest.razon_social
  ";
  $byDest = [];
  $stR = $pdo->prepare($sqlRows);
  foreach ($pres as $p) {
    $stR->execute([(int)$p['id']]);
    while ($r = $stR->fetch(PDO::FETCH_ASSOC)) {
      $destId = (int)$r['destinatario_id'];
      if (!isset($byDest[$destId])) $byDest[$destId] = [
        'destinatario_id'=>$destId,
        'destinatario'=>(string)$r['destinatario'],
        'pedido_dest_ids'=>[],
      ];
      $byDest[$destId]['pedido_dest_ids'][(int)$r['pedido_dest_id']] = true;
    }
  }

  // Fallback: si no hay PRE/picks o no se obtuvieron destinatarios, usar los que existan en la tabla de seguimiento
  if (empty($byDest)) {
    try {
      $q = $pdo->prepare("SELECT s.destinatario_id, d.razon_social FROM so_embarque_seguimiento_dest s JOIN para_destinatarios d ON d.id=s.destinatario_id WHERE s.embarque_id=? GROUP BY s.destinatario_id, d.razon_social");
      $q->execute([$embarqueId]);
      while ($r = $q->fetch(PDO::FETCH_ASSOC)) {
        $destId = (int)$r['destinatario_id'];
        $byDest[$destId] = [
          'destinatario_id'=>$destId,
          'destinatario'=>(string)$r['razon_social'],
          'pedido_dest_ids'=>[],
        ];
      }
    } catch (Throwable $ex) { /* ignorar */ }
  }

  // Lectura de seguimiento guardado
  $segByDest = [];
  if (!empty($byDest)) {
    $place = implode(',', array_fill(0, count($byDest), '?'));
    $sql = "SELECT * FROM so_embarque_seguimiento_dest WHERE embarque_id=? AND destinatario_id IN ($place)";
    $params = array_merge([$embarqueId], array_keys($byDest));
    try { $stSeg = $pdo->prepare($sql); $stSeg->execute($params); while ($row = $stSeg->fetch(PDO::FETCH_ASSOC)) { $segByDest[(int)$row['destinatario_id']] = $row; } } catch (Throwable $ex) { /* si la tabla no existe todavía */ }
  }

  // Facturas/docs por pedido_dest (best-effort)
  $invoicesByPedidoDest = [];
  $pedidoDestAll = [];
  foreach ($byDest as $d) { foreach (array_keys($d['pedido_dest_ids']) as $pid) { if ($pid) $pedidoDestAll[$pid] = true; } }
  $pedidoDestIds = array_keys($pedidoDestAll);
  if (!empty($pedidoDestIds)) {
    $q = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'so_pedido_dest_doc'");
    $q->execute(); $hasDoc = ((int)$q->fetchColumn()) > 0;
    if ($hasDoc) {
      $place = implode(',', array_fill(0, count($pedidoDestIds), '?'));
      $sql = "SELECT pedido_dest_id, COALESCE(doc_tipo,'') AS doc_tipo, COALESCE(doc_numero,'') AS doc_numero FROM so_pedido_dest_doc WHERE pedido_dest_id IN ($place)";
      $stInv = $pdo->prepare($sql); $stInv->execute($pedidoDestIds);
      while ($row = $stInv->fetch(PDO::FETCH_ASSOC)) {
        $pid = (int)$row['pedido_dest_id'];
        if (!isset($invoicesByPedidoDest[$pid])) $invoicesByPedidoDest[$pid] = [];
        $num = trim(($row['doc_tipo']?($row['doc_tipo'].': '):'').($row['doc_numero'] ?? ''));
        if ($num !== '') $invoicesByPedidoDest[$pid][] = $num;
      }
    } else {
      // Fallback: leer posibles columnas en so_pedido_dest (factura/doc_numero)
      $q2 = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'so_pedido_dest'");
      $q2->execute(); $hasPedidoDest = ((int)$q2->fetchColumn()) > 0;
      if ($hasPedidoDest) {
        $col = $pdo->prepare("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'so_pedido_dest' AND COLUMN_NAME IN ('factura','doc_numero')");
        $col->execute(); $cols = $col->fetchAll(PDO::FETCH_COLUMN) ?: [];
        if (!empty($cols)) {
          $place = implode(',', array_fill(0, count($pedidoDestIds), '?'));
          $sql = 'SELECT id, '.implode(',', $cols).' FROM so_pedido_dest WHERE id IN ('.$place.')';
          $stInv = $pdo->prepare($sql); $stInv->execute($pedidoDestIds);
          while ($row = $stInv->fetch(PDO::FETCH_ASSOC)) {
            $pid = (int)$row['id'];
            $vals = [];
            if (in_array('factura', $cols, true) && !empty($row['factura'])) $vals[] = 'Factura: '.$row['factura'];
            if (in_array('doc_numero', $cols, true) && !empty($row['doc_numero'])) $vals[] = 'Doc: '.$row['doc_numero'];
            if (!empty($vals)) $invoicesByPedidoDest[$pid] = $vals;
          }
        }
      }
    }
  }

  // Armar dataset final por destinatario, computando campos derivados
  $rows = [];
  $totEspera=0; $totDesc=0; $totCtrl=0; $totTotal=0; $filas=0;
  foreach ($byDest as $dId => $base) {
    $seg = $segByDest[$dId] ?? [];
    $pids = array_keys($base['pedido_dest_ids']);
    $docs = [];
    foreach ($pids as $pid) { if (isset($invoicesByPedidoDest[$pid])) { $docs = array_merge($docs, $invoicesByPedidoDest[$pid]); } }
    $facturas = implode(' | ', array_unique($docs));

    $hr_llegada = $seg['hr_llegada'] ?? null;
    $hr_inicio  = $seg['hr_inicio'] ?? null;
    $hr_termino = $seg['hr_termino'] ?? null;
    $hr_salida  = $seg['hr_salida'] ?? null;

    $espera = diff_hhmm($hr_inicio, $hr_llegada);
    $desc   = diff_hhmm($hr_termino, $hr_inicio);
    $ctrl   = diff_hhmm($hr_salida, $hr_termino);
    $total  = diff_hhmm($hr_salida, $hr_llegada);

    $rows[] = [
      'facturas'=>$facturas,
      'cliente_final'=>$base['destinatario'],
      'inicio_carga'=>$seg['inicio_carga'] ?? null,
      'fin_carga'=>$seg['fin_carga'] ?? null,
      'km_inicial'=>$seg['km_inicial'] ?? null,
      'km_llegada'=>$seg['km_llegada'] ?? null,
      'hr_llegada'=>$hr_llegada,
      'hr_inicio'=>$hr_inicio,
      'espera_desc'=>$espera,
      'hr_termino'=>$hr_termino,
      'tiempo_desc'=>$desc,
      'hr_salida'=>$hr_salida,
      'tiempo_ctrl'=>$ctrl,
      'tiempo_total'=>$total,
      'tipo_carga'=>$seg['tipo_carga'] ?? null,
      'salida_camara'=>$seg['salida_camara'] ?? null,
      'temp_carga_c'=>$seg['temp_carga_c'] ?? null,
      'temp_desc_c'=>$seg['temp_desc_c'] ?? null,
      'aviso_quien'=>$seg['aviso_quien'] ?? null,
      'aviso_hora'=>$seg['aviso_hora'] ?? null,
      'problema'=>$seg['problema'] ?? null,
    ];

    $filas++;
    $totEspera += parse_hhmm($espera ?? '');
    $totDesc   += parse_hhmm($desc ?? '');
    $totCtrl   += parse_hhmm($ctrl ?? '');
    $totTotal  += parse_hhmm($total ?? '');
  }

  // PDF
  $mpdf = new Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 35,
    'margin_bottom' => 15,
  ]);

  $header = '<div style="border-bottom:2px solid #000; font-size:11px;">
    <table width="100%" cellpadding="0" cellspacing="0">
      <tr>
        <td style="font-weight:bold; font-size:16px;">Seguimiento de Embarque</td>
        <td align="right">Fecha impresión: '.date('Y-m-d H:i').'</td>
      </tr>
      <tr>
        <td>
          Embarque: <b>'.htmlspecialchars((string)$e['codigo']).'</b>
          &nbsp;|&nbsp; Depósito: <b>'.htmlspecialchars((string)$e['deposito_code']).'</b>
          &nbsp;|&nbsp; Estado: <b>'.htmlspecialchars((string)$e['estado']).'</b>
        </td>
        <td align="right">Página {PAGENO}/{nbpg}</td>
      </tr>
      <tr>
        <td colspan="2">
          Filas: <b>'.$filas.'</b>
          &nbsp;|&nbsp; Espera: <b>'.fmt_hhmm($totEspera).'</b>
          &nbsp;|&nbsp; Descarga: <b>'.fmt_hhmm($totDesc).'</b>
          &nbsp;|&nbsp; Control: <b>'.fmt_hhmm($totCtrl).'</b>
          &nbsp;|&nbsp; Total: <b>'.fmt_hhmm($totTotal).'</b>
        </td>
      </tr>
    </table>
  </div>';
  $mpdf->SetHTMLHeader($header);

  $css = '<style>
    body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 10.5px; }
    table.tbl { width: 100%; border-collapse: collapse; }
    table.tbl th, table.tbl td { border: 1px solid #000; padding: 3px 4px; white-space: nowrap; }
    table.tbl th { background: #eee; font-weight: bold; }
    .small { font-size: 9px; }
  </style>';

  $html = $css;
  $html .= '<table class="tbl"><thead><tr>
    <th>No. Factura</th>
    <th>Cliente Final</th>
    <th>Inicio Carga</th>
    <th>Fin Carga</th>
    <th>Km Inicial</th>
    <th>Km Llegada</th>
    <th>Hr Llegada</th>
    <th>Hr Inicio</th>
    <th>Espera para Descargar</th>
    <th>Hr. Termino</th>
    <th>Tiempo de Descarga</th>
    <th>Hr. Salida</th>
    <th>Tiempo de Control</th>
    <th>Tiempo TOTAL</th>
    <th>Tipo de Carga</th>
    <th>Salida de Cámara</th>
  <th>Carga de camión temp. C°</th>
    <th>Descarga temp. C°</th>
    <th>Aviso a Quien</th>
    <th>Hora</th>
    <th>Problema</th>
  </tr></thead><tbody>';

  foreach ($rows as $r) {
    $html .= '<tr>'
      .'<td>'.htmlspecialchars((string)($r['facturas'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['cliente_final'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['inicio_carga'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['fin_carga'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['km_inicial'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['km_llegada'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['hr_llegada'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['hr_inicio'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['espera_desc'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['hr_termino'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['tiempo_desc'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['hr_salida'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['tiempo_ctrl'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['tiempo_total'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['tipo_carga'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['salida_camara'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['temp_carga_c'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['temp_desc_c'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['aviso_quien'] ?? '')).'</td>'
      .'<td>'.htmlspecialchars((string)($r['aviso_hora'] ?? '')).'</td>'
      .'<td class="small">'.htmlspecialchars((string)($r['problema'] ?? '')).'</td>'
      .'</tr>';
  }

  $html .= '</tbody></table>';

  $mpdf->WriteHTML($html);

  $fname = 'seguimiento_embarque_'. preg_replace('/[^A-Za-z0-9_\-]/','',(string)$e['codigo']) .'_'. date('Ymd_His') .'.pdf';
  $mpdf->Output($fname, 'I');

} catch (Throwable $ex) {
  fail(500, 'No se pudo generar el PDF: '.$ex->getMessage());
}
