<?php
declare(strict_types=1);
/**
 * API: Actualizar cabecera de Embarque
 */
header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');
$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

function out(array $p, int $c=200){ http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

try {
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') out(['ok'=>false,'error'=>'Método no permitido'], 405);
  $raw = file_get_contents('php://input') ?: '';
  $body = json_decode($raw, true);
  if (!is_array($body)) $body = $_POST;

  $id = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
  if ($id<=0) out(['ok'=>false,'error'=>'embarque_id requerido'], 422);

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');
  $pdo->beginTransaction();

  // Obtener estado actual para validar restricciones y bloquear registro
  $st = $pdo->prepare("SELECT ee.orden, ee.code, e.movil_id
                         FROM so_embarque e
                         JOIN so_embarque_estado ee ON ee.id = e.estado_id
                        WHERE e.id = ?
                        LIMIT 1
                        FOR UPDATE");
  $st->execute([$id]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) out(['ok'=>false,'error'=>'Embarque no encontrado'], 404);
  $orden = (int)$row['orden'];
  $code  = strtoupper((string)$row['code']);
  $currentMovilId = isset($row['movil_id']) && $row['movil_id'] !== null ? (int)$row['movil_id'] : null;

  $hasOcupadoColumn = columnExists($pdo, 'para_moviles', 'ocupado');
  $hasDisponiblesTable = tableExists($pdo, 'oper_movil_disponible');
  $hasNoAsistioCol = $hasDisponiblesTable && columnExists($pdo, 'oper_movil_disponible', 'no_asistio');
  $hasNoUtilizadoCol = $hasDisponiblesTable && columnExists($pdo, 'oper_movil_disponible', 'no_utilizado');

  // Si FINALIZADO: no se admite ningún cambio
  if ($code === 'FINALIZADO') {
    $pdo->rollBack();
    out(['ok'=>false,'error'=>'Embarque finalizado: no admite cambios'], 409);
  }

  // Build dynamic update (restringir si ya está CARGADO o más avanzado: orden>=3)
  $allFields = [
    'movil_id','chofer_id','llegada_at','carga_inicio_at','carga_fin_at','salida_at',
    'ticket_porteria','ticket_bascula','temp_salida_c','ayudantes_cant','km_inicial','observacion'
  ];
  $timeOnlyFields = ['llegada_at','carga_inicio_at','carga_fin_at','salida_at'];
  $fields = ($orden >= 3) ? $timeOnlyFields : $allFields;
  $sets = []; $vals=[];
  $movilChange = false;
  $newMovilId = $currentMovilId;

  foreach ($fields as $f) {
    if (!array_key_exists($f, $body)) {
      continue;
    }
    $value = $body[$f];
    if ($f === 'movil_id') {
      $newMovilId = ($value === '' || $value === null) ? null : (int) $value;
      if ($newMovilId !== null && $newMovilId <= 0) {
        $newMovilId = null;
      }
      $movilChange = ($newMovilId !== $currentMovilId);
      $vals[':movil_id'] = $newMovilId;
      $sets[] = 'movil_id = :movil_id';
      continue;
    }
    $sets[] = "$f = :$f";
    $vals[":$f"] = $value !== '' ? $value : null;
  }

  if ($movilChange && $fields && in_array('movil_id', $fields, true)) {
    if ($newMovilId !== null) {
      // Validar móvil seleccionado
      $stMov = $pdo->prepare('SELECT id, activo' . ($hasOcupadoColumn ? ', ocupado' : '') . ' FROM para_moviles WHERE id = ? LIMIT 1 FOR UPDATE');
      $stMov->execute([$newMovilId]);
      $movRow = $stMov->fetch(PDO::FETCH_ASSOC);
      if (!$movRow) {
        $pdo->rollBack();
        out(['ok'=>false,'error'=>'Móvil no encontrado'], 422);
      }
      if ((int)($movRow['activo'] ?? 0) !== 1) {
        $pdo->rollBack();
        out(['ok'=>false,'error'=>'El móvil seleccionado está inactivo'], 409);
      }
      if ($hasOcupadoColumn && (int)($movRow['ocupado'] ?? 0) === 1) {
        $pdo->rollBack();
        out(['ok'=>false,'error'=>'El móvil seleccionado ya está ocupado'], 409);
      }

      if ($hasDisponiblesTable) {
        $conditions = ['fecha = CURDATE()', 'movil_id = ?'];
        $params = [$newMovilId];
        $conditions[] = 'disponible = 1';
        if ($hasNoAsistioCol) {
          $conditions[] = '(no_asistio IS NULL OR no_asistio = 0)';
        }
        if ($hasNoUtilizadoCol) {
          $conditions[] = '(no_utilizado IS NULL OR no_utilizado = 0)';
        }
        $sqlDispon = 'SELECT 1 FROM oper_movil_disponible WHERE ' . implode(' AND ', $conditions) . ' LIMIT 1';
        $stDisp = $pdo->prepare($sqlDispon);
        $stDisp->execute($params);
        if (!$stDisp->fetchColumn()) {
          $pdo->rollBack();
          out(['ok'=>false,'error'=>'El móvil seleccionado no está disponible para la fecha de hoy'], 409);
        }
      }
    }
  }

  if (!$sets) {
    $pdo->commit();
    out(['ok'=>true,'changed'=>false]);
  }
  $vals[':id'] = $id;
  $sql = 'UPDATE so_embarque SET ' . implode(', ', $sets) . ' WHERE id=:id';
  $st = $pdo->prepare($sql);
  $st->execute($vals);

  if ($movilChange && $hasOcupadoColumn) {
    if ($currentMovilId && $currentMovilId !== $newMovilId) {
      $rel = $pdo->prepare('UPDATE para_moviles SET ocupado = 0, updated_at = NOW() WHERE id = ?');
      $rel->execute([$currentMovilId]);
    }
    if ($newMovilId !== null) {
      $occ = $pdo->prepare('UPDATE para_moviles SET ocupado = 1, updated_at = NOW() WHERE id = ?');
      $occ->execute([$newMovilId]);
    }
  }

  $pdo->commit();

  out(['ok'=>true,'changed'=>($st->rowCount()>0)]);
} catch (Throwable $e) {
  if (isset($pdo) && $pdo->inTransaction()) {
    $pdo->rollBack();
  }
  out(['ok'=>false,'error'=>'No se pudo actualizar','message'=>$e->getMessage()], 500);
}
