<?php
declare(strict_types=1);
/**
 * API: Registrar VUELTA del embarque (cierra el embarque)
 * POST JSON: {
 *   embarque_id: number,
 *   llegada_vuelta_at?: string (ISO, opcional, por defecto NOW()),
 *   doc_tipo?: string (REMITO|FACTURA|OTRO),
 *   doc_numero?: string,
 *   rendicion?: string
 * }
 * Efectos:
 *  - Inserta en so_retorno (llegada_at, observacion con doc/rendición/usuario)
 *  - Cambia estado del embarque a FINALIZADO
 */
header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');
$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

function out(array $p, int $c=200){ http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

session_start();
if (empty($_SESSION['usuario_id'])) { out(['ok'=>false,'error'=>'No autenticado'], 401); }
$userId = (int)$_SESSION['usuario_id'];

try {
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') out(['ok'=>false,'error'=>'Método no permitido'], 405);
  $raw = file_get_contents('php://input') ?: '';
  $body = json_decode($raw, true);
  if (!is_array($body)) $body = $_POST;

  $id = isset($body['embarque_id']) ? (int)$body['embarque_id'] : 0;
  if ($id<=0) out(['ok'=>false,'error'=>'embarque_id requerido'], 422);
  $llegadaVuelta = trim((string)($body['llegada_vuelta_at'] ?? ''));
  $docTipo = strtoupper(trim((string)($body['doc_tipo'] ?? '')));
  $docNumero = trim((string)($body['doc_numero'] ?? ''));
  $rendicion = trim((string)($body['rendicion'] ?? ''));

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // Obtener nombre del usuario si existe
  $userName = null;
  try {
    $stU = $pdo->prepare("SELECT COALESCE(u_userfullname, username, CONCAT('user#', id)) AS nombre FROM sys_users WHERE id=? LIMIT 1");
    $stU->execute([$userId]);
    $userName = (string)($stU->fetchColumn() ?: ('user#'.$userId));
  } catch (Throwable $e) {
    $userName = 'user#'.$userId;
  }

  $pdo->beginTransaction();

  // Obtener móvil asociado y bloquear registro del embarque
  $stMov = $pdo->prepare('SELECT movil_id FROM so_embarque WHERE id = ? LIMIT 1 FOR UPDATE');
  $stMov->execute([$id]);
  $movilId = $stMov->fetchColumn();
  $movilId = $movilId !== false ? (int)$movilId : null;
  $hasOcupadoColumn = columnExists($pdo, 'para_moviles', 'ocupado');

  // Insertar retorno
  $obsParts = [];
  if ($docTipo !== '') $obsParts[] = 'Doc: ' . $docTipo;
  if ($docNumero !== '') $obsParts[] = 'Nro: ' . $docNumero;
  if ($rendicion !== '') $obsParts[] = 'Rendición: ' . $rendicion;
  $obsParts[] = 'Registrado por: ' . $userName . ' (ID ' . $userId . ')';
  $observacion = implode(' | ', $obsParts);

  if ($llegadaVuelta === '') {
    // usar NOW()
    $stR = $pdo->prepare("INSERT INTO so_retorno (embarque_id, llegada_at, observacion) VALUES (?, NOW(), ?)");
    $stR->execute([$id, $observacion]);
  } else {
    $stR = $pdo->prepare("INSERT INTO so_retorno (embarque_id, llegada_at, observacion) VALUES (?, ?, ?)");
    $stR->execute([$id, $llegadaVuelta, $observacion]);
  }
  $retId = (int)$pdo->lastInsertId();

  // Cambiar estado a VUELTA si existe (case-insensitive), de lo contrario FINALIZADO
  $estadosStmt = $pdo->query("SELECT id, UPPER(code) AS code FROM so_embarque_estado WHERE UPPER(code) IN ('FINALIZADO','VUELTA')");
  $estadoMap = [];
  foreach ($estadosStmt ?: [] as $row) {
    $estadoMap[$row['code']] = (int)$row['id'];
  }
  $estFin = $estadoMap['FINALIZADO'] ?? 0;
  if ($estFin <= 0) {
    throw new RuntimeException('No existe estado FINALIZADO');
  }
  $estVuelta = $estadoMap['VUELTA'] ?? 0;
  $targetEstadoId = $estVuelta > 0 ? $estVuelta : $estFin;
  $targetCode = $estVuelta > 0 ? 'VUELTA' : 'FINALIZADO';
  $stE = $pdo->prepare("UPDATE so_embarque SET estado_id=? WHERE id=?");
  $stE->execute([$targetEstadoId, $id]);

  if ($hasOcupadoColumn && $movilId) {
    $stOcc = $pdo->prepare('UPDATE para_moviles SET ocupado = 0, updated_at = NOW() WHERE id = ?');
    $stOcc->execute([$movilId]);
  }

  $pdo->commit();
  out(['ok'=>true, 'retorno_id'=>$retId, 'registrado_por'=>$userName, 'estado_code'=>$targetCode]);
} catch (Throwable $e) {
  if (isset($pdo) && $pdo->inTransaction()) { $pdo->rollBack(); }
  out(['ok'=>false,'error'=>'No se pudo registrar la vuelta','message'=>$e->getMessage()], 500);
}
