<?php
declare(strict_types=1);

/**
 * API: Listar ítems de una Salida (pedido consolidado)
 * Ruta: api/operaciones/so_items.php
 * Método: GET
 *
 * Parámetros:
 * - so_id      (int) id de so_pedido
 * - so_codigo  (string) código de so_pedido (SO-YYYYMMDD-...)
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out(array $p, int $c = 200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'GET') {
        out(['ok'=>false,'error'=>'Método no permitido'], 405);
    }

    $soId = isset($_GET['so_id']) ? (int)$_GET['so_id'] : 0;
    $soCodigo = isset($_GET['so_codigo']) ? trim((string)$_GET['so_codigo']) : '';

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    if ($soId <= 0 && $soCodigo !== '') {
        $s = $pdo->prepare('SELECT id FROM so_pedido WHERE codigo=? LIMIT 1');
        $s->execute([$soCodigo]);
        $soId = (int)$s->fetchColumn();
    }
    if ($soId <= 0) out(['ok'=>false,'error'=>'Debe indicar so_id o so_codigo'], 422);

    $stH = $pdo->prepare('SELECT id, codigo, cliente_ref, fecha_pedido AS fecha FROM so_pedido WHERE id=? LIMIT 1');
    $stH->execute([$soId]);
    $hdr = $stH->fetch(PDO::FETCH_ASSOC);
    if (!$hdr) out(['ok'=>false,'error'=>'Salida no encontrada'], 404);

    // Columnas estilo DataTables
    $columns = [
        ['data'=>'destinatario',   'title'=>'Destinatario'],
        ['data'=>'producto_sku',   'title'=>'SKU'],
        ['data'=>'producto',       'title'=>'Producto'],
        ['data'=>'lote_codigo',    'title'=>'Lote'],
        ['data'=>'expected_uv',    'title'=>'UV Cajas','type'=>'number'],
        ['data'=>'expected_uc',    'title'=>'UC totales','type'=>'number'],
        ['data'=>'prepared_uv',    'title'=>'Prep. UV','type'=>'number'],
        ['data'=>'prepared_uc',    'title'=>'Prep. UC','type'=>'number'],
        ['data'=>'shipped_uv',     'title'=>'Desp. UV','type'=>'number'],
        ['data'=>'shipped_uc',     'title'=>'Desp. UC','type'=>'number'],
    ];

    $st = $pdo->prepare('
        SELECT
          dest.razon_social AS destinatario,
          p.sku AS producto_sku,
          p.denominacion AS producto,
          i.lote_codigo,
          i.expected_uv,
          i.expected_uc,
          i.prepared_uv,
          i.prepared_uc,
          i.shipped_uv,
          i.shipped_uc
        FROM so_pedido_dest_item i
        JOIN so_pedido_dest d   ON d.id = i.pedido_dest_id
        JOIN para_destinatarios dest ON dest.id = d.destinatario_id
        JOIN para_productos p   ON p.id = i.producto_id
        WHERE d.pedido_id = ?
        ORDER BY d.id ASC, i.id ASC
    ');
    $st->execute([$soId]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    out([
        'ok' => true,
        'so' => [
            'id'          => (int)$hdr['id'],
            'codigo'      => (string)$hdr['codigo'],
            'cliente_ref' => (string)$hdr['cliente_ref'],
            'fecha'       => (string)$hdr['fecha'],
        ],
        'columns' => $columns,
        'data'    => $rows,
    ]);

} catch (Throwable $e) {
    out(['ok'=>false,'error'=>'No se pudo cargar el detalle','message'=>$e->getMessage()], 500);
}
