<?php
declare(strict_types=1);

/**
 * API: Listado de pedidos confirmados (so_pedido) con resumen para DataTables
 * Ruta: api/operaciones/so_list.php
 * Método: GET/POST (DataTables Editor helper soporta ambos, aquí sólo lectura GET)
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out(array $p, int $c = 200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE); exit; }

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    // Estados visibles (principalmente 'RECIBIDO', 'EN_PREPARACION', etc.)
    // Por ahora listamos todos los pedidos recientes (últimos 60 días)
    $st = $pdo->prepare("SELECT p.id, p.codigo, p.cliente_ref, p.fecha_pedido, p.created_at, pe.code AS estado_code
        FROM so_pedido p
        LEFT JOIN so_pedido_estado pe ON pe.id = p.estado_id
        WHERE p.fecha_pedido >= DATE_SUB(CURDATE(), INTERVAL 60 DAY)
        ORDER BY p.created_at DESC, p.id DESC");
    $st->execute();
    $pedidos = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    // Precompute resumen por pedido
    $resumen = [];
    if ($pedidos) {
        $ids = array_map(fn($r) => (int)$r['id'], $pedidos);
        $in = implode(',', array_fill(0, count($ids), '?'));
        $sql = "SELECT d.pedido_id,
                       COUNT(DISTINCT d.id) AS destinatarios,
                       COUNT(i.id) AS items,
                       COALESCE(SUM(i.expected_uv),0) AS expected_uv,
                       COALESCE(SUM(i.expected_uc),0) AS expected_uc
                  FROM so_pedido_dest d
                  JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id
                 WHERE d.pedido_id IN ($in)
              GROUP BY d.pedido_id";
        $st2 = $pdo->prepare($sql);
        $st2->execute($ids);
        while ($row = $st2->fetch(PDO::FETCH_ASSOC)) {
            $resumen[(int)$row['pedido_id']] = [
                'destinatarios' => (int)$row['destinatarios'],
                'items'         => (int)$row['items'],
                'expected_uv'   => (int)$row['expected_uv'],
                'expected_uc'   => (int)$row['expected_uc'],
            ];
        }
    }

    // DataTables expects an array of rows; keep flat keys with namespaces like ingresos endpoint
    $data = [];
    foreach ($pedidos as $p) {
        $pid = (int)$p['id'];
        $data[] = [
            'so_pedido' => [
                'id'           => $pid,
                'codigo'       => (string)$p['codigo'],
                'cliente_ref'  => (string)$p['cliente_ref'],
                'fecha_pedido' => (string)$p['fecha_pedido'],
                'created_at'   => (string)($p['created_at'] ?? ''),
            ],
            'estado' => [ 'code' => (string)($p['estado_code'] ?? '') ],
            'resumen' => $resumen[$pid] ?? [ 'destinatarios'=>0, 'items'=>0, 'expected_uv'=>0, 'expected_uc'=>0 ],
        ];
    }

    out(['data' => $data]);
} catch (Throwable $e) {
    out(['error' => 'No se pudo cargar el listado', 'message' => $e->getMessage()], 500);
}
