<?php

declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/SoPreparacionReport.php';

$autoload = $ROOT . '/vendor/autoload.php';
if (!is_file($autoload)) {
    http_response_code(500);
    echo 'Falta vendor/autoload.php. Instala dependencias con Composer.';
    exit;
}
require_once $autoload;

if (!class_exists(\Mpdf\Mpdf::class)) {
    http_response_code(500);
    echo 'No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf';
    exit;
}

function fmt_num($value): string
{
    return number_format((float) $value, 0, ',', '.');
}

try {
    $preId = isset($_GET['pre_id']) ? (int) $_GET['pre_id'] : 0;
    $soId = isset($_GET['so_id']) ? (int) $_GET['so_id'] : 0;

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $dataset = so_pre_fetch_dataset($pdo, [
        'pre_id' => $preId,
        'so_id' => $soId,
    ]);

    if (!$dataset['pre']) {
        http_response_code(404);
        echo 'No se encontró la preparación solicitada.';
        exit;
    }

    $pre = $dataset['pre'];
    $pedido = $dataset['pedido'] ?? [];
    $rows = $dataset['rows'];
    $totals = $dataset['totals'];

    $preCodigo = (string) ($pre['codigo'] ?? ('PRE-' . $pre['id']));
    $pedidoCodigo = (string) ($pedido['codigo'] ?? '');
    $cliente = (string) ($pedido['cliente_nombre'] ?? $pedido['cliente'] ?? '');
    $deposito = (string) ($pre['deposito'] ?? '');
    $fechaPre = (string) ($pre['created_at'] ?? $pre['asignado_at'] ?? '');

    $fileName = 'Picking_' . preg_replace('/[^A-Za-z0-9_-]+/', '_', $preCodigo) . '.pdf';
    header('Content-Disposition: inline; filename="' . $fileName . '"');

    $mpdf = new \Mpdf\Mpdf([
        'format' => 'A4',
        'margin_left' => 10,
        'margin_right' => 10,
        'margin_top' => 12,
        'margin_bottom' => 12,
    ]);

    $styles = '<style>
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
        h1 { font-size: 18px; margin: 0 0 8px 0; }
        .muted { color: #666; }
        .section { margin-top: 10px; }
        .kv { display: flex; flex-wrap: wrap; gap: 12px; }
        .kv div { min-width: 180px; }
        table { width: 100%; border-collapse: collapse; margin-top: 8px; }
        th, td { border: 0.6px solid #999; padding: 4px 6px; }
        th { background: #f3f3f3; }
        .right { text-align: right; }
        .small { font-size: 10px; }
    </style>';

    ob_start();
    ?>
    <?= $styles ?>
    <h1>Documento de Picking · <?= htmlspecialchars($preCodigo, ENT_QUOTES, 'UTF-8') ?></h1>
    <div class="muted small">Generado: <?= date('Y-m-d H:i') ?></div>

    <div class="section">
        <strong>Resumen</strong>
        <div class="kv">
            <div><b>PRE:</b> <?= htmlspecialchars($preCodigo, ENT_QUOTES, 'UTF-8') ?></div>
            <?php if ($pedidoCodigo !== ''): ?>
            <div><b>Pedido:</b> <?= htmlspecialchars($pedidoCodigo, ENT_QUOTES, 'UTF-8') ?></div>
            <?php endif; ?>
            <?php if ($cliente !== ''): ?>
            <div><b>Cliente:</b> <?= htmlspecialchars($cliente, ENT_QUOTES, 'UTF-8') ?></div>
            <?php endif; ?>
            <?php if ($deposito !== ''): ?>
            <div><b>Depósito:</b> <?= htmlspecialchars($deposito, ENT_QUOTES, 'UTF-8') ?></div>
            <?php endif; ?>
            <?php if ($fechaPre !== ''): ?>
            <div><b>Creado:</b> <?= htmlspecialchars($fechaPre, ENT_QUOTES, 'UTF-8') ?></div>
            <?php endif; ?>
        </div>
    </div>

    <div class="section">
        <table>
            <thead>
                <tr>
                    <th>#</th>
                    <th>Producto</th>
                    <th>Lote</th>
                    <th>Venc.</th>
                    <th>Pallet</th>
                    <th>Desde</th>
                    <th>Hasta</th>
                    <th class="right">UV</th>
                    <th class="right">UC</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!$rows): ?>
                <tr><td colspan="9" class="muted">Sin movimientos registrados para esta preparación.</td></tr>
                <?php else: ?>
                <?php $idx = 0; foreach ($rows as $row): $idx++; ?>
                <tr>
                    <td><?= $idx ?></td>
                    <td><?= htmlspecialchars((string) ($row['producto'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars((string) ($row['lote'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars((string) ($row['vto'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars((string) ($row['pallet'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars((string) ($row['from_pos'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars((string) ($row['to_pos'] ?? ''), ENT_QUOTES, 'UTF-8') ?></td>
                    <td class="right"><?= fmt_num($row['uv_cajas'] ?? 0) ?></td>
                    <td class="right"><?= fmt_num($row['uc_unidades'] ?? 0) ?></td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
            <tfoot>
                <tr>
                    <th colspan="7" class="right">Totales</th>
                    <th class="right"><?= fmt_num($totals['uv'] ?? 0) ?></th>
                    <th class="right"><?= fmt_num($totals['uc'] ?? 0) ?></th>
                </tr>
            </tfoot>
        </table>
    </div>

    <?php if ($rows): ?>
    <div class="section small">Incluye <?= count($rows) ?> movimientos de picking.</div>
    <?php endif; ?>
    <?php
    $html = ob_get_clean();
    $mpdf->WriteHTML($html);
    $mpdf->Output();

} catch (Throwable $e) {
    http_response_code(500);
    echo 'Error generando PDF: ' . $e->getMessage();
}
