<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/SoPreparacionErrors.php';

function so_pre_err_out(array $payload, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function so_pre_err_parse_datetime(string $value): ?string
{
    $value = trim($value);
    if ($value === '') {
        return null;
    }

    $normalized = str_replace('T', ' ', $value);
    if (strlen($normalized) === 16) {
        $normalized .= ':00';
    }

    $formats = ['Y-m-d H:i:s', 'Y-m-d H:i'];
    foreach ($formats as $fmt) {
        $dt = DateTime::createFromFormat($fmt, $normalized);
        if ($dt instanceof DateTime) {
            return $dt->format('Y-m-d H:i:s');
        }
    }

    $ts = strtotime($value);
    if ($ts !== false) {
        return date('Y-m-d H:i:s', $ts);
    }

    return null;
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        so_pre_err_out(['ok' => false, 'error' => 'Método no permitido'], 405);
    }

    $payload = $_POST;
    if (!$payload) {
        $raw = file_get_contents('php://input') ?: '';
        if ($raw !== '') {
            $payload = json_decode($raw, true) ?: [];
        }
    }

    $soId = isset($payload['so_id']) ? (int) $payload['so_id'] : 0;
    $motivoId = isset($payload['motivo_id']) ? (int) $payload['motivo_id'] : 0;
    $responsableId = isset($payload['responsable_user_id']) ? (int) $payload['responsable_user_id'] : null;
    $responsableNombre = isset($payload['responsable_nombre']) ? trim((string) $payload['responsable_nombre']) : '';
    $observacion = isset($payload['observacion']) ? trim((string) $payload['observacion']) : '';
    $loggedAtInput = isset($payload['logged_at']) ? (string) $payload['logged_at'] : '';
    $preId = isset($payload['pre_id']) ? (int) $payload['pre_id'] : null;

    if ($soId <= 0) {
        so_pre_err_out(['ok' => false, 'error' => 'so_id requerido'], 422);
    }
    if ($motivoId <= 0) {
        so_pre_err_out(['ok' => false, 'error' => 'motivo_id requerido'], 422);
    }

    $loggedAt = so_pre_err_parse_datetime($loggedAtInput);
    if (!$loggedAt) {
        so_pre_err_out(['ok' => false, 'error' => 'Fecha y hora inválidas'], 422);
    }

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    if (!so_prep_err_table_exists($pdo, so_prep_err_table_name())) {
        so_pre_err_out([
            'ok' => false,
            'error' => 'Tabla de errores no encontrada. Ejecute la migración correspondiente.',
            'table' => so_prep_err_table_name(),
        ], 500);
    }

    if ($preId === null || $preId <= 0) {
        $preInfo = so_prep_err_resolve_pre_info($pdo, $soId);
        if ($preInfo) {
            $preId = (int) $preInfo['id'];
        }
    }

    if ($responsableId && $responsableNombre === '' && so_prep_err_table_exists($pdo, 'sys_users')) {
        $nameCol = so_prep_err_detect_column($pdo, 'sys_users', ['full_name', 'nombre', 'name']);
        $altCol = so_prep_err_detect_column($pdo, 'sys_users', ['username', 'user', 'email']);
        $cols = [];
        if ($nameCol) {
            $cols[] = "`{$nameCol}`";
        }
        if ($altCol) {
            $cols[] = "`{$altCol}`";
        }
        if (!$cols) {
            $cols[] = "CONCAT('Usuario #', id)";
        }
        $sql = 'SELECT ' . implode(', ', $cols) . ' FROM sys_users WHERE id = ? LIMIT 1';
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$responsableId]);
        $userRow = $stmt->fetch(PDO::FETCH_NUM) ?: [];
        foreach ($userRow as $value) {
            $text = trim((string) $value);
            if ($text !== '') {
                $responsableNombre = $text;
                break;
            }
        }
    }

    $responsableNombre = substr($responsableNombre, 0, 190);
    $observacion = $observacion !== '' ? substr($observacion, 0, 1000) : null;

    $insertPayload = [
        'so_id' => $soId,
        'pre_id' => $preId,
        'motivo_id' => $motivoId,
        'responsable_user_id' => $responsableId ?: null,
        'responsable_nombre' => $responsableNombre !== '' ? $responsableNombre : null,
        'observacion' => $observacion,
        'logged_at' => $loggedAt,
    ];

    $id = so_prep_err_insert($pdo, $insertPayload);

    so_pre_err_out(['ok' => true, 'id' => $id]);
} catch (Throwable $e) {
    so_pre_err_out([
        'ok' => false,
        'error' => 'Error registrando el error',
        'message' => $e->getMessage(),
    ], 500);
}
