<?php
declare(strict_types=1);

/**
 * API: Actualizar tiempos/responsable/foto del PRE (Paso 5)
 * Método: POST (multipart/form-data o JSON)
 * Campos admitidos:
 *  - pre_id (int) obligatorio
 *  - action: 'asignar' | 'inicio' | 'fin' | 'set' (set campos explícitos)
 *  - tipo: 'PREPARACION' | 'CONTROL' (default PREPARACION)
 *  - responsable_user_id (int, opcional)
 *  - inicio_at / fin_at (YYYY-mm-dd HH:ii:ss, opcionales si action='set')
 *  - observacion (string, opcional)
 *  - foto (file, opcional)
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out(array $p, int $c=200){ http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

try {
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') out(['ok'=>false,'error'=>'Método no permitido'], 405);

  $payload = $_POST;
  if (empty($payload) && !empty($_SERVER['CONTENT_TYPE']) && stripos($_SERVER['CONTENT_TYPE'], 'application/json') !== false) {
    $raw = file_get_contents('php://input') ?: '';
    $payload = json_decode($raw, true) ?: [];
  }

  $preId = (int)($payload['pre_id'] ?? 0);
  $action = strtolower(trim((string)($payload['action'] ?? 'set')));
  $tipo = strtoupper(trim((string)($payload['tipo'] ?? 'PREPARACION')));
  if ($tipo !== 'PREPARACION' && $tipo !== 'CONTROL') $tipo = 'PREPARACION';
  $responsableId = isset($payload['responsable_user_id']) ? (int)$payload['responsable_user_id'] : null;
  $inicioAt = isset($payload['inicio_at']) ? (string)$payload['inicio_at'] : null;
  $finAt    = isset($payload['fin_at']) ? (string)$payload['fin_at'] : null;
  $obs      = isset($payload['observacion']) ? (string)$payload['observacion'] : null;
  if ($preId <= 0) out(['ok'=>false,'error'=>'pre_id requerido'], 422);

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  // asegurar existencia
  $st = $pdo->prepare('SELECT id FROM so_preembarque WHERE id=? LIMIT 1');
  $st->execute([$preId]);
  if (!(int)$st->fetchColumn()) out(['ok'=>false,'error'=>'Preembarque no encontrado'], 404);

  // upsert tarea para el tipo solicitado
  $stFindT = $pdo->prepare("SELECT id FROM so_pre_tarea WHERE preembarque_id=? AND tipo=? LIMIT 1");
  $stFindT->execute([$preId, $tipo]);
  $tId = (int)$stFindT->fetchColumn();
  if ($tId <= 0) {
    $pdo->prepare("INSERT INTO so_pre_tarea (preembarque_id, tipo) VALUES (?, ?)")->execute([$preId, $tipo]);
    $tId = (int)$pdo->lastInsertId();
  }

  // actualizar por acción
  if ($action === 'asignar') {
    // Marcar asignado_at si no estaba asignado, pero actualizar responsable siempre que venga
    $pdo->prepare('UPDATE so_preembarque SET asignado_at = COALESCE(asignado_at, NOW()) WHERE id=?')->execute([$preId]);
    if ($responsableId) $pdo->prepare('UPDATE so_pre_tarea SET responsable_user_id=? WHERE id=?')->execute([$responsableId, $tId]);
    // Estado: ASIGNADO para inicio de ciclo de PREPARACION
    if ($tipo === 'PREPARACION') {
      $stE = $pdo->prepare("SELECT id FROM so_preembarque_estado WHERE code='ASIGNADO' LIMIT 1");
      $stE->execute();
      $estId = (int)$stE->fetchColumn();
      if ($estId > 0) $pdo->prepare('UPDATE so_preembarque SET estado_id=? WHERE id=?')->execute([$estId, $preId]);
    }
  } elseif ($action === 'inicio') {
    // Si el usuario pulsa Inicio explícitamente, registramos NOW() (aunque ya exista)
    $pdo->prepare('UPDATE so_preembarque SET inicio_at=NOW() WHERE id=?')->execute([$preId]);
    $pdo->prepare('UPDATE so_pre_tarea SET inicio_at=NOW() WHERE id=?')->execute([$tId]);
    // Estado: EN_PROCESO al iniciar PREPARACION
    if ($tipo === 'PREPARACION') {
      $stE = $pdo->prepare("SELECT id FROM so_preembarque_estado WHERE code='EN_PROCESO' LIMIT 1");
      $stE->execute();
      $estId = (int)$stE->fetchColumn();
      if ($estId > 0) $pdo->prepare('UPDATE so_preembarque SET estado_id=? WHERE id=?')->execute([$estId, $preId]);
    }
  } elseif ($action === 'fin') {
    // El botón Finalizar debe guardar la hora siempre
    $pdo->prepare('UPDATE so_preembarque SET fin_at=NOW() WHERE id=?')->execute([$preId]);
    $pdo->prepare('UPDATE so_pre_tarea SET fin_at=NOW() WHERE id=?')->execute([$tId]);
    // Estado: COMPLETADO al finalizar CONTROL (cierra el pre-embarque)
    if ($tipo === 'CONTROL') {
      $stE = $pdo->prepare("SELECT id FROM so_preembarque_estado WHERE code='COMPLETADO' LIMIT 1");
      $stE->execute();
      $estId = (int)$stE->fetchColumn();
      if ($estId > 0) $pdo->prepare('UPDATE so_preembarque SET estado_id=? WHERE id=?')->execute([$estId, $preId]);
    }
  } else { // set explícito
    if ($inicioAt) $pdo->prepare('UPDATE so_preembarque SET inicio_at=? WHERE id=?')->execute([$inicioAt, $preId]);
    if ($finAt)    $pdo->prepare('UPDATE so_preembarque SET fin_at=? WHERE id=?')->execute([$finAt, $preId]);
    if ($responsableId) $pdo->prepare('UPDATE so_pre_tarea SET responsable_user_id=? WHERE id=?')->execute([$responsableId, $tId]);
  }
  if ($obs !== null) {
    // Guardar observación general y también la nota específica de la tarea
    $pdo->prepare('UPDATE so_preembarque SET observacion=? WHERE id=?')->execute([$obs, $preId]);
    $pdo->prepare('UPDATE so_pre_tarea SET nota=? WHERE id=?')->execute([$obs, $tId]);
  }

  // foto (opcional)
  $fotoUrl = null;
  if (!empty($_FILES['foto']) && is_uploaded_file($_FILES['foto']['tmp_name'])) {
    $dir = $ROOT . '/public/storage/preparacion';
    if (!is_dir($dir)) @mkdir($dir, 0775, true);
    // guardar por tipo (prep/control) para no pisar
    $suffix = ($tipo === 'CONTROL') ? '-ctrl' : '';
    $dest = $dir . '/' . $preId . $suffix . '.jpg';
    if (!@move_uploaded_file($_FILES['foto']['tmp_name'], $dest)) {
      // fallback copy
      @copy($_FILES['foto']['tmp_name'], $dest);
    }
    if (is_file($dest)) {
      $fotoUrl = '/storage/preparacion/' . $preId . $suffix . '.jpg';
      $pdo->prepare('UPDATE so_pre_tarea SET foto_url=? WHERE id=?')->execute([$fotoUrl, $tId]);
    }
  }

  out(['ok'=>true,'pre_id'=>$preId,'tarea_id'=>$tId,'tipo'=>$tipo,'foto_url'=>$fotoUrl]);

} catch (Throwable $e) {
  out(['ok'=>false,'error'=>'Error actualizando tiempos','message'=>$e->getMessage()], 500);
}

?>
