<?php
declare(strict_types=1);

/**
 * API: Preparación automática (FEFO) hacia zona PREP
 * Llama al SP sp_so_preparar_auto, que:
 *  - Crea/usa un so_preembarque PRE-{pedido}
 *  - Elige stock por FEFO (excluye PREP/CUARENTENA)
 *  - Inserta wh_move MOVE (from → PREP) y so_pre_pick
 *  - Actualiza prepared_* vía trigger
 *
 * Entrada: POST JSON|form
 *  - so_id | so_codigo (uno requerido)
 *  - deposito_code (opcional, default 'DEP1')
 *  - prep_posicion_id (opcional): si se envía, se usará esa posición como PREP
 *                                 (se resuelve a code para el SP)
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out_pre(array $p, int $c = 200): void { http_response_code($c); echo json_encode($p, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); exit; }

function ensureSoPedidoEstados(PDO $pdo): void {
    static $done = false;
    if ($done) return;
    $rows = [
        ['RECIBIDO', 'Recibido', 1],
        ['EN_PREPARACION', 'En preparación', 2],
        ['PARCIAL', 'Parcialmente preparado', 3],
        ['PREPARADO', 'Preparado', 4],
        ['EN_EMBARQUE', 'En embarque', 5],
        ['DESPACHADO', 'Despachado', 6],
        ['CERRADO', 'Cerrado', 7],
        ['CANCELADO', 'Cancelado', 9],
    ];
    $stmt = $pdo->prepare("INSERT INTO so_pedido_estado (code, nombre, orden) VALUES (?,?,?) ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden)");
    foreach ($rows as $row) {
        try { $stmt->execute($row); } catch (Throwable $e) { /* ignore */ }
    }
    $done = true;
}

function ensureSoPreembarqueEstados(PDO $pdo): void {
    static $done = false;
    if ($done) return;
    $rows = [
        ['PENDIENTE', 'Pendiente', 1],
        ['ASIGNADO', 'Asignado', 2],
        ['EN_PROCESO', 'En proceso', 3],
        ['COMPLETADO', 'Completado', 4],
        ['CANCELADO', 'Cancelado', 9],
    ];
    $stmt = $pdo->prepare("INSERT INTO so_preembarque_estado (code, nombre, orden) VALUES (?,?,?) ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden)");
    foreach ($rows as $row) {
        try { $stmt->execute($row); } catch (Throwable $e) { /* ignore */ }
    }
    $done = true;
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        out_pre(['ok'=>false,'error'=>'Método no permitido'], 405);
    }

    // Configurar timeout
    ini_set('max_execution_time', 60);
    set_time_limit(60);

    $raw = file_get_contents('php://input');
    $isJson = isset($_SERVER['CONTENT_TYPE']) && stripos((string)$_SERVER['CONTENT_TYPE'], 'application/json') !== false;
    $payload = [];
    if ($isJson && $raw) {
        $payload = json_decode($raw, true) ?: [];
    }
    // mezclar con POST form si aplica
    $payload = array_merge($_POST, $payload);

    $soId      = isset($payload['so_id']) ? (int)$payload['so_id'] : 0;
    $soCodigo  = isset($payload['so_codigo']) ? trim((string)$payload['so_codigo']) : '';
    $depCode   = isset($payload['deposito_code']) && $payload['deposito_code'] !== '' ? trim((string)$payload['deposito_code']) : 'DEP1';
    $prepPosId = isset($payload['prep_posicion_id']) && $payload['prep_posicion_id'] !== '' ? (int)$payload['prep_posicion_id'] : null;
    $debug     = isset($payload['debug']) && (string)$payload['debug'] !== '' ? (int)$payload['debug'] : 0;
    $simulate  = isset($payload['simulate']) && (string)$payload['simulate'] !== '' ? (int)$payload['simulate'] : 0;
    $ucToUv    = isset($payload['uc_to_uv']) && (string)$payload['uc_to_uv'] !== '' ? (int)$payload['uc_to_uv'] : 0;
    $uvToUc    = isset($payload['uv_to_uc']) && (string)$payload['uv_to_uc'] !== '' ? (int)$payload['uv_to_uc'] : 1;

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    ensureSoPedidoEstados($pdo);
    ensureSoPreembarqueEstados($pdo);
    
    // Set session settings to match CLI behavior
    $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    $pdo->exec('SET SESSION autocommit = 1');
    $pdo->exec('SET SESSION sql_mode = "STRICT_TRANS_TABLES,NO_ZERO_DATE,NO_ZERO_IN_DATE,ERROR_FOR_DIVISION_BY_ZERO"');
    // Enable SP logging and UC->UV conversion as requested
    try {
        $pdo->exec('SET @so_pre_log := ' . (int)$debug);
        $pdo->exec('SET @so_pre_uc_to_uv := ' . (int)$ucToUv);
        $pdo->exec('SET @so_pre_uv_to_uc := ' . (int)$uvToUc);
    } catch (Throwable $e) { /* no-op */ }

    // Resolver código del pedido si sólo viene id
    if ($soCodigo === '' && $soId > 0) {
        $st = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id=? LIMIT 1');
        $st->execute([$soId]);
        $soCodigo = (string)$st->fetchColumn();
    }
    if ($soCodigo === '') out_pre(['ok'=>false,'error'=>'Debe indicar so_id o so_codigo'], 422);

    // Resolver código de la posición PREP si se indica ID (SP recibe code)
    $prepPosCode = null;
    if ($prepPosId !== null) {
        $st = $pdo->prepare('SELECT code FROM wh_posicion WHERE id=? LIMIT 1');
        $st->execute([$prepPosId]);
        $prepPosCode = $st->fetchColumn() ?: null;
        if ($prepPosCode === null) out_pre(['ok'=>false,'error'=>"prep_posicion_id inválido o no existe (id={$prepPosId})"], 422);
    }

    // Validar que el SP exista
    $chk = $pdo->query("SHOW PROCEDURE STATUS WHERE Name='sp_so_preparar_auto'")->fetch(PDO::FETCH_ASSOC);
    if (!$chk) {
        out_pre(['ok'=>false,'error'=>'No existe sp_so_preparar_auto. Aplique las migraciones step9/step10.'], 500);
    }

    // Ejecutar SP
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?,?,?,?)');
    $stmt->execute([$soCodigo, $depCode, $prepPosCode, $simulate]);
    
    // Recoger resultsets con protección contra loops infinitos
    $sets = [];
    $maxResultSets = 10; // límite de seguridad
    $resultSetCount = 0;
    
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && $rows) $sets[] = $rows;
        $resultSetCount++;
        
        // Protección contra loops infinitos
        if ($resultSetCount >= $maxResultSets) {
            break;
        }
    } while ($stmt->nextRowset());
    
    $resumen = $sets ? end($sets) : [];
    $plan = null;
    if ($simulate && count($sets) >= 2) {
        // penúltimo es el plan
        if (count($sets) === 2) { $plan = $sets[0]; }
        else { $plan = $sets[count($sets)-2]; }
    }

    // Liberar statement explícitamente
    $stmt->closeCursor();
    $stmt = null;

    if ($debug) {
        error_log("sp_so_preparar_auto completed for $soCodigo. Result sets: $resultSetCount");
    }

    // Datos de pre-embarque para link a documento
    $preCode = 'PRE-' . $soCodigo;
    $stPre = $pdo->prepare('SELECT id, zona_posicion_id FROM so_preembarque WHERE codigo=? LIMIT 1');
    $stPre->execute([$preCode]);
    $pre = $stPre->fetch(PDO::FETCH_ASSOC) ?: null;
    $docUrl = null;
    if ($pre) {
        $docUrl = url('/salidas/preparacion/doc') . '?pre_id=' . (int)$pre['id'];
        
        // Actualizar estado del pedido a REPOSICION si no es simulación
        if (!$simulate && $soId > 0) {
            $stEstado = $pdo->prepare('SELECT id FROM so_pedido_estado WHERE code=? LIMIT 1');
            $stEstado->execute(['REPOSICION']);
            $estadoReposicionId = $stEstado->fetchColumn();
            
            if ($estadoReposicionId) {
                $stUpdate = $pdo->prepare('UPDATE so_pedido SET estado_id=? WHERE id=?');
                $stUpdate->execute([$estadoReposicionId, $soId]);
            }
        }
    }

    $debugLogs = null;
    if ($debug && $pre) {
        try {
            $stDbg = $pdo->prepare('SELECT * FROM so_preparar_auto_log WHERE (pedido_codigo=? OR pre_id=?) ORDER BY id DESC LIMIT 200');
            $stDbg->execute([$soCodigo, (int)($pre['id'] ?? 0)]);
            $debugLogs = $stDbg->fetchAll(PDO::FETCH_ASSOC);
        } catch (Throwable $e) {
            $debugLogs = [['error' => 'no debug logs', 'message' => $e->getMessage()]];
        }
    }

    out_pre([
        'ok'      => true,
        'pedido'  => [ 'codigo' => $soCodigo ],
        'pre'     => $pre ?: null,
        'resumen' => $resumen,
        'doc_url' => $docUrl,
        'debug'   => $debug ? $debugLogs : null,
        'simulate'=> (bool)$simulate,
        'plan'    => $plan,
    ]);
} catch (Throwable $e) {
    out_pre(['ok'=>false,'error'=>'No se pudo ejecutar la preparación automática','message'=>$e->getMessage()], 500);
}
