<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
ini_set('max_execution_time', 60);

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    // Get JSON payload
    $raw = file_get_contents('php://input');
    $payload = json_decode($raw, true) ?: [];
    
    $soId = (int)($payload['so_id'] ?? 0);
    if ($soId <= 0) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'so_id required']);
        exit;
    }
    
    // Use exact same approach as working CLI test
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Get pedido codigo
    $stmt = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id=? LIMIT 1');
    $stmt->execute([$soId]);
    $soCodigo = $stmt->fetchColumn();
    
    if (!$soCodigo) {
        http_response_code(404);
        echo json_encode(['ok' => false, 'error' => 'Pedido not found']);
        exit;
    }
    
    // Execute SP exactly like CLI test (no special settings)
    $start = microtime(true);
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?,?,?,?)');
    $stmt->execute([$soCodigo, 'DEP1', null, 0]);
    
    // Collect results exactly like CLI test
    $sets = [];
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && $rows) {
            $sets[] = $rows;
        }
    } while ($stmt->nextRowset());
    
    $duration = round((microtime(true) - $start) * 1000, 2);
    
    // Get pre-embarque data
    $preCode = 'PRE-' . $soCodigo;
    $stPre = $pdo->prepare('SELECT id FROM so_preembarque WHERE codigo=? LIMIT 1');
    $stPre->execute([$preCode]);
    $preId = $stPre->fetchColumn();
    
    // Update estado to REPOSICION
    if ($preId) {
        $stEstado = $pdo->prepare('SELECT id FROM so_pedido_estado WHERE code=? LIMIT 1');
        $stEstado->execute(['REPOSICION']);
        $estadoId = $stEstado->fetchColumn();
        
        if ($estadoId) {
            $stUpdate = $pdo->prepare('UPDATE so_pedido SET estado_id=? WHERE id=?');
            $stUpdate->execute([$estadoId, $soId]);
        }
    }
    
    echo json_encode([
        'ok' => true,
        'pedido' => ['codigo' => $soCodigo],
        'pre' => ['id' => $preId],
        'doc_url' => $preId ? url('/salidas/preparacion/doc') . '?pre_id=' . $preId : null,
        'duration_ms' => $duration,
        'resultsets' => count($sets)
    ]);
    
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false, 
        'error' => $e->getMessage(),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ]);
}
?>