<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out_pre(array $p, int $c = 200): void { 
    http_response_code($c); 
    echo json_encode($p, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); 
    exit; 
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        out_pre(['ok'=>false,'error'=>'Método no permitido'], 405);
    }

    // Configurar timeout
    ini_set('max_execution_time', 60);
    set_time_limit(60);

    $raw = file_get_contents('php://input');
    $isJson = isset($_SERVER['CONTENT_TYPE']) && stripos((string)$_SERVER['CONTENT_TYPE'], 'application/json') !== false;
    $payload = [];
    if ($isJson && $raw) {
        $payload = json_decode($raw, true) ?: [];
    }
    $payload = array_merge($_POST, $payload);

    $soId      = isset($payload['so_id']) ? (int)$payload['so_id'] : 0;
    $soCodigo  = isset($payload['so_codigo']) ? trim((string)$payload['so_codigo']) : '';
    $depCode   = isset($payload['deposito_code']) && $payload['deposito_code'] !== '' ? trim((string)$payload['deposito_code']) : 'DEP1';
    $prepPosId = isset($payload['prep_posicion_id']) && $payload['prep_posicion_id'] !== '' ? (int)$payload['prep_posicion_id'] : null;
    $debug     = isset($payload['debug']) && (string)$payload['debug'] !== '' ? (int)$payload['debug'] : 0;
    $simulate  = isset($payload['simulate']) && (string)$payload['simulate'] !== '' ? (int)$payload['simulate'] : 0;

    // Get PDO with exact same settings as CLI
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Use exact same session settings as CLI that works
    $pdo->exec("SET NAMES utf8mb4");
    $pdo->exec("SET SESSION autocommit = 0"); // Match CLI behavior
    $pdo->exec("SET SESSION transaction_isolation = 'READ-COMMITTED'");

    // Resolver código del pedido si sólo viene id
    if ($soCodigo === '' && $soId > 0) {
        $st = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id=? LIMIT 1');
        $st->execute([$soId]);
        $soCodigo = (string)$st->fetchColumn();
    }
    if ($soCodigo === '') out_pre(['ok'=>false,'error'=>'Debe indicar so_id o so_codigo'], 422);

    // Resolver código de la posición PREP si se indica ID
    $prepPosCode = null;
    if ($prepPosId !== null) {
        $st = $pdo->prepare('SELECT code FROM wh_posicion WHERE id=? LIMIT 1');
        $st->execute([$prepPosId]);
        $prepPosCode = $st->fetchColumn() ?: null;
        if ($prepPosCode === null) out_pre(['ok'=>false,'error'=>"prep_posicion_id inválido o no existe (id={$prepPosId})"], 422);
    }

    // Execute SP exactly like the working CLI version
    $start = microtime(true);
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?,?,?,?)');
    $success = $stmt->execute([$soCodigo, $depCode, $prepPosCode, $simulate]);
    
    if (!$success) {
        out_pre(['ok'=>false,'error'=>'Failed to execute stored procedure'], 500);
    }

    // Collect results like CLI version - simple approach
    $results = [];
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows) {
            $results[] = $rows;
        }
    } while ($stmt->nextRowset() && count($results) < 5); // safety limit

    $stmt->closeCursor();
    $stmt = null;
    
    $duration = round((microtime(true) - $start) * 1000, 2);

    // Get pre-embarque data
    $preCode = 'PRE-' . $soCodigo;
    $stPre = $pdo->prepare('SELECT id, zona_posicion_id FROM so_preembarque WHERE codigo=? LIMIT 1');
    $stPre->execute([$preCode]);
    $pre = $stPre->fetch(PDO::FETCH_ASSOC) ?: null;
    
    $docUrl = null;
    if ($pre) {
        $docUrl = url('/salidas/preparacion/doc') . '?pre_id=' . (int)$pre['id'];
        
        // Update pedido state to REPOSICION if not simulation
        if (!$simulate && $soId > 0) {
            $stEstado = $pdo->prepare('SELECT id FROM so_pedido_estado WHERE code=? LIMIT 1');
            $stEstado->execute(['REPOSICION']);
            $estadoReposicionId = $stEstado->fetchColumn();
            
            if ($estadoReposicionId) {
                $stUpdate = $pdo->prepare('UPDATE so_pedido SET estado_id=? WHERE id=?');
                $stUpdate->execute([$estadoReposicionId, $soId]);
            }
        }
    }

    // Commit all changes
    $pdo->commit();

    out_pre([
        'ok'      => true,
        'pedido'  => ['codigo' => $soCodigo],
        'pre'     => $pre ?: null,
        'resumen' => $results ? end($results) : [],
        'doc_url' => $docUrl,
        'debug'   => $debug ? "SP executed in {$duration}ms" : null,
        'simulate'=> (bool)$simulate,
        'duration_ms' => $duration,
    ]);

} catch (Throwable $e) {
    // Rollback on error
    try {
        if (isset($pdo)) $pdo->rollBack();
    } catch (Throwable $rollbackError) {
        // ignore rollback errors
    }
    
    out_pre([
        'ok'=>false,
        'error'=>'No se pudo ejecutar la preparación automática',
        'message'=>$e->getMessage(),
        'file'=>basename($e->getFile()),
        'line'=>$e->getLine()
    ], 500);
}
?>