<?php
declare(strict_types=1);

/**
 * API: Preparación rápida (solo PICKING)
 * - Crea/obtiene preembarque PRE-<so_codigo>
 * - Consume necesidades pendientes solo desde posiciones de ambiente PICKING (FEFO por vencimiento)
 * - Inserta wh_move (MOVE, PREPARACION) y so_pre_pick
 * - No realiza reposición desde reserva (rápido y no bloqueante)
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out_fast(array $p, int $c = 200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

try {
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') out_fast(['ok'=>false,'error'=>'Método no permitido'], 405);

  $raw = file_get_contents('php://input') ?: '';
  $isJson = isset($_SERVER['CONTENT_TYPE']) && stripos((string)$_SERVER['CONTENT_TYPE'], 'application/json') !== false;
  $payload = [];
  if ($isJson && $raw !== '') $payload = json_decode($raw, true) ?: [];
  $payload = array_merge($_POST, $payload);

  $soId = (int)($payload['so_id'] ?? 0);
  $depositoCode = trim((string)($payload['deposito_code'] ?? 'DEP1'));
  if ($soId <= 0) out_fast(['ok'=>false,'error'=>'so_id requerido'], 422);

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');
  // Reducir locking
  $pdo->exec('SET SESSION innodb_lock_wait_timeout = 30');
  $pdo->exec('SET SESSION lock_wait_timeout = 30');
  $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');

  // Datos base
  $st = $pdo->prepare('SELECT id, codigo FROM so_pedido WHERE id=? LIMIT 1');
  $st->execute([$soId]);
  $so = $st->fetch(PDO::FETCH_ASSOC);
  if (!$so) out_fast(['ok'=>false,'error'=>'Pedido no encontrado'], 404);
  $soCodigo = (string)$so['codigo'];

  $st = $pdo->prepare('SELECT id FROM wh_deposito WHERE code=? LIMIT 1');
  $st->execute([$depositoCode]);
  $depId = (int)$st->fetchColumn();
  if ($depId <= 0) out_fast(['ok'=>false,'error'=>'Depósito no encontrado'], 422);

  // Posición PREP menos ocupada
  $sqlPrep = "
    SELECT p.id
      FROM wh_posicion p
      JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP'
      LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=?
     WHERE p.deposito_id=? AND p.activo=1
     GROUP BY p.id
     ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
     LIMIT 1";
  $st = $pdo->prepare($sqlPrep);
  $st->execute([$depId, $depId]);
  $prepPosId = (int)$st->fetchColumn();
  if ($prepPosId <= 0) out_fast(['ok'=>false,'error'=>'No hay posiciones PREP en el depósito'], 422);

  // Preembarque
  $preCode = 'PRE-' . $soCodigo;
  $pdo->beginTransaction();
  $st = $pdo->prepare('SELECT id FROM so_preembarque WHERE codigo=? LIMIT 1 FOR UPDATE');
  $st->execute([$preCode]);
  $preId = (int)$st->fetchColumn();
  if ($preId <= 0) {
    $stE = $pdo->prepare('SELECT id FROM so_preembarque_estado WHERE code="PENDIENTE" LIMIT 1');
    $stE->execute();
    $preEstId = (int)$stE->fetchColumn();
    $stIns = $pdo->prepare('INSERT INTO so_preembarque (codigo, pedido_id, deposito_id, estado_id, zona_posicion_id, asignado_at, inicio_at) VALUES (?,?,?,?,?,NOW(),NOW())');
    $stIns->execute([$preCode, $soId, $depId, $preEstId ?: null, $prepPosId]);
    $preId = (int)$pdo->lastInsertId();
  } else {
    $stUp = $pdo->prepare('UPDATE so_preembarque SET zona_posicion_id=COALESCE(zona_posicion_id, ?) WHERE id=?');
    $stUp->execute([$prepPosId, $preId]);
  }

  // Ítems pendientes del pedido
  $sqlPend = "
    SELECT i.id AS item_id, i.producto_id, i.lote_codigo,
           GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv,
           GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc
      FROM so_pedido_dest_item i
      JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
     WHERE d.pedido_id=?
       AND (GREATEST(i.expected_uv - i.prepared_uv,0) > 0 OR GREATEST(i.expected_uc - i.prepared_uc,0) > 0)
  ";
  $st = $pdo->prepare($sqlPend);
  $st->execute([$soId]);
  $pend = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Preparar statements
  $stCand = $pdo->prepare("SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento AS venc
                             FROM wh_stock s
                             JOIN wh_posicion p ON p.id=s.posicion_id
                             JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
                             LEFT JOIN wh_lote l ON l.id=s.lote_id
                            WHERE s.deposito_id=? AND s.producto_id=?
                              AND (?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?))
                              AND (s.qty_uv>0 OR s.qty_uc>0)
                            ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC");
  $stMv = $pdo->prepare('INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)');
  $stPick = $pdo->prepare('INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por) VALUES (?,?,?,?,?,?,?,?,NULL)');

  $totalMoves = 0; $itemsTouched = 0;
  foreach ($pend as $it) {
    $itemId = (int)$it['item_id'];
    $prodId = (int)$it['producto_id'];
    $loteCode = (string)($it['lote_codigo'] ?? '');
    $needUv = max(0, (int)$it['need_uv']);
    $needUc = max(0, (int)$it['need_uc']);
    if ($needUv === 0 && $needUc === 0) continue;

    $stCand->execute([$depId, $prodId, $loteCode, $loteCode]);
    $cands = $stCand->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if (!$cands) continue;
    $itemsTouched++;

    foreach ($cands as $c) {
      if ($needUv <= 0 && $needUc <= 0) break;
      $fromPos = (int)$c['posicion_id'];
      $loteId  = $c['lote_id'] !== null ? (int)$c['lote_id'] : null;
      $palletId= $c['pallet_id'] !== null ? (int)$c['pallet_id'] : null;
      $curUv   = max(0, (int)$c['qty_uv']);
      $curUc   = max(0, (int)$c['qty_uc']);
      $takeUv  = min($curUv, $needUv);
      $takeUc  = min($curUc, $needUc);
      if ($takeUv === 0 && $takeUc === 0) continue;

      // Registrar movimiento y tarea de picking
      $stMv->execute([$depId, 'MOVE', 'PREPARACION', $palletId, $prodId, $loteId, $fromPos, $prepPosId, $takeUv, $takeUc, 'PRE-'.$soId]);
      $stPick->execute([$preId, $itemId, $fromPos, $prepPosId, $palletId, $loteId, $takeUv, $takeUc]);
      $totalMoves++;
      $needUv -= $takeUv; $needUc -= $takeUc;
    }
  }

  $pdo->commit();

  $resp = [
    'ok' => true,
    'message' => 'Preparación rápida ejecutada',
    'pre' => ['id' => $preId],
    'items_afectados' => $itemsTouched,
    'movimientos' => $totalMoves,
    'doc_url' => url('/salidas/preparacion/doc') . '?pre_id=' . $preId
  ];
  out_fast($resp);
} catch (Throwable $e) {
  try { if ($pdo && $pdo->inTransaction()) $pdo->rollBack(); } catch (Throwable $e2) {}
  out_fast(['ok'=>false,'error'=>'Error en preparación rápida','message'=>$e->getMessage()], 500);
}
