<?php
declare(strict_types=1);

/**
 * API: Preparación manual paso-a-paso (1 ítem)
 * - Resuelve depósito, posiciones PREP/PICKING (menos ocupadas)
 * - Asegura preembarque PRE-<so_codigo>
 * - Construye pendientes (_pend)
 * - Toma el primer ítem pendiente y ejecuta:
 *     a) consumo desde PICKING -> PREP (una iteración)
 *     b) si faltan, reposición desde otros ambientes -> PICKING (una iteración)
 *     c) reintento de pick desde PICKING -> PREP (una iteración)
 * - Devuelve estado, ids y residuos de necesidad
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out_json(array $p, int $c = 200): void {
  http_response_code($c);
  echo json_encode($p, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

try {
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') out_json(['ok'=>false,'error'=>'Método no permitido'], 405);

  $raw = file_get_contents('php://input') ?: '';
  $isJson = isset($_SERVER['CONTENT_TYPE']) && stripos((string)$_SERVER['CONTENT_TYPE'], 'application/json') !== false;
  $payload = [];
  if ($isJson && $raw !== '') $payload = json_decode($raw, true) ?: [];
  $payload = array_merge($_POST, $payload);

  $soId = (int)($payload['so_id'] ?? 0);
  $depositoCode = trim((string)($payload['deposito_code'] ?? 'DEP1'));
  if ($soId <= 0) out_json(['ok'=>false,'error'=>'so_id requerido'], 422);

  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4 COLLATE utf8mb4_0900_ai_ci');
  $pdo->exec("SET collation_connection = 'utf8mb4_0900_ai_ci'");

  // Base
  $st = $pdo->prepare('SELECT id, codigo FROM so_pedido WHERE id=?');
  $st->execute([$soId]);
  $so = $st->fetch(PDO::FETCH_ASSOC);
  if (!$so) out_json(['ok'=>false,'error'=>'Pedido no encontrado'], 404);
  $soCodigo = (string)$so['codigo'];

  $st = $pdo->prepare('SELECT id FROM wh_deposito WHERE code=?');
  $st->execute([$depositoCode]);
  $depId = (int)$st->fetchColumn();
  if ($depId <= 0) out_json(['ok'=>false,'error'=>'Depósito no encontrado'], 422);

  // Pos PREP menos ocupada
  $st = $pdo->prepare(
    "SELECT p.id
       FROM wh_posicion p
       JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP'
       LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=?
      WHERE p.deposito_id=? AND p.activo=1
      GROUP BY p.id
      ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
      LIMIT 1"
  );
  $st->execute([$depId, $depId]);
  $posPrepId = (int)$st->fetchColumn();
  if ($posPrepId <= 0) out_json(['ok'=>false,'error'=>'No hay posiciones PREP en el depósito'], 422);

  // Pos PICKING menos ocupada
  $st = $pdo->prepare(
    "SELECT p.id
       FROM wh_posicion p
       JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
       LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=?
      WHERE p.deposito_id=? AND p.activo=1
      GROUP BY p.id
      ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
      LIMIT 1"
  );
  $st->execute([$depId, $depId]);
  $posPickId = (int)$st->fetchColumn();
  if ($posPickId <= 0) out_json(['ok'=>false,'error'=>'No hay posiciones PICKING en el depósito'], 422);

  // Asegurar pre
  $preCode = 'PRE-' . $soCodigo;
  $pdo->beginTransaction();
  $st = $pdo->prepare('SELECT id FROM so_preembarque WHERE codigo=? FOR UPDATE');
  $st->execute([$preCode]);
  $preId = (int)$st->fetchColumn();
  if ($preId <= 0) {
    $estId = (int)$pdo->query("SELECT id FROM so_preembarque_estado WHERE code='PENDIENTE' LIMIT 1")->fetchColumn();
    $st2 = $pdo->prepare('INSERT INTO so_preembarque (codigo,pedido_id,deposito_id,estado_id,zona_posicion_id,asignado_at,inicio_at) VALUES (?,?,?,?,?,NOW(),NOW())');
    $st2->execute([$preCode, $soId, $depId, $estId ?: null, $posPrepId]);
    $preId = (int)$pdo->lastInsertId();
  } else {
    $pdo->prepare('UPDATE so_preembarque SET zona_posicion_id=COALESCE(zona_posicion_id, ?) WHERE id=?')->execute([$posPrepId, $preId]);
  }

  // Construir pendientes
  $pdo->exec('DROP TEMPORARY TABLE IF EXISTS _pend');
  $pdo->exec('CREATE TEMPORARY TABLE _pend (
    pedido_dest_item_id BIGINT UNSIGNED,
    pedido_dest_id      BIGINT UNSIGNED,
    destinatario_id     BIGINT UNSIGNED,
    producto_id         BIGINT UNSIGNED,
    lote_codigo         VARCHAR(64),
    need_uv             INT,
    need_uc             INT
  ) ENGINE=InnoDB');

  $sqlPend = "INSERT INTO _pend
  SELECT i.id, d.id, d.destinatario_id, i.producto_id, i.lote_codigo,
         GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv,
         GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc
    FROM so_pedido_dest_item i
    JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
   WHERE d.pedido_id = ?
     AND (GREATEST(i.expected_uv - i.prepared_uv,0) > 0 OR GREATEST(i.expected_uc - i.prepared_uc,0) > 0)";
  $pdo->prepare($sqlPend)->execute([$soId]);

  // Tomar un ítem
  $it = $pdo->query('SELECT pedido_dest_item_id, producto_id, lote_codigo, need_uv, need_uc FROM _pend LIMIT 1')->fetch(PDO::FETCH_ASSOC);
  if (!$it) { $pdo->commit(); out_json(['ok'=>true,'message'=>'Sin pendientes','pre'=>['id'=>$preId],'doc_url'=>url('/salidas/preparacion/doc').'?pre_id='.$preId]); }

  $itemId = (int)$it['pedido_dest_item_id'];
  $prodId = (int)$it['producto_id'];
  $loteCode = (string)($it['lote_codigo'] ?? '');
  $needUv = max(0, (int)$it['need_uv']);
  $needUc = max(0, (int)$it['need_uc']);

  // Candidatos PICKING
  $pdo->exec('DROP TEMPORARY TABLE IF EXISTS _cand_pick');
  $pdo->exec('CREATE TEMPORARY TABLE _cand_pick (
    posicion_id BIGINT UNSIGNED,
    lote_id     BIGINT UNSIGNED,
    pallet_id   BIGINT UNSIGNED,
    qty_uv      INT,
    qty_uc      INT,
    venc        DATE
  ) ENGINE=InnoDB');

  $st = $pdo->prepare("INSERT INTO _cand_pick (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
    SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
      FROM wh_stock s
      JOIN wh_posicion p ON p.id=s.posicion_id
      JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
      LEFT JOIN wh_lote l ON l.id=s.lote_id
     WHERE s.deposito_id = ? AND s.producto_id = ?
       AND (?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?))
       AND (s.qty_uv>0 OR s.qty_uc>0)
     ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC");
  $st->execute([$depId, $prodId, $loteCode, $loteCode]);

  // Pick 1 iteración
  $picked = ['uv'=>0,'uc'=>0];
  $row = $pdo->query('SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc FROM _cand_pick ORDER BY COALESCE(venc,\'9999-12-31\') ASC LIMIT 1')->fetch(PDO::FETCH_ASSOC);
  if ($row) {
    $fromPos = (int)$row['posicion_id'];
    $loteId  = $row['lote_id'] !== null ? (int)$row['lote_id'] : null;
    $palletId= $row['pallet_id'] !== null ? (int)$row['pallet_id'] : null;
    $cur = $pdo->prepare('SELECT qty_uv, qty_uc FROM wh_stock WHERE deposito_id=? AND posicion_id=? AND producto_id=? AND (lote_id <=> ?) AND (pallet_id <=> ?) LIMIT 1');
    $cur->execute([$depId, $fromPos, $prodId, $loteId, $palletId]);
    $q = $cur->fetch(PDO::FETCH_ASSOC) ?: ['qty_uv'=>0,'qty_uc'=>0];
    $takeUv = min(max(0,(int)$q['qty_uv']), $needUv);
    $takeUc = min(max(0,(int)$q['qty_uc']), $needUc);
    if ($takeUv>0 || $takeUc>0) {
      $pdo->prepare('INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)')
          ->execute([$depId,'MOVE','PREPARACION',$palletId,$prodId,$loteId,$fromPos,$posPrepId,$takeUv,$takeUc,'PRE-'.$soId]);
      $pdo->prepare('INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por) VALUES (?,?,?,?,?,?,?,?,NULL)')
          ->execute([$preId,$itemId,$fromPos,$posPrepId,$palletId,$loteId,$takeUv,$takeUc]);
      $needUv -= $takeUv; $needUc -= $takeUc; $picked = ['uv'=>$takeUv,'uc'=>$takeUc];
    }
  }

  // Reposición 1 iteración si falta
  $repo = ['uv'=>0,'uc'=>0];
  if ($needUv>0 || $needUc>0) {
    $pdo->exec('DROP TEMPORARY TABLE IF EXISTS _src_rows');
    $pdo->exec('CREATE TEMPORARY TABLE _src_rows (
      from_pos_id BIGINT UNSIGNED,
      pallet_id   BIGINT UNSIGNED,
      lote_id     BIGINT UNSIGNED,
      qty_uv      INT,
      qty_uc      INT,
      venc        DATE
    ) ENGINE=InnoDB');
    $st = $pdo->prepare("INSERT INTO _src_rows (from_pos_id, pallet_id, lote_id, qty_uv, qty_uc, venc)
      SELECT s.posicion_id, s.pallet_id, s.lote_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
        FROM wh_stock s
        JOIN wh_posicion p ON p.id=s.posicion_id
        JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING')
        LEFT JOIN wh_lote l ON l.id=s.lote_id
       WHERE s.deposito_id=? AND s.producto_id=?
         AND (?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?))
         AND (s.qty_uv>0 OR s.qty_uc>0)
       ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC");
    $st->execute([$depId,$prodId,$loteCode,$loteCode]);
    $src = $pdo->query('SELECT from_pos_id, pallet_id, lote_id, qty_uv, qty_uc FROM _src_rows LIMIT 1')->fetch(PDO::FETCH_ASSOC);
    if ($src) {
      $from = (int)$src['from_pos_id'];
      $pl   = $src['pallet_id']!==null?(int)$src['pallet_id']:null;
      $lt   = $src['lote_id']!==null?(int)$src['lote_id']:null;
      $repoUv = min(max(0,(int)$src['qty_uv']), $needUv);
      $repoUc = min(max(0,(int)$src['qty_uc']), $needUc);
      if ($repoUv>0 || $repoUc>0) {
        $pdo->prepare('INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)')
            ->execute([$depId,'MOVE','REPOSICION_PICKING',$pl,$prodId,$lt,$from,$posPickId,$repoUv,$repoUc,'PREP-REP-'.$soId]);
        $needUv -= $repoUv; $needUc -= $repoUc; $repo = ['uv'=>$repoUv,'uc'=>$repoUc];
      }
    }
  }

  // Reintento pick 1 iteración si aún falta
  $picked2 = ['uv'=>0,'uc'=>0];
  if ($needUv>0 || $needUc>0) {
    $pdo->exec('DROP TEMPORARY TABLE IF EXISTS _cand_pick2');
    $pdo->exec('CREATE TEMPORARY TABLE _cand_pick2 LIKE _cand_pick');
    $st = $pdo->prepare("INSERT INTO _cand_pick2 (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
      SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
        FROM wh_stock s
        JOIN wh_posicion p ON p.id=s.posicion_id
        JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
        LEFT JOIN wh_lote l ON l.id=s.lote_id
       WHERE s.deposito_id=? AND s.producto_id=?
         AND (?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?))
         AND (s.qty_uv>0 OR s.qty_uc>0)
       ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC");
    $st->execute([$depId,$prodId,$loteCode,$loteCode]);
    $r2 = $pdo->query('SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc FROM _cand_pick2 ORDER BY COALESCE(venc,\'9999-12-31\') ASC LIMIT 1')->fetch(PDO::FETCH_ASSOC);
    if ($r2) {
      $fromPos = (int)$r2['posicion_id'];
      $loteId  = $r2['lote_id']!==null?(int)$r2['lote_id']:null;
      $palletId= $r2['pallet_id']!==null?(int)$r2['pallet_id']:null;
      $cur = $pdo->prepare('SELECT qty_uv, qty_uc FROM wh_stock WHERE deposito_id=? AND posicion_id=? AND producto_id=? AND (lote_id <=> ?) AND (pallet_id <=> ?) LIMIT 1');
      $cur->execute([$depId, $fromPos, $prodId, $loteId, $palletId]);
      $q = $cur->fetch(PDO::FETCH_ASSOC) ?: ['qty_uv'=>0,'qty_uc'=>0];
      $takeUv = min(max(0,(int)$q['qty_uv']), $needUv);
      $takeUc = min(max(0,(int)$q['qty_uc']), $needUc);
      if ($takeUv>0 || $takeUc>0) {
        $pdo->prepare('INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)')
            ->execute([$depId,'MOVE','PREPARACION',$palletId,$prodId,$loteId,$fromPos,$posPrepId,$takeUv,$takeUc,'PRE-'.$soId]);
        $pdo->prepare('INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por) VALUES (?,?,?,?,?,?,?,?,NULL)')
            ->execute([$preId,$itemId,$fromPos,$posPrepId,$palletId,$loteId,$takeUv,$takeUc]);
        $needUv -= $takeUv; $needUc -= $takeUc; $picked2 = ['uv'=>$takeUv,'uc'=>$takeUc];
      }
    }
  }

  $pdo->commit();

  out_json([
    'ok' => true,
    'message' => 'Paso manual ejecutado (1 ítem)',
    'so_codigo' => $soCodigo,
    'pre' => ['id'=>$preId],
    'posiciones' => ['prep'=>$posPrepId, 'picking'=>$posPickId],
    'item' => ['id'=>$itemId, 'producto_id'=>$prodId, 'lote_code'=>$loteCode],
    'acciones' => ['pick1'=>$picked, 'repo'=>$repo, 'pick2'=>$picked2],
    'pendiente_restante' => ['uv'=>$needUv, 'uc'=>$needUc],
    'doc_url' => url('/salidas/preparacion/doc') . '?pre_id=' . $preId
  ]);
} catch (Throwable $e) {
  try { if ($pdo && $pdo->inTransaction()) $pdo->rollBack(); } catch (Throwable $e2) {}
  out_json(['ok'=>false,'error'=>'Error en preparación manual','message'=>$e->getMessage()], 500);
}
