<?php
require_once dirname(__DIR__, 2) . '/vendor/autoload.php';
require_once dirname(__DIR__, 2) . '/config/db.php';

// Verificar parámetros
if (!isset($_GET['pre_id']) || !is_numeric($_GET['pre_id'])) {
    http_response_code(400);
    echo "ID de preembarque requerido";
    exit;
}

$pre_id = (int)$_GET['pre_id'];

try {
    // Obtener información del preembarque
    $stmt = $pdo->prepare("
        SELECT sp.*, so.numero as so_numero, so.cliente_id, so.observaciones
        FROM so_preembarque sp
        JOIN so_salida so ON so.id = sp.so_id
        WHERE sp.id = ?
    ");
    $stmt->execute([$pre_id]);
    $preembarque = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$preembarque) {
        http_response_code(404);
        echo "Preembarque no encontrado";
        exit;
    }

    $fecha_preparacion = $preembarque['fin_at'] ? date('d/m/Y H:i', strtotime($preembarque['fin_at'])) : 'En proceso';

    // Obtener movimientos de preparación
    $stmt = $pdo->prepare("
        SELECT 
            m.*,
            p.sku,
            p.denominacion as producto_nombre,
            wp_from.codigo as posicion_origen,
            wp_to.codigo as posicion_destino
        FROM wh_move m
        LEFT JOIN para_productos p ON p.id = m.producto_id
        LEFT JOIN wh_posicion wp_from ON wp_from.id = m.from_posicion_id
        LEFT JOIN wh_posicion wp_to ON wp_to.id = m.to_posicion_id
        WHERE m.referencia = ?
        ORDER BY m.created_at, m.id
    ");
    $stmt->execute(['PRE-' . $preembarque['so_id']]);
    $movimientos = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Configurar mPDF
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'orientation' => 'P',
        'margin_left' => 15,
        'margin_right' => 15,
        'margin_top' => 20,
        'margin_bottom' => 20
    ]);

    // CSS
    $css = '
    <style>
        body { font-family: Arial, sans-serif; font-size: 12px; }
        .header { text-align: center; margin-bottom: 20px; }
        .header h1 { color: #2c3e50; margin: 0; }
        .header h2 { color: #34495e; margin: 5px 0; }
        .info-box { 
            background: #f8f9fa; 
            padding: 15px; 
            border-radius: 5px; 
            margin-bottom: 20px;
            border-left: 4px solid #007bff;
        }
        .info-row { margin-bottom: 8px; }
        .info-label { font-weight: bold; color: #495057; }
        table { 
            width: 100%; 
            border-collapse: collapse; 
            margin-top: 15px;
        }
        th { 
            background: #007bff; 
            color: white; 
            padding: 12px 8px; 
            text-align: left;
            font-weight: bold;
        }
        td { 
            padding: 10px 8px; 
            border-bottom: 1px solid #dee2e6;
        }
        tr:nth-child(even) { background-color: #f8f9fa; }
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .footer { 
            margin-top: 30px; 
            padding-top: 15px; 
            border-top: 1px solid #dee2e6;
            font-size: 10px;
            color: #6c757d;
        }
        .status-badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 10px;
            font-weight: bold;
        }
        .status-completed { background: #d4edda; color: #155724; }
        .status-pending { background: #fff3cd; color: #856404; }
    </style>';

    // HTML del documento
    $html = $css . '
    <div class="header">
        <h1>Movimientos de Preparación</h1>
        <h2>Preembarque #' . htmlspecialchars($pre_id) . '</h2>
    </div>

    <div class="info-box">
        <div class="info-row">
            <span class="info-label">Orden de Salida:</span> ' . htmlspecialchars($preembarque['so_numero']) . '
        </div>
        <div class="info-row">
            <span class="info-label">Fecha de Preparación:</span> ' . htmlspecialchars($fecha_preparacion) . '
        </div>
        <div class="info-row">
            <span class="info-label">Estado:</span> 
            <span class="status-badge ' . ($preembarque['fin_at'] ? 'status-completed' : 'status-pending') . '">
                ' . ($preembarque['fin_at'] ? 'Completado' : 'En Proceso') . '
            </span>
        </div>
        ' . ($preembarque['observaciones'] ? '
        <div class="info-row">
            <span class="info-label">Observaciones:</span> ' . htmlspecialchars($preembarque['observaciones']) . '
        </div>' : '') . '
    </div>';

    if (empty($movimientos)) {
        $html .= '<p style="text-align: center; font-style: italic; color: #6c757d;">No se encontraron movimientos para este preembarque.</p>';
    } else {
        $html .= '
        <table>
            <thead>
                <tr>
                    <th>Producto</th>
                    <th>SKU</th>
                    <th>Origen</th>
                    <th>Destino</th>
                    <th class="text-right">Cantidad</th>
                    <th class="text-center">Fecha/Hora</th>
                </tr>
            </thead>
            <tbody>';

        foreach ($movimientos as $mov) {
            $fecha_mov = date('d/m/Y H:i', strtotime($mov['created_at']));
            $cantidad = number_format($mov['delta'], 2);
            
            $html .= '
                <tr>
                    <td>' . htmlspecialchars($mov['producto_nombre'] ?: 'N/A') . '</td>
                    <td>' . htmlspecialchars($mov['sku'] ?: 'N/A') . '</td>
                    <td>' . htmlspecialchars($mov['posicion_origen'] ?: 'N/A') . '</td>
                    <td>' . htmlspecialchars($mov['posicion_destino'] ?: 'N/A') . '</td>
                    <td class="text-right">' . $cantidad . '</td>
                    <td class="text-center">' . $fecha_mov . '</td>
                </tr>';
        }

        $html .= '
            </tbody>
        </table>';

        // Resumen
        $total_movimientos = count($movimientos);
        $html .= '
        <div style="margin-top: 20px; padding: 15px; background: #e9ecef; border-radius: 5px;">
            <strong>Resumen:</strong><br>
            Total de movimientos: ' . $total_movimientos . '
        </div>';
    }

    $html .= '
    <div class="footer">
        <div class="text-center">
            Generado el ' . date('d/m/Y H:i:s') . ' | Sistema SOL
        </div>
    </div>';

    // Generar PDF
    $mpdf->WriteHTML($html);
    
    // Nombre del archivo
    $filename = 'movimientos_preparacion_' . $pre_id . '_' . date('Ymd_His') . '.pdf';
    
    // Enviar PDF
    $mpdf->Output($filename, 'D'); // 'D' = download

} catch (Exception $e) {
    http_response_code(500);
    echo "Error al generar PDF: " . $e->getMessage();
}
?>