<?php
declare(strict_types=1);

/**
 * BACKUP (2025-10-14): API: Preparación automática SIMPLIFICADA
 * Copia del archivo original antes de pruebas sin SP
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');
// Aumentar el tiempo máximo para este proceso pesado
@ini_set('max_execution_time', '300');
@set_time_limit(300);

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out_simple(array $p, int $c = 200): void { 
	http_response_code($c); 
	echo json_encode($p, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); 
	exit; 
}

try {
	if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
		out_simple(['ok'=>false,'error'=>'Método no permitido'], 405);
	}

	$raw = file_get_contents('php://input');
	$isJson = isset($_SERVER['CONTENT_TYPE']) && stripos((string)$_SERVER['CONTENT_TYPE'], 'application/json') !== false;
	$payload = [];
	if ($isJson && $raw) {
		$payload = json_decode($raw, true) ?: [];
	}
	$payload = array_merge($_POST, $payload);

	$soId = isset($payload['so_id']) ? (int)$payload['so_id'] : 0;
	$simulate = isset($payload['simulate']) ? (int)$payload['simulate'] : 0;
    
	if ($soId <= 0) {
		out_simple(['ok'=>false,'error'=>'so_id requerido'], 422);
	}

	$pdo = get_pdo();
	$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
	// Configuración agresiva de timeouts
	$pdo->exec('SET SESSION innodb_lock_wait_timeout = 300');
	$pdo->exec('SET SESSION lock_wait_timeout = 300');
	$pdo->exec('SET SESSION autocommit = 1');
	$pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    
	// Obtener código del pedido
	$stmt = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id = ?');
	$stmt->execute([$soId]);
	$soCodigo = $stmt->fetchColumn();
    
	if (!$soCodigo) {
		out_simple(['ok'=>false,'error'=>'Pedido no encontrado'], 404);
	}
    
	// Limpiar cualquier transacción pendiente
	try {
		$pdo->exec('ROLLBACK');
	} catch (Exception $e) {
		// Ignorar errores de rollback
	}
    
	// Ejecutar SP directamente
	$stmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
	$start = microtime(true);
    
	$stmt->execute([
		$soCodigo,
		'DEP1',        // deposito_code
		null,          // prep_posicion_code (auto-select)
		$simulate      // simulate
	]);
    
	$duration = microtime(true) - $start;
    
	// Procesar resultados
	$sets = [];
	do {
		$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
		if ($rows !== false && !empty($rows)) {
			$sets[] = $rows;
		}
	} while ($stmt->nextRowset());
    
	$stmt->closeCursor();
	$stmt = null;
    
	// Obtener datos del preembarque
	$preCode = 'PRE-' . $soCodigo;
	$stPre = $pdo->prepare('SELECT id FROM so_preembarque WHERE codigo=? LIMIT 1');
	$stPre->execute([$preCode]);
	$preId = $stPre->fetchColumn();
    
	$result = [
		'ok' => true,
		'message' => 'Plan FEFO ejecutado exitosamente',
		'duration' => round($duration, 2),
		'result_sets' => count($sets),
		'so_codigo' => $soCodigo,
		'simulate' => (bool)$simulate
	];
    
	if ($preId) {
		$result['pre'] = ['id' => $preId];
		$result['doc_url'] = url('/salidas/preparacion/doc') . '?pre_id=' . $preId;
	}
    
	if ($simulate && !empty($sets)) {
		$result['plan'] = $sets[0] ?? [];
	}
    
	out_simple($result);
    
} catch (Throwable $e) {
	out_simple([
		'ok' => false,
		'error' => 'Error en preparación automática',
		'message' => $e->getMessage(),
		'file' => $e->getFile(),
		'line' => $e->getLine()
	], 500);
}
?>
