<?php
declare(strict_types=1);

/**
 * API: Preparación automática SIMPLIFICADA
 * Ejecuta solo el stored procedure sin lógica adicional compleja
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');
// Aumentar el tiempo máximo para este proceso pesado
@ini_set('max_execution_time', '300');
@set_time_limit(300);

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function out_simple(array $p, int $c = 200): void { 
    http_response_code($c); 
    echo json_encode($p, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); 
    exit; 
}

function ensureSoPedidoEstados(PDO $pdo): void {
    static $done = false;
    if ($done) return;
    $rows = [
        ['RECIBIDO', 'Recibido', 1],
        ['EN_PREPARACION', 'En preparación', 2],
        ['PARCIAL', 'Parcialmente preparado', 3],
        ['PREPARADO', 'Preparado', 4],
        ['EN_EMBARQUE', 'En embarque', 5],
        ['DESPACHADO', 'Despachado', 6],
        ['CERRADO', 'Cerrado', 7],
        ['CANCELADO', 'Cancelado', 9],
    ];
    $stmt = $pdo->prepare("INSERT INTO so_pedido_estado (code, nombre, orden) VALUES (?,?,?) ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden)");
    foreach ($rows as $row) {
        try { $stmt->execute($row); } catch (Throwable $e) { /* ignore */ }
    }
    $done = true;
}

function ensureSoPreembarqueEstados(PDO $pdo): void {
    static $done = false;
    if ($done) return;
    $rows = [
        ['PENDIENTE', 'Pendiente', 1],
        ['ASIGNADO', 'Asignado', 2],
        ['EN_PROCESO', 'En proceso', 3],
        ['COMPLETADO', 'Completado', 4],
        ['CANCELADO', 'Cancelado', 9],
    ];
    $stmt = $pdo->prepare("INSERT INTO so_preembarque_estado (code, nombre, orden) VALUES (?,?,?) ON DUPLICATE KEY UPDATE nombre=VALUES(nombre), orden=VALUES(orden)");
    foreach ($rows as $row) {
        try { $stmt->execute($row); } catch (Throwable $e) { /* ignore */ }
    }
    $done = true;
}

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        out_simple(['ok'=>false,'error'=>'Método no permitido'], 405);
    }

    $raw = file_get_contents('php://input');
    $isJson = isset($_SERVER['CONTENT_TYPE']) && stripos((string)$_SERVER['CONTENT_TYPE'], 'application/json') !== false;
    $payload = [];
    if ($isJson && $raw) {
        $payload = json_decode($raw, true) ?: [];
    }
    $payload = array_merge($_POST, $payload);

    $soId = isset($payload['so_id']) ? (int)$payload['so_id'] : 0;
    $simulate = isset($payload['simulate']) ? (int)$payload['simulate'] : 0;
    $directToPrep = isset($payload['direct_to_prep']) ? (int)$payload['direct_to_prep'] : 0;
    $debug = isset($payload['debug']) ? (int)$payload['debug'] : 0;
    $ucToUv = isset($payload['uc_to_uv']) ? (int)$payload['uc_to_uv'] : 0;
    $uvToUc = isset($payload['uv_to_uc']) ? (int)$payload['uv_to_uc'] : 0;
    $cands = isset($payload['cands']) ? (int)$payload['cands'] : null; // límite de candidatos
    $iters = isset($payload['iters']) ? (int)$payload['iters'] : null; // límite de iteraciones por bucle
    $depositoCodeIn = isset($payload['deposito_code']) ? trim((string)$payload['deposito_code']) : '';
    $prepPosCode = isset($payload['prep_posicion_code']) ? trim((string)$payload['prep_posicion_code']) : '';
    $timeout = isset($payload['timeout']) ? max(60, (int)$payload['timeout']) : 300;
    
    if ($soId <= 0) {
        out_simple(['ok'=>false,'error'=>'so_id requerido'], 422);
    }

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    ensureSoPedidoEstados($pdo);
    ensureSoPreembarqueEstados($pdo);
    
    // Configuración agresiva de timeouts
    $pdo->exec('SET SESSION innodb_lock_wait_timeout = ' . (int)$timeout);
    $pdo->exec('SET SESSION lock_wait_timeout = ' . (int)$timeout);
    $pdo->exec('SET SESSION autocommit = 1');
    $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    // Aplicar también tiempo de ejecución PHP configurable
    @ini_set('max_execution_time', (string)$timeout);
    @set_time_limit($timeout);
    
    // Obtener código del pedido
    $stmt = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id = ?');
    $stmt->execute([$soId]);
    $soCodigo = $stmt->fetchColumn();
    
    if (!$soCodigo) {
        out_simple(['ok'=>false,'error'=>'Pedido no encontrado'], 404);
    }
    // Alinear collation de la sesión para evitar errores 1267
    try {
        $pdo->exec("SET NAMES utf8mb4 COLLATE utf8mb4_0900_ai_ci");
        $pdo->exec("SET collation_connection = 'utf8mb4_0900_ai_ci'");
    // Activar logging del SP si debug está activo
    $pdo->exec('SET @so_pre_log := ' . (int)$debug);
        $pdo->exec('SET @so_pre_direct_to_prep := ' . (int)$directToPrep);
        $pdo->exec('SET @so_pre_uc_to_uv := ' . (int)$ucToUv);
        $pdo->exec('SET @so_pre_uv_to_uc := ' . (int)$uvToUc);
        // Defaults seguros si no vienen en payload
        if ($cands === null || $cands <= 0) { $cands = 200; }
        if ($iters === null || $iters <= 0) { $iters = 1000; }
        $pdo->exec('SET @so_pre_max_cands := ' . (int)$cands);
        $pdo->exec('SET @so_pre_max_iters := ' . (int)$iters);
    } catch (Throwable $e) { /* no-op */ }

    // Resolver depósito (si no viene en payload, intentar derivar del pedido)
    // $depositoCode = $depositoCodeIn;
    // if ($depositoCode === '') {
    //     try {
    //         $stDep = $pdo->prepare('SELECT d.code as codigo FROM so_pedido sp INNER JOIN wh_deposito d ON d.id = sp.deposito_id WHERE sp.id = ? LIMIT 1');
    //         $stDep->execute([$soId]);
    //         $depCode = $stDep->fetchColumn();
    //         if (is_string($depCode) && $depCode !== '') {
    //             $depositoCode = $depCode;
    //         }
    //     } catch (Throwable $e) {
    //         // no-op, fallback más abajo
    //     }
    // }
    // if ($depositoCode === '') {
    //     $depositoCode = 'DEP1';
    // }
    $depositoCode = 'DEP1';
    // Ejecutar SP directamente
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
    $start = microtime(true);

    $stmt->execute([
        $soCodigo,
        $depositoCode, // deposito_code
        ($prepPosCode !== '' ? $prepPosCode : null), // prep_posicion_code (auto-select si null)
        $simulate      // simulate
    ]);

    $duration = microtime(true) - $start;

    // Procesar resultados
    $sets = [];
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && !empty($rows)) {
            $sets[] = $rows;
        }
    } while ($stmt->nextRowset());

    $stmt->closeCursor();
    $stmt = null;

    // Obtener datos del preembarque
    $preCode = 'PRE-' . $soCodigo;
    $stPre = $pdo->prepare('SELECT id FROM so_preembarque WHERE codigo=? LIMIT 1');
    $stPre->execute([$preCode]);
    $preId = $stPre->fetchColumn();

    $result = [
        'ok' => true,
        'message' => 'Plan FEFO ejecutado',
        'duration' => round($duration, 2),
        'result_sets' => count($sets),
        'so_codigo' => $soCodigo,
        'simulate' => (bool)$simulate,
        'deposito_code' => $depositoCode
    ];

    if ($preId) {
        $result['pre'] = ['id' => $preId];
        $result['doc_url'] = url('/salidas/preparacion/doc') . '?pre_id=' . $preId;
    }

    if ($simulate && !empty($sets)) {
        $result['plan'] = $sets[0] ?? [];
    }

    out_simple($result);
    
} catch (Throwable $e) {
    out_simple([
        'ok' => false,
        'error' => 'Error en preparación automática',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ], 500);
}
?>