<?php
// api/operaciones/so_preview.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$ROOT = dirname(__DIR__, 2); // api -> operaciones -> raíz
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function jexit(array $payload, int $httpCode = 200): void {
  http_response_code($httpCode);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

try {
  $pdo = get_pdo();

  $batchId = (int)($_GET['batch_id'] ?? 0);
  if ($batchId <= 0) {
    jexit(['ok' => false, 'error' => 'batch_id requerido'], 400);
  }

  // Resumen del batch (so_import_batch)
  $stmtB = $pdo->prepare('
    SELECT
      id,
      filename,
      rows_total,
      rows_ok,
      rows_error,
      created_at
    FROM so_import_batch
    WHERE id = :id
    LIMIT 1
  ');
  $stmtB->execute([':id' => $batchId]);
  $batch = $stmtB->fetch(PDO::FETCH_ASSOC);
  if (!$batch) {
    jexit(['ok' => false, 'error' => 'Batch no encontrado'], 404);
  }

  // Previsualización desde so_import_row (JSON)
  $stmt = $pdo->prepare('
    SELECT
      sir.id,
      sir.rownum AS fila_nro,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.cod"))                 AS cod,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.razon_social"))        AS razon_social,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.nombre_destinatario")) AS nombre_destinatario,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.direccion"))           AS direccion,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.telefono"))            AS telefono,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.sku_cliente"))         AS sku,
      CAST(JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.uv_cajas")) AS UNSIGNED)    AS uv_cajas,
      CAST(JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.uc_sueltas")) AS UNSIGNED)  AS uc_sueltas,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.numero_doc"))          AS numero_doc,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.tipo_factura"))        AS tipo_factura,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.fecha_salida"))        AS fecha_salida,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.pre_embarque"))         AS pre_embarque,
      JSON_UNQUOTE(JSON_EXTRACT(sir.raw, "$.cliente_id"))          AS cliente_id
    FROM so_import_row sir
    WHERE sir.batch_id = :bid
    ORDER BY sir.rownum ASC, sir.id ASC
  ');
  $stmt->execute([':bid' => $batchId]);
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  // Normalización básica + detección de errores por fila
  $computedErrors = 0;
  foreach ($rows as &$r) {
    $r['fecha_salida'] = ($r['fecha_salida'] ?? '') !== '' ? $r['fecha_salida'] : null;
    // Asegurar números enteros
    $r['uv_cajas'] = isset($r['uv_cajas']) ? (int)$r['uv_cajas'] : 0;
    $r['uc_sueltas'] = isset($r['uc_sueltas']) ? (int)$r['uc_sueltas'] : 0;
    $sku = trim((string)($r['sku'] ?? ''));
    $err = '';
    if ($sku === '') {
      $err = 'SKU vacío';
    } elseif ((int)$r['uv_cajas'] + (int)$r['uc_sueltas'] <= 0) {
      $err = 'Cantidades vacías';
    }
    $r['error'] = $err;
    if ($err !== '') $computedErrors++;
  }
  unset($r);

  // Definición de columnas para DataTables
  $columns = [
    ['data' => 'fila_nro',           'title' => '#'],
    ['data' => 'cod',                'title' => 'Cod'],
    ['data' => 'razon_social',       'title' => 'Razón social'],
    ['data' => 'nombre_destinatario','title' => 'Destinatario'],
    ['data' => 'direccion',          'title' => 'Dirección'],
    ['data' => 'telefono',           'title' => 'Teléfono'],
    ['data' => 'sku',                'title' => 'SKU'],
    ['data' => 'uv_cajas',           'title' => 'Cajas',      'type' => 'number'],
    ['data' => 'uc_sueltas',         'title' => 'UC sueltas', 'type' => 'number'],
    ['data' => 'numero_doc',         'title' => 'Nro doc.'],
    ['data' => 'tipo_factura',       'title' => 'Tipo factura'],
    ['data' => 'fecha_salida',       'title' => 'Fecha salida'],
    ['data' => 'pre_embarque',       'title' => 'Pre_embarque'],
    ['data' => 'error',              'title' => 'Error'],
  ];

  jexit([
    'ok'          => true,
    'batch_id'    => $batchId,
    'filename'    => $batch['filename'],
    'rows_total'  => $batch['rows_total'],
    'rows_ok'     => $batch['rows_ok'],
    'rows_error'  => $batch['rows_error'] ?? $computedErrors,
    'created_at'  => $batch['created_at'],
    'columns'     => $columns,
    'data'        => $rows,
  ]);
} catch (Throwable $e) {
  jexit(['ok' => false, 'error' => 'Error listando preview'], 500);
}
