<?php
declare(strict_types=1);

// SSE: Streamea estado de ejecución del Plan FEFO (MySQL processlist + contadores básicos)
// Uso: GET /api/operaciones/so_progress_sse.php?so_id=30

header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('Connection: keep-alive');
header('X-Accel-Buffering: no'); // Nginx
header('Content-Encoding: none'); // Evitar gzip en proxies/Apache
@ini_set('output_buffering', 'off');
@ini_set('zlib.output_compression', '0');
@ini_set('implicit_flush', '1');
@set_time_limit(0);

// Desactivar compresión en Apache si aplica
if (function_exists('apache_setenv')) {
    @apache_setenv('no-gzip', '1');
}

// Cerrar sesión para no bloquear otros requests del mismo usuario
if (session_status() === PHP_SESSION_ACTIVE) {
    @session_write_close();
}

// Vaciar buffers existentes
while (function_exists('ob_get_level') && ob_get_level() > 0) {
    @ob_end_flush();
}
@ob_implicit_flush(true);

// Preambulo para romper buffers de proxies/servidor (>=2KB)
echo ": " . str_repeat(" ", 2048) . "\n\n";
// Sugerir reintento del EventSource
echo "retry: 2000\n\n";
@flush();

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/db.php';

function sse_send(string $event, array|string $data): void {
    if (is_array($data)) {
        $data = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }
    echo "event: {$event}\n";
    foreach (explode("\n", (string)$data) as $line) {
        echo 'data: ' . $line . "\n";
    }
    echo "\n";
    @ob_flush();
    @flush();
}

try {
    // Modo ping: prueba de streaming sin tocar DB
    if (isset($_GET['ping'])) {
        $start = time();
        $maxSeconds = isset($_GET['max']) ? (int)$_GET['max'] : 15;
        sse_send('start', ['mode' => 'ping', 'ts' => date('c')]);
        $i = 0;
        while ((time() - $start) <= $maxSeconds) {
            if (connection_aborted()) break;
            sse_send('log', ['t' => date('H:i:s'), 'state' => 'ping', 'sql' => '...']);
            sse_send('stats', ['log_rows' => $i, 'preembarques' => 0]);
            $i++;
            usleep(500_000);
        }
        sse_send('end', ['reason' => 'done', 'seconds' => (time() - $start)]);
        exit;
    }
    $soId = isset($_GET['so_id']) ? (int)$_GET['so_id'] : 0;
    if ($soId <= 0) {
        sse_send('error', ['message' => 'so_id requerido']);
        exit;
    }

    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Resolver código de pedido
    $st = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id=?');
    $st->execute([$soId]);
    $soCodigo = (string)($st->fetchColumn() ?: '');
    if ($soCodigo === '') {
        sse_send('error', ['message' => 'Pedido no encontrado']);
        exit;
    }

    $start = time();
    $maxSeconds = isset($_GET['max']) ? (int)$_GET['max'] : 180; // 3 minutos
    $lastInfo = '';
    $lastCounts = ['logs' => -1, 'pre' => -1];
    $lastProcHash = '';

    sse_send('start', ['so_id' => $soId, 'so_codigo' => $soCodigo, 'ts' => date('c')]);
    try {
        $who = $pdo->query('SELECT CURRENT_USER() cu, USER() u, DATABASE() db')->fetch(PDO::FETCH_ASSOC) ?: [];
        sse_send('env', ['current_user' => $who['cu'] ?? null, 'user' => $who['u'] ?? null, 'db' => $who['db'] ?? null]);
    } catch (Throwable $e) { /* ignore */ }

    while (true) {
        if (connection_aborted()) {
            break;
        }
        // 1) Proceso MySQL activo (intento de detectar el SP o sus queries internas)
        $procInfo = null;
        try {
            // Preparar DB actual una vez
            static $currDb = null;
            if ($currDb === null) {
                try { $currDb = (string)$pdo->query('select database()')->fetchColumn(); } catch (Throwable $e) { $currDb = ''; }
            }

            $rs = $pdo->query('SHOW PROCESSLIST');
            $procSample = [];
            foreach ($rs as $row) {
                $sameDb = (($row['db'] ?? '') !== '' && strcasecmp((string)$row['db'], $currDb) === 0);
                $cmd = (string)($row['Command'] ?? '');
                $info = (string)($row['Info'] ?? '');
                $state = (string)($row['State'] ?? '');
                $isTarget = ($cmd === 'Execute') || ($cmd === 'Query' && stripos($info, 'CALL sp_so_preparar_auto') !== false);

                // Capturar muestra compacta de hilos en esta DB (máx 5)
                if ($sameDb && count($procSample) < 5) {
                    $procSample[] = [
                        'id' => (int)$row['Id'],
                        'cmd' => $cmd,
                        'state' => $state,
                        'time' => (int)$row['Time'],
                        'info' => mb_substr($info, 0, 100),
                    ];
                }

                if ($sameDb && $isTarget) {
                    $procInfo = [
                        'id' => (int)$row['Id'],
                        'time' => (int)$row['Time'],
                        'state' => $state,
                        'info' => mb_substr($info !== '' ? $info : $cmd, 0, 180),
                    ];
                    break;
                }
            }

            // Emitir resumen si cambió
            $hash = md5(json_encode($procSample));
            if ($hash !== $lastProcHash) {
                sse_send('proc', ['threads' => $procSample]);
                $lastProcHash = $hash;
            }
        } catch (Throwable $e) {
            // ignorar errores de SHOW PROCESSLIST en host restringido
        }

        if ($procInfo) {
            $snippet = $procInfo['info'];
            if ($snippet !== $lastInfo) {
                sse_send('log', [
                    't' => date('H:i:s'),
                    'state' => $procInfo['state'],
                    'time' => $procInfo['time'],
                    'sql' => $snippet,
                ]);
                $lastInfo = $snippet;
            }
        }

        // 2) Contadores de progreso (pueden no avanzar si la transacción no fue commit)
        try {
            $st = $pdo->prepare('SELECT COUNT(*) FROM so_preparar_auto_log WHERE pedido_codigo=?');
            $st->execute([$soCodigo]);
            $cLogs = (int)$st->fetchColumn();

            $st = $pdo->prepare('SELECT COUNT(*) FROM so_preembarque WHERE codigo=?');
            $st->execute(['PRE-' . $soCodigo]);
            $cPre = (int)$st->fetchColumn();

            if ($cLogs !== $lastCounts['logs'] || $cPre !== $lastCounts['pre']) {
                sse_send('stats', ['log_rows' => $cLogs, 'preembarques' => $cPre]);
                $lastCounts = ['logs' => $cLogs, 'pre' => $cPre];
            }
        } catch (Throwable $e) {
            // ignorar
        }

        // Condiciones de salida
        if ((time() - $start) > $maxSeconds) {
            sse_send('end', ['reason' => 'timeout', 'seconds' => (time() - $start)]);
            break;
        }

        // Pequeño sleep para no saturar
        usleep(400_000); // 400ms
    }
} catch (Throwable $e) {
    sse_send('error', ['message' => $e->getMessage()]);
}

// Fin SSE
?>
