<?php
header('Content-Type: application/json; charset=utf-8');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/db.php';

try {
    $soId = (int)($_GET['so_id'] ?? 0);
    if ($soId <= 0) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'so_id requerido']);
        exit;
    }

    $pdo = get_pdo();
    
    // 1. Información básica del pedido
    $stmt = $pdo->prepare('
        SELECT p.*, e.code as estado_code, e.nombre as estado_nombre, c.razon_social as cliente_nombre
        FROM so_pedido p 
        LEFT JOIN so_pedido_estado e ON e.id = p.estado_id 
        LEFT JOIN para_clientes c ON c.id = p.cliente_id
        WHERE p.id = ?
    ');
    $stmt->execute([$soId]);
    $pedido = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$pedido) {
        http_response_code(404);
        echo json_encode(['ok' => false, 'error' => 'Pedido no encontrado']);
        exit;
    }
    
    // 2. Estadísticas de items
    $stmt = $pdo->prepare('
        SELECT 
            COUNT(DISTINCT i.id) as total_items,
            COUNT(DISTINCT d.id) as total_destinos,
            SUM(i.expected_uv) as total_expected_uv,
            SUM(i.expected_uc) as total_expected_uc,
            SUM(i.prepared_uv) as total_prepared_uv,
            SUM(i.prepared_uc) as total_prepared_uc,
            COUNT(DISTINCT i.producto_id) as productos_unicos
        FROM so_pedido_dest d 
        LEFT JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id 
        WHERE d.pedido_id = ?
    ');
    $stmt->execute([$soId]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // 3. Progreso de preparación
    $progreso_uv = $stats['total_expected_uv'] > 0 
        ? round(($stats['total_prepared_uv'] / $stats['total_expected_uv']) * 100, 1)
        : 0;
    $progreso_uc = $stats['total_expected_uc'] > 0 
        ? round(($stats['total_prepared_uc'] / $stats['total_expected_uc']) * 100, 1)
        : 0;
    
    // 4. Lista de productos
    $stmt = $pdo->prepare('
        SELECT 
            pr.sku, 
            pr.denominacion,
            SUM(i.expected_uv) as expected_uv,
            SUM(i.expected_uc) as expected_uc,
            SUM(i.prepared_uv) as prepared_uv,
            SUM(i.prepared_uc) as prepared_uc
        FROM so_pedido_dest d 
        LEFT JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id 
        LEFT JOIN para_productos pr ON pr.id = i.producto_id
        WHERE d.pedido_id = ?
        GROUP BY pr.id, pr.sku, pr.denominacion
        ORDER BY pr.sku
    ');
    $stmt->execute([$soId]);
    $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $response = [
        'ok' => true,
        'pedido' => [
            'id' => $pedido['id'],
            'codigo' => $pedido['codigo'],
            'fecha_pedido' => $pedido['fecha_pedido'],
            'cliente_nombre' => $pedido['cliente_nombre'],
            'estado_code' => $pedido['estado_code'],
            'estado_nombre' => $pedido['estado_nombre'],
            'created_at' => $pedido['created_at']
        ],
        'estadisticas' => [
            'total_items' => (int)$stats['total_items'],
            'total_destinos' => (int)$stats['total_destinos'],
            'productos_unicos' => (int)$stats['productos_unicos'],
            'total_expected_uv' => (int)$stats['total_expected_uv'],
            'total_expected_uc' => (int)$stats['total_expected_uc'],
            'total_prepared_uv' => (int)$stats['total_prepared_uv'],
            'total_prepared_uc' => (int)$stats['total_prepared_uc'],
            'progreso_uv' => $progreso_uv,
            'progreso_uc' => $progreso_uc
        ],
        'productos' => $productos
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false, 
        'error' => 'Error interno del servidor',
        'details' => $e->getMessage()
    ]);
}
?>