<?php
// api/operaciones/so_upload.php
declare(strict_types=1);

\header('Content-Type: application/json; charset=utf-8');

// En prod: 0; en dev puedes subir a 1 evitando E_DEPRECATED
\error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
\ini_set('display_errors', '0');

$BASE = \dirname(__DIR__, 2); // api -> operaciones -> raíz del proyecto

// Importante: NO incluir ViewHelpers.php aquí para evitar choque con syslog()
require_once $BASE . '/config/config.php'; // env()
require_once $BASE . '/config/db.php';     // get_pdo()

/* -------------------------
 * Helpers
 * ------------------------- */
function jexit(array $payload, int $httpCode = 200): void {
  \http_response_code($httpCode);
  echo \json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}
function ensure_dir(string $dir): void {
  if (!\is_dir($dir)) { @\mkdir($dir, 0775, true); }
}
function random_suffix(int $len = 6): string {
  $chars = 'abcdefghjkmnpqrstuvwxyzABCDEFGHJKMNPQRSTUVWXYZ23456789';
  $out = '';
  for ($i = 0; $i < $len; $i++) { $out .= $chars[\random_int(0, \strlen($chars) - 1)]; }
  return $out;
}
function to_date_ymd($raw): ?string {
  if ($raw === null || $raw === '') return null;
  if (\is_numeric($raw)) {
    try {
      $dt = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject((float)$raw);
      return $dt ? $dt->format('Y-m-d') : null;
    } catch (\Throwable $e) {
      return null;
    }
  }
  $ts = \strtotime((string)$raw);
  return $ts ? \date('Y-m-d', $ts) : null;
}

/* -------------------------
 * Validaciones
 * ------------------------- */
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
  jexit(['ok' => false, 'error' => 'Método no permitido'], 405);
}

$clienteId   = isset($_POST['cliente_id']) ? \trim((string)$_POST['cliente_id']) : '';
$observacion = isset($_POST['observacion']) ? \trim((string)$_POST['observacion']) : '';

if ($clienteId === '') {
  jexit(['ok' => false, 'error' => 'Falta seleccionar el cliente'], 400);
}
if (!isset($_FILES['so_file']) || !\is_array($_FILES['so_file']) || (($_FILES['so_file']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK)) {
  jexit(['ok' => false, 'error' => 'No se recibió el archivo Excel'], 400);
}

$up = $_FILES['so_file'];
$origName = $up['name'] ?? 'archivo.xlsx';
$ext = \strtolower(\pathinfo($origName, PATHINFO_EXTENSION));
if (!\in_array($ext, ['xlsx', 'xls'], true)) {
  jexit(['ok' => false, 'error' => 'Formato no permitido. Use XLSX/XLS según el modelo'], 400);
}

/* -------------------------
 * Guardado del archivo (auditoría)
 * ------------------------- */
$baseUrl = \rtrim((string)\env('BASE_URL', '/'), '/');
$publicStorageRel = '/assets/storage/uploads/so/' . \date('Y') . '/' . \date('m');
$publicStorageAbs = \rtrim($BASE . '/public' . $publicStorageRel, '/');
ensure_dir($publicStorageAbs);

$fname   = 'so_' . \date('Ymd_His') . '_' . random_suffix() . '.' . $ext;
$destAbs = $publicStorageAbs . '/' . $fname;
$destRel = $publicStorageRel . '/' . $fname;

if (!@\move_uploaded_file($up['tmp_name'], $destAbs)) {
  jexit(['ok' => false, 'error' => 'No se pudo guardar el archivo subido'], 500);
}

/* -------------------------
 * DB: so_import_batch + so_import_row(raw JSON)
 * ------------------------- */
try {
  $pdo = get_pdo();
  $pdo->beginTransaction();

  // 1) Crear batch
  $sqlInsBatch = "INSERT INTO so_import_batch
    (tipo, filename, sheet_name, rows_total, rows_ok, rows_error, created_at)
    VALUES
    ('PEDIDO_SALIDA', :filename, NULL, NULL, NULL, NULL, NOW())";
  $stb = $pdo->prepare($sqlInsBatch);
  $stb->execute([
    ':filename'   => $origName,
  ]);
  $batchId = (int)$pdo->lastInsertId();

  // 2) Leer Excel → construir RAW JSON por fila → so_import_row
  $vendor = $BASE . '/vendor/autoload.php';
  if (!\is_file($vendor)) {
    throw new \RuntimeException('No se encontró vendor/autoload.php (PhpSpreadsheet requerido).');
  }
  require_once $vendor;

  $reader = $ext === 'xlsx'
    ? new \PhpOffice\PhpSpreadsheet\Reader\Xlsx()
    : new \PhpOffice\PhpSpreadsheet\Reader\Xls();
  $reader->setReadDataOnly(true);
  $spreadsheet = $reader->load($destAbs);
  $sheet = $spreadsheet->getSheet(0);

  $highestRow = $sheet->getHighestDataRow();
  $highestCol = $sheet->getHighestDataColumn();
  $highestColIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestCol);

  // Cabeceras normalizadas (fila 1)
  $headers = [];
  for ($c = 1; $c <= $highestColIndex; $c++) {
    $letter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($c);
    $val = (string)$sheet->getCell($letter . '1')->getValue();
    $key = \mb_strtolower(\trim($val));
    $key = \preg_replace('/\s+/', '_', $key);
    $headers[$c] = $key;
  }

  // Aliases específicos para SALIDAS (según requerimiento)
  // raw debe incluir: cod, razon_social, nombre_destinatario, direccion, telefono, sku,
  // uv_cajas, uc_sueltas, numero_doc, tipo_factura, fecha_salida, pre_embarque (opcional) + cliente_id
  $mapWanted = [
    'cod'                 => ['cod','codigo','código','cliente_cod','codigo_cliente','id_cliente','cliente_id_excel'],
    'razon_social'        => ['razon_social','razón_social','cliente','cliente_nombre','rs'],
    // Ojo: typo posible "nombre_detinatario" en planilla
    'nombre_destinatario' => ['nombre_destinatario','nombre_detinatario','destinatario','nombre_receptor'],
    'direccion'           => ['direccion','dirección','domicilio','dir'],
    'telefono'            => ['telefono','teléfono','tel','phone'],
    'sku'                 => ['sku','sku_cliente','producto','producto_codigo','codigo_producto'],
    'uv_cajas'            => ['uv_cajas','cajas','bultos','cx'],
    'uc_sueltas'          => ['uc_sueltas','sueltas','unidades_sueltas','sueltos'],
    'numero_doc'          => ['numero_doc','nro_doc','numero_documento','documento','doc'],
    'tipo_factura'        => ['tipo_factura','tipo','factura_tipo','comprobante_tipo'],
    'fecha_salida'        => ['fecha_salida','fecha','fecha_despacho','salida'],
    'pre_embarque'        => ['pre_embarque','preembarque','pre','pre-embarque','Pre_embarque'],
  ];
  $findColByAliases = function(array $aliases) use ($headers): ?int {
    foreach ($headers as $colIdx => $name) {
      foreach ($aliases as $a) {
        if ($name === $a) return $colIdx;
      }
    }
    return null;
  };
  $colIndex = [];
  foreach ($mapWanted as $k => $aliases) {
    $colIndex[$k] = $findColByAliases($aliases);
  }
  $cell = function(?int $idx, int $row) use ($sheet) {
    if (empty($idx)) return null;
    $letter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($idx);
    return $sheet->getCell($letter . $row)->getCalculatedValue();
  };

  $sqlRow = "INSERT INTO so_import_row (batch_id, rownum, raw) VALUES (:batch_id, :rownum, :raw)";
  $sti = $pdo->prepare($sqlRow);

  $rowsOk = 0;
  $rowsErr = 0;
  $rownumCounter = 0;
  $firstErrors = [];

  for ($r = 2; $r <= $highestRow; $r++) {
    try {
      // Extraer campos
      $sku   = \trim((string)($cell($colIndex['sku'], $r) ?? ''));
  $cod      = \trim((string)($cell($colIndex['cod'], $r) ?? ''));
  $rsocial  = \trim((string)($cell($colIndex['razon_social'], $r) ?? ''));
  $destinat = \trim((string)($cell($colIndex['nombre_destinatario'], $r) ?? ''));
  $dir      = \trim((string)($cell($colIndex['direccion'], $r) ?? ''));
  $tel      = \trim((string)($cell($colIndex['telefono'], $r) ?? ''));
  $nrodoc   = \trim((string)($cell($colIndex['numero_doc'], $r) ?? ''));
  $tfact    = \trim((string)($cell($colIndex['tipo_factura'], $r) ?? ''));
  $fsalida  = to_date_ymd($cell($colIndex['fecha_salida'], $r));

      // Si la fila está totalmente vacía, saltar
      if ($sku === '' && $rsocial === '' && $destinat === '' && $dir === '' && $tel === '' && $nrodoc === '' && $tfact === '' && $fsalida === null && (int)($cell($colIndex['uv_cajas'], $r) ?? 0) === 0 && (int)($cell($colIndex['uc_sueltas'], $r) ?? 0) === 0) {
        continue;
      }

      // Cantidades (salidas)
      $uv_cajas    = (int)\max(0, (float)($cell($colIndex['uv_cajas'], $r) ?? 0));
      $uc_sueltas  = (int)\max(0, (float)($cell($colIndex['uc_sueltas'], $r) ?? 0));

      // Armar RAW JSON (incluir cliente_id del formulario)
      $preGrpRaw = $cell($colIndex['pre_embarque'] ?? null, $r);
      $preGrp = null;
      if ($preGrpRaw !== null && $preGrpRaw !== '') {
        $preGrp = is_numeric($preGrpRaw) ? (string)(int)$preGrpRaw : trim((string)$preGrpRaw);
      }
      $raw = [
        'cliente_id'        => ($clienteId !== '' ? (int)$clienteId : null),
        'cod'               => ($cod !== '' ? $cod : null),
        'razon_social'      => ($rsocial !== '' ? $rsocial : null),
        'nombre_destinatario'=> ($destinat !== '' ? $destinat : null),
        'direccion'         => ($dir !== '' ? $dir : null),
        'telefono'          => ($tel !== '' ? $tel : null),
        'sku_cliente'       => $sku,
        'uv_cajas'          => $uv_cajas,
        'uc_sueltas'        => $uc_sueltas,
        'numero_doc'        => ($nrodoc !== '' ? $nrodoc : null),
        'tipo_factura'      => ($tfact !== '' ? $tfact : null),
        'fecha_salida'      => $fsalida,
        'pre_embarque'      => $preGrp,
      ];

      $rownumCounter++;
      $sti->execute([
        ':batch_id' => $batchId,
        ':rownum'   => $rownumCounter,
        ':raw'      => \json_encode($raw, JSON_UNESCAPED_UNICODE),
      ]);

      $rowsOk++;
    } catch (\Throwable $eRow) {
      $rowsErr++;
      if (\count($firstErrors) < 3) {
        $firstErrors[] = "fila_excel={$r}: " . $eRow->getMessage();
      }
    }
  }

  // 3) Actualizar contadores
  $upd = $pdo->prepare("UPDATE so_import_batch
                          SET rows_total = :rtotal, rows_ok = :rok, rows_error = :rerr
                        WHERE id = :id");
  $upd->execute([
    ':rtotal'   => $rowsOk + $rowsErr,
    ':rok'      => $rowsOk,
    ':rerr'     => $rowsErr,
    ':id'       => $batchId,
  ]);

  $pdo->commit();

  jexit([
    'ok'          => true,
    'batch_id'    => $batchId,
    'rows_ok'     => $rowsOk,
    'rows_error'  => $rowsErr,
    'file_url'    => $baseUrl . $destRel,
    'file_name'   => $origName,
  ]);
} catch (\Throwable $e) {
  if (isset($pdo) && $pdo instanceof \PDO && $pdo->inTransaction()) {
    $pdo->rollBack();
  }
  jexit([
    'ok'    => false,
    'error' => 'Error procesando el archivo: ' . $e->getMessage(),
  ], 500);
}
