<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $start = microtime(true);
    
    // Configurar timeout igual que el endpoint real
    ini_set('max_execution_time', 60);
    set_time_limit(60);
    
    // Simular los mismos datos que envía el frontend
    $soId = 20;
    $soCodigo = '';
    $depCode = 'DEP1';
    $prepPosId = 3;
    $debug = 0;
    $simulate = 0;
    
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    // Resolver código del pedido
    if ($soCodigo === '' && $soId > 0) {
        $st = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id=? LIMIT 1');
        $st->execute([$soId]);
        $soCodigo = (string)$st->fetchColumn();
    }
    
    if ($soCodigo === '') {
        throw new Exception('No se pudo resolver el código del pedido');
    }

    // Resolver código de la posición PREP
    $prepPosCode = null;
    if ($prepPosId !== null) {
        $st = $pdo->prepare('SELECT code FROM wh_posicion WHERE id=? LIMIT 1');
        $st->execute([$prepPosId]);
        $prepPosCode = $st->fetchColumn() ?: null;
    }

    echo json_encode([
        'step' => 'preparation',
        'so_id' => $soId,
        'so_codigo' => $soCodigo,
        'prep_pos_code' => $prepPosCode,
        'elapsed_ms' => round((microtime(true) - $start) * 1000, 2)
    ], JSON_UNESCAPED_UNICODE);
    echo "\n";
    flush();

    // Validar SP existe
    $stepStart = microtime(true);
    $chk = $pdo->query("SHOW PROCEDURE STATUS WHERE Name='sp_so_preparar_auto'")->fetch(PDO::FETCH_ASSOC);
    if (!$chk) {
        throw new Exception('SP no existe');
    }
    
    echo json_encode([
        'step' => 'sp_validation',
        'elapsed_ms' => round((microtime(true) - $stepStart) * 1000, 2)
    ], JSON_UNESCAPED_UNICODE);
    echo "\n";
    flush();

    // Ejecutar SP con timeout monitoring
    $stepStart = microtime(true);
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?,?,?,?)');
    $stmt->execute([$soCodigo, $depCode, $prepPosCode, $simulate]);
    
    echo json_encode([
        'step' => 'sp_execution',
        'elapsed_ms' => round((microtime(true) - $stepStart) * 1000, 2)
    ], JSON_UNESCAPED_UNICODE);
    echo "\n";
    flush();

    // Recoger resultsets con protección
    $stepStart = microtime(true);
    $sets = [];
    $maxResultSets = 10;
    $resultSetCount = 0;
    
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && $rows) $sets[] = $rows;
        $resultSetCount++;
        
        if ($resultSetCount >= $maxResultSets) {
            break;
        }
    } while ($stmt->nextRowset());
    
    $stmt->closeCursor();
    $stmt = null;
    
    echo json_encode([
        'step' => 'resultset_processing',
        'result_sets' => $resultSetCount,
        'elapsed_ms' => round((microtime(true) - $stepStart) * 1000, 2)
    ], JSON_UNESCAPED_UNICODE);
    echo "\n";
    flush();

    $end = microtime(true);
    $totalDuration = round(($end - $start) * 1000, 2);
    
    echo json_encode([
        'step' => 'complete',
        'ok' => true,
        'total_duration_ms' => $totalDuration,
        'result_sets_found' => count($sets),
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Throwable $e) {
    $end = microtime(true);
    $totalDuration = round(($end - $start) * 1000, 2);
    
    http_response_code(500);
    echo json_encode([
        'step' => 'error',
        'ok' => false,
        'error' => $e->getMessage(),
        'total_duration_ms' => $totalDuration,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
}
?>