<?php

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

// En dev puedes mostrar errores, pero evita E_DEPRECATED que rompe JSON
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0'); // en prod: 0; en dev: 1 (pero siempre sin E_DEPRECATED)

/**
 * Endpoint: /api/parametros/productos.php
 * Usa DataTables Editor contra tabla para_productos.
 * Requiere: vendor/Editor/lib/DataTables.php (ya adaptado a .env)
 */

// Estás en /api/parametros → subir 2 niveles hasta la raíz /sol
$ROOT = dirname(__DIR__, 2);  // ✅ antes decía dirname(__DIR__, 1) → incorrecto
require_once $ROOT . '/vendor/Editor/lib/DataTables.php';

use
    DataTables\Editor,
    DataTables\Editor\Field,
    DataTables\Editor\Format,
    DataTables\Editor\Mjoin,
    DataTables\Editor\Options,
    DataTables\Editor\Upload,
    DataTables\Editor\Validate,
    DataTables\Editor\ValidateOptions;

$estadoText = static function (array $row): string {
  $v = 0;
  if (isset($row['para_productos']['activo'])) {
    $v = (int)$row['para_productos']['activo'];
  } elseif (isset($row['activo'])) {
    $v = (int)$row['activo'];
  }
  return $v === 1 ? 'Activo' : 'Inactivo';
};

try {
  // -----------------------------------------------------
  // Editor para para_productos
  // -----------------------------------------------------
  $editor = Editor::inst($db, 'para_productos', 'para_productos.id')
    // ---------------- Fields ----------------
    ->fields(
      Field::inst('para_productos.id')->set(false),

      // Básicos
      Field::inst('para_productos.sku')
        ->validator(Validate::notEmpty(
          ValidateOptions::inst()->message('El código (SKU) es requerido.')
        )),
      Field::inst('para_productos.denominacion')
        ->validator(Validate::notEmpty(
          ValidateOptions::inst()->message('La denominación es requerida.')
        )),

      // FKs con Options (Select2 en front)
      Field::inst('para_productos.operativa_id')
        ->options(
          Options::inst()
            ->table('sys_operativas')
            ->value('id')
            ->label('nombre')
            ->order('nombre asc')
            ->where(function ($q) {
              $q->where('deleted_at', null, '=');
            })
        )
        ->validator(Validate::dbValues()),
      Field::inst('para_productos.cliente_id')
        ->options(
          Options::inst()
            ->table('para_clientes')
            ->value('id')
            ->label('razon_social')             // ← IMPORTANTE: razon_social (no nombre)
            ->order('razon_social asc')
            ->where(function ($q) {
              $q->where('deleted_at', null, '=');
            })
        )
        ->validator(Validate::dbValues()),

      Field::inst('para_productos.clase_id')
        ->options(
          Options::inst()
            ->table('para_clases')
            ->value('id')
            ->label('nombre')
            ->order('nombre asc')
            ->where(fn($q) => $q->where('deleted_at', null, '='))
        )
        ->validator(Validate::dbValues()),
      Field::inst('para_productos.grupo_id')
        ->options(
          Options::inst()
            ->table('para_grupos')
            ->value('id')
            ->label('nombre')
            ->order('nombre asc')
            ->where(fn($q) => $q->where('deleted_at', null, '='))
        )
        ->validator(Validate::dbValues()),
      Field::inst('para_productos.metodo_rotacion_id')
        ->options(
          Options::inst()
            ->table('para_metodos_rotacion')
            ->value('id')
            ->label('nombre')
            ->order('nombre asc')
            ->where(fn($q) => $q->where('deleted_at', null, '='))
        )
        ->validator(Validate::dbValues()),

      // Atributos varios
      Field::inst('para_productos.marca'),
      Field::inst('para_productos.material'),
      Field::inst('para_productos.conservacion'),
      Field::inst('para_productos.color_web'),

      // Parámetros
      Field::inst('para_productos.stock_min_default'),
      Field::inst('para_productos.vida_util_dias'),
      Field::inst('para_productos.venc_amarillo_dias'),
      Field::inst('para_productos.venc_rojo_dias'),
      Field::inst('para_productos.estadia_amarillo_dias'),
      Field::inst('para_productos.estadia_rojo_dias'),

      // Imágenes
      Field::inst('para_productos.img_main'),
      Field::inst('para_productos.img_uc'),
      Field::inst('para_productos.img_uv'),

      // Nota y estado
      Field::inst('para_productos.nota'),
      Field::inst('para_productos.activo'),

      // Auditoría (readonly desde cliente)
      Field::inst('para_productos.created_at')->set(false),
      Field::inst('para_productos.updated_at')->set(false),
      Field::inst('para_productos.deleted_at')->set(false),
      Field::inst('para_productos.created_by')->set(false),
      Field::inst('para_productos.updated_by')->set(false),
      Field::inst('para_productos.deleted_by')->set(false),

      // ---- Campos JOIN de solo lectura (para mostrar en tabla) ----
      Field::inst('sys_operativas.nombre')->set(false),
      Field::inst('para_clientes.razon_social')->set(false),
      Field::inst('para_clases.nombre')->set(false),
      Field::inst('para_grupos.nombre')->set(false),
      Field::inst('para_metodos_rotacion.nombre')->set(false)
    )

    // ---------------- Joins (LEFT) para mostrar labels ----------------
    ->leftJoin('sys_operativas',        'sys_operativas.id',        '=', 'para_productos.operativa_id')
    ->leftJoin('para_clientes',        'para_clientes.id',        '=', 'para_productos.cliente_id')
    ->leftJoin('para_clases',          'para_clases.id',          '=', 'para_productos.clase_id')
    ->leftJoin('para_grupos',          'para_grupos.id',          '=', 'para_productos.grupo_id')
    ->leftJoin('para_metodos_rotacion', 'para_metodos_rotacion.id', '=', 'para_productos.metodo_rotacion_id')

    // ---------------- Filtro: sin borrados ----------------
    ->where('para_productos.deleted_at', null)

    // ---------------- Depuración ----------------
    ->debug(false);

    $editor->process($_POST)->json();

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'error' => 'Error del servidor',
    'message' => $e->getMessage(),
  ], JSON_UNESCAPED_UNICODE);
}
