<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']); exit;
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $hasTbl = function(PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t,$cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };
  $usePlural = $hasTbl($pdo,'wh_moves');
  $useSing   = !$usePlural && $hasTbl($pdo,'wh_move');

  // -------- Filtros --------
  $desde        = trim((string)($_GET['desde'] ?? ''));
  $hasta        = trim((string)($_GET['hasta'] ?? ''));
  $cliente_id   = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q            = trim((string)($_GET['q'] ?? ''));

  // WHERE dinámico
  $wherePlural = ["m.move_type='INBOUND'"];
  $whereSing   = ["m.tipo='IN'"];
  $params = [];

  if ($desde !== '') {
    $wherePlural[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) >= :desde";
    $whereSing[]   = "DATE(m.created_at) >= :desde";
    $params[':desde'] = $desde;
  }
  if ($hasta !== '') {
    $wherePlural[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) <= :hasta";
    $whereSing[]   = "DATE(m.created_at) <= :hasta";
    $params[':hasta'] = $hasta;
  }
  if ($cliente_id !== '') {
    $wherePlural[] = "m.cliente_id = :cliente";
    $whereSing[]   = "pal.cliente_id = :cliente";
    $params[':cliente'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $wherePlural[] = "p.operativa_id = :operativa";
    $whereSing[]   = "p.operativa_id = :operativa";
    $params[':operativa'] = $operativa_id;
  }
  if ($q !== '') {
    $wherePlural[] = "(p.sku LIKE :q1 OR p.denominacion LIKE :q2)";
    $whereSing[]   = "(p.sku LIKE :q1 OR p.denominacion LIKE :q2)";
    $params[':q1'] = "%{$q}%";
    $params[':q2'] = "%{$q}%";
  }

  $whereSqlPlural = "WHERE " . implode(' AND ', $wherePlural);
  $whereSqlSing   = "WHERE " . implode(' AND ', $whereSing);

  // Agregación por cliente + producto
  if ($usePlural) {
    $sql = "
      SELECT
        COALESCE(cli.razon_social, CONCAT('Cliente #', m.cliente_id)) AS cliente,
        p.sku,
        p.denominacion,
        SUM(COALESCE(mi.uc_unidades, 0)) AS ing_uc,
        SUM(COALESCE(mi.uv_cajas, 0))    AS ing_cajas,
        COUNT(DISTINCT m.id)             AS docs,
        MAX(DATE_FORMAT(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at), '%Y-%m-%d %H:%i')) AS ultimo_in
      FROM wh_moves m
      JOIN wh_move_items   mi ON mi.move_id = m.id
      JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
      JOIN para_productos   p ON p.id = pi.producto_id
      LEFT JOIN para_clientes cli ON cli.id = m.cliente_id
      $whereSqlPlural
      GROUP BY m.cliente_id, cli.razon_social, p.id
      ORDER BY cliente ASC, ing_uc DESC, p.sku ASC
    ";
  } elseif ($useSing) {
    $sql = "
      SELECT
        COALESCE(cli.razon_social, CONCAT('Cliente #', pal.cliente_id)) AS cliente,
        p.sku,
        p.denominacion,
        SUM(COALESCE(pi.uc_unidades, 0)) AS ing_uc,
        SUM(COALESCE(pi.uv_cajas, 0))    AS ing_cajas,
        COUNT(DISTINCT m.id)             AS docs,
        MAX(DATE_FORMAT(m.created_at, '%Y-%m-%d %H:%i')) AS ultimo_in
      FROM wh_move m
      JOIN wh_pallet       pal ON pal.id = m.pallet_id
      JOIN wh_pallet_item  pi  ON pi.pallet_id = pal.id
      JOIN para_productos  p   ON p.id = pi.producto_id
      LEFT JOIN para_clientes cli ON cli.id = pal.cliente_id
      $whereSqlSing
      GROUP BY pal.cliente_id, cli.razon_social, p.id
      ORDER BY cliente ASC, ing_uc DESC, p.sku ASC
    ";
  } else {
    throw new RuntimeException('No existe tabla de movimientos');
  }

  $st = $pdo->prepare($sql);
  foreach ($params as $k=>$v) $st->bindValue($k, $v);
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  echo json_encode(['ok' => true, 'data' => $rows], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  $debug = [
    'ok' => false,
    'error' => 'Error cargando reporte',
    'message' => $e->getMessage(),
    'file' => $e->getFile(),
    'line' => $e->getLine()
  ];
  // Solo en desarrollo, agregar más detalles
  if (isset($_GET['debug'])) {
    $debug['sql'] = $sql ?? null;
    $debug['params'] = $params ?? null;
    $debug['trace'] = $e->getTraceAsString();
  }
  echo json_encode($debug, JSON_UNESCAPED_UNICODE);
}
