<?php
declare(strict_types=1);
header('Content-Type: application/json');

require_once dirname(__DIR__, 2) . '/config/db.php';

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Obtener parámetro opcional de depósito
    $depositoId = isset($_GET['deposito_id']) && is_numeric($_GET['deposito_id']) 
        ? (int)$_GET['deposito_id'] 
        : null;

    // ============================================================================
    // 1. TOTALES GENERALES
    // ============================================================================
    $sqlTotales = "
        SELECT 
            COUNT(DISTINCT p.id) as total_posiciones,
            SUM(CASE WHEN p.ocupado = 1 THEN 1 ELSE 0 END) as posiciones_ocupadas,
            SUM(CASE WHEN p.ocupado = 0 OR p.ocupado IS NULL THEN 1 ELSE 0 END) as posiciones_libres,
            COALESCE(SUM(p.capacidad_pallets), 0) as capacidad_total_pallets,
            COUNT(DISTINCT CASE WHEN p.ocupado = 1 THEN p.id END) as posiciones_con_pallets
        FROM wh_posicion p
        WHERE (p.activo = 1 OR p.activo IS NULL)
          AND (p.deleted_at IS NULL)
    ";
    
    if ($depositoId) {
        $sqlTotales .= " AND p.deposito_id = :deposito_id";
    }
    
    $stmtTotales = $pdo->prepare($sqlTotales);
    if ($depositoId) {
        $stmtTotales->bindValue(':deposito_id', $depositoId, PDO::PARAM_INT);
    }
    $stmtTotales->execute();
    $totales = $stmtTotales->fetch(PDO::FETCH_ASSOC);

    // Calcular pallets actuales
    $sqlPallets = "
        SELECT COUNT(*) as total_pallets
        FROM wh_pallet pal
        WHERE pal.posicion_id IS NOT NULL 
          AND pal.posicion_id > 0
          AND (pal.deleted_at IS NULL)
    ";
    if ($depositoId) {
        $sqlPallets .= " AND pal.deposito_id = :deposito_id";
    }
    
    $stmtPallets = $pdo->prepare($sqlPallets);
    if ($depositoId) {
        $stmtPallets->bindValue(':deposito_id', $depositoId, PDO::PARAM_INT);
    }
    $stmtPallets->execute();
    $totales['pallets_actuales'] = (int)$stmtPallets->fetchColumn();
    
    // Calcular pallets disponibles
    $totales['pallets_disponibles'] = (int)$totales['capacidad_total_pallets'] - (int)$totales['pallets_actuales'];

    // ============================================================================
    // 2. TOTALES POR DEPÓSITO
    // ============================================================================
    $sqlDepositos = "
        SELECT 
            d.id,
            d.code,
            d.nombre,
            COUNT(DISTINCT p.id) as total_posiciones,
            SUM(CASE WHEN p.ocupado = 1 THEN 1 ELSE 0 END) as posiciones_ocupadas,
            SUM(CASE WHEN p.ocupado = 0 OR p.ocupado IS NULL THEN 1 ELSE 0 END) as posiciones_libres,
            COALESCE(SUM(p.capacidad_pallets), 0) as capacidad_total_pallets,
            (SELECT COUNT(*) FROM wh_pallet pal 
             WHERE pal.posicion_id IS NOT NULL 
               AND pal.posicion_id > 0
               AND pal.deposito_id = d.id
               AND (pal.deleted_at IS NULL)) as pallets_actuales
        FROM wh_deposito d
        LEFT JOIN wh_posicion p ON p.deposito_id = d.id 
            AND (p.activo = 1 OR p.activo IS NULL)
            AND (p.deleted_at IS NULL)
        WHERE (d.activo = 1 OR d.activo IS NULL)
          AND (d.deleted_at IS NULL)
    ";
    
    if ($depositoId) {
        $sqlDepositos .= " AND d.id = :deposito_id";
    }
    
    $sqlDepositos .= " GROUP BY d.id, d.code, d.nombre ORDER BY d.code";
    
    $stmtDepositos = $pdo->prepare($sqlDepositos);
    if ($depositoId) {
        $stmtDepositos->bindValue(':deposito_id', $depositoId, PDO::PARAM_INT);
    }
    $stmtDepositos->execute();
    $depositos = $stmtDepositos->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcular pallets disponibles por depósito
    foreach ($depositos as &$dep) {
        $dep['pallets_disponibles'] = (int)$dep['capacidad_total_pallets'] - (int)$dep['pallets_actuales'];
        $dep['porcentaje_ocupacion'] = $dep['total_posiciones'] > 0 
            ? round(($dep['posiciones_ocupadas'] / $dep['total_posiciones']) * 100, 1) 
            : 0;
    }
    unset($dep);

    // ============================================================================
    // 3. TOTALES POR AMBIENTE
    // ============================================================================
    $sqlAmbientes = "
        SELECT 
            a.id,
            a.code,
            a.nombre,
            COUNT(DISTINCT p.id) as total_posiciones,
            SUM(CASE WHEN p.ocupado = 1 THEN 1 ELSE 0 END) as posiciones_ocupadas,
            SUM(CASE WHEN p.ocupado = 0 OR p.ocupado IS NULL THEN 1 ELSE 0 END) as posiciones_libres,
            COALESCE(SUM(p.capacidad_pallets), 0) as capacidad_total_pallets,
            (SELECT COUNT(*) FROM wh_pallet pal 
             JOIN wh_posicion pos ON pos.id = pal.posicion_id
             WHERE pos.ambiente_id = a.id
               AND pal.posicion_id IS NOT NULL 
               AND pal.posicion_id > 0
               AND (pal.deleted_at IS NULL)
               " . ($depositoId ? "AND pal.deposito_id = :deposito_id" : "") . ") as pallets_actuales
        FROM wh_ambiente a
        LEFT JOIN wh_posicion p ON p.ambiente_id = a.id 
            AND (p.activo = 1 OR p.activo IS NULL)
            AND (p.deleted_at IS NULL)
            " . ($depositoId ? "AND p.deposito_id = :deposito_id" : "") . "
        WHERE (a.activo = 1 OR a.activo IS NULL)
          AND (a.deleted_at IS NULL)
        GROUP BY a.id, a.code, a.nombre 
        ORDER BY a.code
    ";
    
    $stmtAmbientes = $pdo->prepare($sqlAmbientes);
    if ($depositoId) {
        $stmtAmbientes->bindValue(':deposito_id', $depositoId, PDO::PARAM_INT);
    }
    $stmtAmbientes->execute();
    $ambientes = $stmtAmbientes->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcular pallets disponibles por ambiente
    foreach ($ambientes as &$amb) {
        $amb['pallets_disponibles'] = (int)$amb['capacidad_total_pallets'] - (int)$amb['pallets_actuales'];
        $amb['porcentaje_ocupacion'] = $amb['total_posiciones'] > 0 
            ? round(($amb['posiciones_ocupadas'] / $amb['total_posiciones']) * 100, 1) 
            : 0;
    }
    unset($amb);

    // ============================================================================
    // 4. M2 ÚTILES (excluir CUARENTENA y PREPARACIÓN)
    // Según indicación: ambiente_id != 1 (CUARENTENA) y ambiente_id != 6 (si existe PREP)
    // ============================================================================
    
    // Primero identificar IDs de CUARENTENA y PREP
    $stmtCua = $pdo->query("SELECT id FROM wh_ambiente WHERE code = 'CUARENTENA' LIMIT 1");
    $cuaId = $stmtCua ? (int)$stmtCua->fetchColumn() : 0;
    
    $stmtPrep = $pdo->query("SELECT id FROM wh_ambiente WHERE code IN ('PREP', 'PREPARACION') LIMIT 1");
    $prepId = $stmtPrep ? (int)$stmtPrep->fetchColumn() : 0;
    
    // Calcular M2 útiles
    // Asumiendo que cada posición ocupa aproximadamente 1.2m x 1.0m = 1.2 m² (pallet estándar)
    // Puedes ajustar este valor según tus necesidades
    $m2PorPosicion = 1.2;
    
    $sqlM2 = "
        SELECT COUNT(*) as posiciones_utiles
        FROM wh_posicion p
        WHERE (p.activo = 1 OR p.activo IS NULL)
          AND (p.deleted_at IS NULL)
          AND p.ambiente_id NOT IN (" . ($cuaId ?: 0) . "," . ($prepId ?: 0) . ")
    ";
    
    if ($depositoId) {
        $sqlM2 .= " AND p.deposito_id = :deposito_id";
    }
    
    $stmtM2 = $pdo->prepare($sqlM2);
    if ($depositoId) {
        $stmtM2->bindValue(':deposito_id', $depositoId, PDO::PARAM_INT);
    }
    $stmtM2->execute();
    $posicionesUtiles = (int)$stmtM2->fetchColumn();
    $m2Utiles = round($posicionesUtiles * $m2PorPosicion, 2);

    // ============================================================================
    // 5. DISTRIBUCIÓN POR DEPÓSITO Y AMBIENTE (matriz)
    // ============================================================================
    $sqlMatriz = "
        SELECT 
            d.id as deposito_id,
            d.code as deposito_code,
            d.nombre as deposito_nombre,
            a.id as ambiente_id,
            a.code as ambiente_code,
            a.nombre as ambiente_nombre,
            COUNT(DISTINCT p.id) as total_posiciones,
            SUM(CASE WHEN p.ocupado = 1 THEN 1 ELSE 0 END) as posiciones_ocupadas,
            COALESCE(SUM(p.capacidad_pallets), 0) as capacidad_pallets
        FROM wh_deposito d
        CROSS JOIN wh_ambiente a
        LEFT JOIN wh_posicion p ON p.deposito_id = d.id 
            AND p.ambiente_id = a.id
            AND (p.activo = 1 OR p.activo IS NULL)
            AND (p.deleted_at IS NULL)
        WHERE (d.activo = 1 OR d.activo IS NULL)
          AND (d.deleted_at IS NULL)
          AND (a.activo = 1 OR a.activo IS NULL)
          AND (a.deleted_at IS NULL)
    ";
    
    if ($depositoId) {
        $sqlMatriz .= " AND d.id = :deposito_id";
    }
    
    $sqlMatriz .= " GROUP BY d.id, d.code, d.nombre, a.id, a.code, a.nombre 
                     ORDER BY d.code, a.code";
    
    $stmtMatriz = $pdo->prepare($sqlMatriz);
    if ($depositoId) {
        $stmtMatriz->bindValue(':deposito_id', $depositoId, PDO::PARAM_INT);
    }
    $stmtMatriz->execute();
    $matriz = $stmtMatriz->fetchAll(PDO::FETCH_ASSOC);

    // ============================================================================
    // RESPUESTA
    // ============================================================================
    echo json_encode([
        'ok' => true,
        'totales' => $totales,
        'depositos' => $depositos,
        'ambientes' => $ambientes,
        'matriz' => $matriz,
        'm2_utiles' => $m2Utiles,
        'posiciones_utiles' => $posicionesUtiles,
        'm2_por_posicion' => $m2PorPosicion,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error al generar reporte',
        'detail' => $e->getMessage()
    ]);
}
