<?php
/**
 * SOL · API · Reporte de Cumplimiento de Entregas
 * Ruta: api/reportes/embarque_cumplimiento.php
 *
 * Analiza embarques para exponer métricas de fulfillment, retrasos y desempeño por destinatario.
 */
declare(strict_types=1);


header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'No autorizado'], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $filters = [
        'fecha_desde'      => trim((string) ($_GET['fecha_desde'] ?? '')),
        'fecha_hasta'      => trim((string) ($_GET['fecha_hasta'] ?? '')),
        'cliente_id'       => trim((string) ($_GET['cliente_id'] ?? '')),
        'destinatario_id'  => trim((string) ($_GET['destinatario_id'] ?? '')),
        'estado'           => trim((string) ($_GET['estado'] ?? '')),
        'movil_id'         => trim((string) ($_GET['movil_id'] ?? '')),
    ];

    if ($filters['fecha_desde'] === '') {
        $filters['fecha_desde'] = date('Y-m-d', strtotime('-30 days'));
    }

    if ($filters['fecha_hasta'] === '') {
        $filters['fecha_hasta'] = date('Y-m-d');
    }

    if ($filters['fecha_desde'] > $filters['fecha_hasta']) {
        [$filters['fecha_desde'], $filters['fecha_hasta']] = [$filters['fecha_hasta'], $filters['fecha_desde']];
    }

    [$embarques, $filterOptions] = loadEmbarques($pdo, $filters);

    if (!$embarques) {
        echo json_encode([
            'ok'                => true,
            'data'              => [],
            'metrics'           => buildEmptyMetrics(),
            'period_rates'      => [],
            'destinatarios_top' => [],
            'filters'           => $filters,
            'filter_options'    => $filterOptions,
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $ids = array_map('intval', array_column($embarques, 'id'));
    $stopStats = fetchStopStats($pdo, $ids);
    $pedidoStats = fetchPedidoStats($pdo, $ids);
    $destinatariosTop = formatDestinatarioStats(fetchDestinatarioStats($pdo, $ids));

    $rows = [];
    $series = [];
    $porEstado = [];

    $totalEmbarques = count($embarques);
    $completos = 0;
    $parciales = 0;
    $fallidos = 0;
    $ontime = 0;
    $late = 0;
    $delaySum = 0.0;
    $delayCount = 0;
    $delayMax = 0;

    $itemsEsperadosUVTotal = 0;
    $itemsDespachadosUVTotal = 0;
    $itemsEsperadosUCTotal = 0;
    $itemsDespachadosUCTotal = 0;

    foreach ($embarques as $emb) {
        $id = (int) $emb['id'];
        $estadoCode = strtoupper((string) ($emb['estado_code'] ?? ''));
        $stateKey = $estadoCode !== '' ? $estadoCode : 'OTRO';
        $porEstado[$stateKey] = ($porEstado[$stateKey] ?? 0) + 1;

        $stop = $stopStats[$id] ?? [
            'total_destinos'      => 0,
            'destinos_entregados' => 0,
            'entregas_parciales'  => 0,
            'destinos_pendientes' => 0,
            'max_delay_days'      => null,
        ];

        $pedido = $pedidoStats[$id] ?? [
            'pedidos_esperados'      => 0,
            'pedidos_despachados'    => 0,
            'items_esperados_uv'     => 0,
            'items_despachados_uv'   => 0,
            'items_esperados_uc'     => 0,
            'items_despachados_uc'   => 0,
        ];

        $totalDestinos      = (int) $stop['total_destinos'];
        $destinosEntregados = (int) $stop['destinos_entregados'];
        $entregasParciales  = (int) $stop['entregas_parciales'];
        $destinosPendientes = (int) $stop['destinos_pendientes'];

        $pedidosEsperados   = (int) $pedido['pedidos_esperados'];
        $pedidosDespachados = (int) $pedido['pedidos_despachados'];
        $itemsEsperadosUV   = (int) $pedido['items_esperados_uv'];
        $itemsDespachadosUV = (int) $pedido['items_despachados_uv'];
        $itemsEsperadosUC   = (int) $pedido['items_esperados_uc'];
        $itemsDespachadosUC = (int) $pedido['items_despachados_uc'];

        $itemsEsperadosUVTotal   += $itemsEsperadosUV;
        $itemsDespachadosUVTotal += $itemsDespachadosUV;
        $itemsEsperadosUCTotal   += $itemsEsperadosUC;
        $itemsDespachadosUCTotal += $itemsDespachadosUC;

        $fulfilled = min($pedidosDespachados, $destinosEntregados);
        $fulfillmentRate = ($pedidosEsperados > 0)
            ? ($fulfilled / $pedidosEsperados) * 100
            : 0.0;

        $delayDays = $stop['max_delay_days'];
        if ($delayDays === null && !empty($emb['salida_at']) && !empty($emb['fecha_programada'])) {
            $programada = new DateTime((string) $emb['fecha_programada']);
            $salida = new DateTime((string) $emb['salida_at']);
            $delayDays = (int) $programada->diff($salida)->format('%r%a');
        }

        if ($delayDays !== null) {
            $delayDays = max(0, (int) $delayDays);
            $delaySum += $delayDays;
            $delayCount++;
            if ($delayDays > $delayMax) {
                $delayMax = $delayDays;
            }
        }

        $deliveryStatus = 'EN_TRANSITO';
        $isClosedState = in_array($estadoCode, ['COMPLETADO', 'ENTREGADO', 'CERRADO', 'CERR'], true);

        if ($totalDestinos > 0 && $destinosEntregados >= $totalDestinos) {
            if ($delayDays !== null && $delayDays > 1) {
                $deliveryStatus = 'COMPLETADO_TARDE';
                $late++;
            } else {
                $deliveryStatus = 'COMPLETADO';
                ++$ontime;
            }
            ++$completos;
        } elseif ($destinosEntregados > 0) {
            $deliveryStatus = 'PARCIAL';
            ++$parciales;
            if ($delayDays !== null && $delayDays > 1) {
                ++$late;
            }
        } elseif ($totalDestinos > 0) {
            if ($isClosedState) {
                $deliveryStatus = 'FALLIDO';
                ++$fallidos;
                ++$late;
            } else {
                $deliveryStatus = 'PENDIENTE';
            }
        } elseif ($pedidosEsperados === 0 && $estadoCode === 'CREADO') {
            $deliveryStatus = 'PENDIENTE';
        }

        $dateKey = $emb['fecha_embarque'] ?? date('Y-m-d');
        if (!isset($series[$dateKey])) {
            $series[$dateKey] = [
                'embarques'           => 0,
                'pedidos_esperados'   => 0,
                'pedidos_despachados' => 0,
                'destinos_totales'    => 0,
                'destinos_entregados' => 0,
            ];
        }
        $series[$dateKey]['embarques']++;
        $series[$dateKey]['pedidos_esperados']   += $pedidosEsperados;
        $series[$dateKey]['pedidos_despachados'] += $pedidosDespachados;
        $series[$dateKey]['destinos_totales']    += $totalDestinos;
        $series[$dateKey]['destinos_entregados'] += $destinosEntregados;

        $rows[] = [
            'id'                   => $id,
            'codigo'               => $emb['codigo'],
            'fecha_embarque'       => $emb['fecha_embarque'],
            'estado_code'          => $estadoCode,
            'estado_nombre'        => $emb['estado_nombre'] ?? 'Sin estado',
            'movil'                => $emb['movil'],
            'total_destinos'       => $totalDestinos,
            'pedidos_esperados'    => $pedidosEsperados,
            'pedidos_despachados'  => $pedidosDespachados,
            'destinos_entregados'  => $destinosEntregados,
            'items_esperados_uv'   => $itemsEsperadosUV,
            'items_despachados_uv' => $itemsDespachadosUV,
            'items_esperados_uc'   => $itemsEsperadosUC,
            'items_despachados_uc' => $itemsDespachadosUC,
            'fulfillment_rate'     => round($fulfillmentRate, 1),
            'delay_days'           => $delayDays,
            'delivery_status'      => $deliveryStatus,
            'entregas_parciales'   => $entregasParciales,
            'destinos_pendientes'  => $destinosPendientes,
        ];
    }

    $periodRates = buildPeriodRates($series);

    $metrics = [
        'total_embarques' => $totalEmbarques,
        'por_estado'      => $porEstado,
        'fulfillment_rates' => [
            'pedidos_completos'    => $completos,
            'entregas_parciales'   => $parciales,
            'entregas_fallidas'    => $fallidos,
            'rate_cumplimiento'    => $totalEmbarques > 0 ? round(($completos / $totalEmbarques) * 100, 1) : 0.0,
            'rate_entrega_parcial' => $totalEmbarques > 0 ? round(($parciales / $totalEmbarques) * 100, 1) : 0.0,
        ],
        'tiempos' => [
            'embarques_a_tiempo'    => $ontime,
            'embarques_con_retraso' => $late,
            'promedio_delay_days'   => $delayCount > 0 ? round($delaySum / $delayCount, 1) : 0.0,
            'delay_maximo_days'     => $delayMax,
            'rate_puntualidad'      => $totalEmbarques > 0 ? round(($ontime / $totalEmbarques) * 100, 1) : 0.0,
        ],
        'volumenes' => [
            'items_esperados_uv'    => $itemsEsperadosUVTotal,
            'items_despachados_uv'  => $itemsDespachadosUVTotal,
            'items_esperados_uc'    => $itemsEsperadosUCTotal,
            'items_despachados_uc'  => $itemsDespachadosUCTotal,
            'rate_despacho_uv'      => $itemsEsperadosUVTotal > 0 ? round(($itemsDespachadosUVTotal / $itemsEsperadosUVTotal) * 100, 1) : 0.0,
            'rate_despacho_uc'      => $itemsEsperadosUCTotal > 0 ? round(($itemsDespachadosUCTotal / $itemsEsperadosUCTotal) * 100, 1) : 0.0,
        ],
    ];

    echo json_encode([
        'ok'                => true,
        'data'              => $rows,
        'metrics'           => $metrics,
        'period_rates'      => $periodRates,
        'destinatarios_top' => $destinatariosTop,
        'filters'           => $filters,
        'filter_options'    => $filterOptions,
    ], JSON_UNESCAPED_UNICODE);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'ok'    => false,
        'error' => 'Error de base de datos: ' . $e->getMessage(),
    ], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok'    => false,
        'error' => 'Error interno del servidor: ' . $e->getMessage(),
    ], JSON_UNESCAPED_UNICODE);
}

function loadEmbarques(PDO $pdo, array $filters): array
{
    $where = [
        'DATE(COALESCE(e.carga_fecha, e.creado_at)) >= ?',
        'DATE(COALESCE(e.carga_fecha, e.creado_at)) <= ?',
    ];

    $params = [
        $filters['fecha_desde'],
        $filters['fecha_hasta'],
    ];

    if ($filters['cliente_id'] !== '') {
        $where[] = 'EXISTS (
            SELECT 1
            FROM so_embarque_parada p
            JOIN para_destinatarios dest ON dest.id = p.destinatario_id
            WHERE p.embarque_id = e.id AND dest.cliente_id = ?
        )';
        $params[] = $filters['cliente_id'];
    }

    if ($filters['destinatario_id'] !== '') {
        $where[] = 'EXISTS (
            SELECT 1
            FROM so_embarque_parada p
            WHERE p.embarque_id = e.id AND p.destinatario_id = ?
        )';
        $params[] = $filters['destinatario_id'];
    }

    if ($filters['estado'] !== '') {
        $where[] = 'est.code = ?';
        $params[] = $filters['estado'];
    }

    if ($filters['movil_id'] !== '') {
        $where[] = 'e.movil_id = ?';
        $params[] = $filters['movil_id'];
    }

    $whereSql = implode(' AND ', $where);

    $sql = "
        SELECT
            e.id,
            e.codigo,
            DATE(COALESCE(e.carga_fecha, e.creado_at)) AS fecha_embarque,
            COALESCE(e.carga_fecha, e.creado_at) AS fecha_programada,
            est.code AS estado_code,
            est.nombre AS estado_nombre,
            m.chapa AS movil,
            e.salida_at
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        WHERE {$whereSql}
        ORDER BY COALESCE(e.carga_fecha, e.creado_at) DESC, e.codigo
        LIMIT 500
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);

    return [$embarques, fetchFilterOptions($pdo)];
}

function fetchStopStats(PDO $pdo, array $ids): array
{
    if (!$ids) {
        return [];
    }

    $placeholders = buildPlaceholders(count($ids));
    $sql = "
        SELECT
            p.embarque_id,
            COUNT(*) AS total_destinos,
            SUM(CASE WHEN p.hora_fin_descarga IS NOT NULL OR p.hora_salida IS NOT NULL THEN 1 ELSE 0 END) AS destinos_entregados,
            SUM(CASE WHEN p.hora_fin_descarga IS NULL AND p.hora_inicio_descarga IS NOT NULL THEN 1 ELSE 0 END) AS entregas_parciales,
            SUM(CASE WHEN p.hora_inicio_descarga IS NULL THEN 1 ELSE 0 END) AS destinos_pendientes,
            MAX(CASE
                WHEN p.hora_fin_descarga IS NOT NULL THEN DATEDIFF(DATE(p.hora_fin_descarga), DATE(COALESCE(e.carga_fecha, e.creado_at)))
                WHEN p.hora_salida IS NOT NULL THEN DATEDIFF(DATE(p.hora_salida), DATE(COALESCE(e.carga_fecha, e.creado_at)))
                ELSE NULL
            END) AS max_delay_days
        FROM so_embarque_parada p
        JOIN so_embarque e ON e.id = p.embarque_id
        WHERE p.embarque_id IN ({$placeholders})
        GROUP BY p.embarque_id
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($ids);

    $result = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $result[(int) $row['embarque_id']] = [
            'total_destinos'      => (int) $row['total_destinos'],
            'destinos_entregados' => (int) $row['destinos_entregados'],
            'entregas_parciales'  => (int) $row['entregas_parciales'],
            'destinos_pendientes' => (int) $row['destinos_pendientes'],
            'max_delay_days'      => $row['max_delay_days'] !== null ? (int) $row['max_delay_days'] : null,
        ];
    }

    return $result;
}

function fetchPedidoStats(PDO $pdo, array $ids): array
{
    if (!$ids) {
        return [];
    }

    $placeholders = buildPlaceholders(count($ids));
    $sql = "
        SELECT
            ep.embarque_id,
            COUNT(DISTINCT ep.preembarque_id) AS pedidos_esperados,
            COUNT(DISTINCT CASE WHEN sp.fin_at IS NOT NULL THEN ep.preembarque_id END) AS pedidos_despachados,
            SUM(pdi.expected_uv) AS items_esperados_uv,
            SUM(pdi.shipped_uv) AS items_despachados_uv,
            SUM(pdi.expected_uc) AS items_esperados_uc,
            SUM(pdi.shipped_uc) AS items_despachados_uc
        FROM so_embarque_pre ep
        JOIN so_preembarque sp ON sp.id = ep.preembarque_id
        JOIN so_pedido ped ON ped.id = sp.pedido_id
        JOIN so_pedido_dest pd ON pd.pedido_id = ped.id
        JOIN so_pedido_dest_item pdi ON pdi.pedido_dest_id = pd.id
        WHERE ep.embarque_id IN ({$placeholders})
        GROUP BY ep.embarque_id
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($ids);

    $result = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $result[(int) $row['embarque_id']] = [
            'pedidos_esperados'      => (int) ($row['pedidos_esperados'] ?? 0),
            'pedidos_despachados'    => (int) ($row['pedidos_despachados'] ?? 0),
            'items_esperados_uv'     => (int) ($row['items_esperados_uv'] ?? 0),
            'items_despachados_uv'   => (int) ($row['items_despachados_uv'] ?? 0),
            'items_esperados_uc'     => (int) ($row['items_esperados_uc'] ?? 0),
            'items_despachados_uc'   => (int) ($row['items_despachados_uc'] ?? 0),
        ];
    }

    return $result;
}

function fetchDestinatarioStats(PDO $pdo, array $ids): array
{
    if (!$ids) {
        return [];
    }

    $placeholders = buildPlaceholders(count($ids));
    $sql = "
        SELECT
            p.destinatario_id,
            dest.cod,
            dest.nombre,
            cli.razon_social AS cliente,
            COUNT(*) AS total_visitas,
            SUM(CASE WHEN p.hora_fin_descarga IS NOT NULL OR p.hora_salida IS NOT NULL THEN 1 ELSE 0 END) AS entregas_completas,
            SUM(CASE WHEN p.hora_fin_descarga IS NULL AND p.hora_inicio_descarga IS NOT NULL THEN 1 ELSE 0 END) AS entregas_parciales,
            SUM(CASE WHEN p.hora_inicio_descarga IS NULL THEN 1 ELSE 0 END) AS entregas_pendientes
        FROM so_embarque_parada p
        JOIN para_destinatarios dest ON dest.id = p.destinatario_id
        LEFT JOIN para_clientes cli ON cli.id = dest.cliente_id
        WHERE p.embarque_id IN ({$placeholders})
        GROUP BY p.destinatario_id, dest.cod, dest.nombre, cli.razon_social
        ORDER BY total_visitas DESC, dest.nombre ASC
        LIMIT 15
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($ids);

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function formatDestinatarioStats(array $rows): array
{
    if (!$rows) {
        return [];
    }

    return array_map(static function (array $row): array {
        $codigo = trim((string) ($row['cod'] ?? ''));
        $nombre = trim((string) ($row['nombre'] ?? ''));
        $label = $codigo !== '' ? trim($codigo . ' - ' . $nombre) : ($nombre !== '' ? $nombre : 'Sin identificación');

        return [
            'destinatario_id'    => (int) ($row['destinatario_id'] ?? 0),
            'destinatario'       => $label,
            'cliente'            => $row['cliente'] ?? '-',
            'total_visitas'      => (int) ($row['total_visitas'] ?? 0),
            'entregas_completas' => (int) ($row['entregas_completas'] ?? 0),
            'entregas_parciales' => (int) ($row['entregas_parciales'] ?? 0),
            'entregas_pendientes'=> (int) ($row['entregas_pendientes'] ?? 0),
        ];
    }, $rows);
}

function buildPeriodRates(array $series): array
{
    if (!$series) {
        return [];
    }

    $periodRates = [];
    foreach ($series as $date => $data) {
        $rateDespacho = $data['pedidos_esperados'] > 0
            ? round(($data['pedidos_despachados'] / $data['pedidos_esperados']) * 100, 1)
            : 0.0;

        $rateEntrega = $data['destinos_totales'] > 0
            ? round(($data['destinos_entregados'] / $data['destinos_totales']) * 100, 1)
            : 0.0;

        $periodRates[] = [
            'fecha'              => $date,
            'embarques'          => $data['embarques'],
            'pedidos_esperados'  => $data['pedidos_esperados'],
            'pedidos_despachados'=> $data['pedidos_despachados'],
            'destinos_totales'   => $data['destinos_totales'],
            'destinos_entregados'=> $data['destinos_entregados'],
            'rate_despacho'      => $rateDespacho,
            'rate_entrega'       => $rateEntrega,
        ];
    }

    usort($periodRates, static function (array $a, array $b): int {
        return strcmp($a['fecha'], $b['fecha']);
    });

    return $periodRates;
}

function fetchFilterOptions(PDO $pdo): array
{
    $estados = $pdo->query("\n        SELECT code, nombre\n        FROM so_embarque_estado\n        WHERE activo = 1\n        ORDER BY orden, nombre\n    ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $clientes = $pdo->query("\n        SELECT id, razon_social\n        FROM para_clientes\n        WHERE deleted_at IS NULL\n        ORDER BY razon_social\n        LIMIT 100\n    ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $destinatarios = $pdo->query("\n        SELECT id, cod, nombre, cliente_id\n        FROM para_destinatarios\n        WHERE deleted_at IS NULL\n        ORDER BY nombre\n        LIMIT 150\n    ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $moviles = $pdo->query("\n        SELECT id, chapa, CONCAT(COALESCE(chapa, ''), ' · ID ', id) AS label\n        FROM para_moviles\n        WHERE activo = 1\n        ORDER BY chapa\n    ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

    return [
        'estados'       => $estados,
        'clientes'      => $clientes,
        'destinatarios' => $destinatarios,
        'moviles'       => $moviles,
    ];
}

function buildEmptyMetrics(): array
{
    return [
        'total_embarques' => 0,
        'por_estado'      => [],
        'fulfillment_rates' => [
            'pedidos_completos'    => 0,
            'entregas_parciales'   => 0,
            'entregas_fallidas'    => 0,
            'rate_cumplimiento'    => 0.0,
            'rate_entrega_parcial' => 0.0,
        ],
        'tiempos' => [
            'embarques_a_tiempo'    => 0,
            'embarques_con_retraso' => 0,
            'promedio_delay_days'   => 0.0,
            'delay_maximo_days'     => 0,
            'rate_puntualidad'      => 0.0,
        ],
        'volumenes' => [
            'items_esperados_uv'    => 0,
            'items_despachados_uv'  => 0,
            'items_esperados_uc'    => 0,
            'items_despachados_uc'  => 0,
            'rate_despacho_uv'      => 0.0,
            'rate_despacho_uc'      => 0.0,
        ],
    ];
}

function buildPlaceholders(int $count): string
{
    return implode(',', array_fill(0, $count, '?'));
}