<?php
/**
 * SOL · API · PDF Reporte de Cumplimiento de Entregas
 * Ruta: api/reportes/embarque_cumplimiento_pdf.php
 * 
 * Genera PDF del análisis de cumplimiento de entregas
 */
declare(strict_types=1);

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'No autorizado']);
    exit;
}

require_once $BASE . '/vendor/autoload.php';

use Mpdf\Mpdf;

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $filters = [
        'fecha_desde'     => trim((string)($_GET['fecha_desde'] ?? '')),
        'fecha_hasta'     => trim((string)($_GET['fecha_hasta'] ?? '')),
        'cliente_id'      => trim((string)($_GET['cliente_id'] ?? '')),
        'destinatario_id' => trim((string)($_GET['destinatario_id'] ?? '')),
        'estado'          => trim((string)($_GET['estado'] ?? '')),
        'movil_id'        => trim((string)($_GET['movil_id'] ?? '')),
    ];

    if ($filters['fecha_desde'] === '') {
        $filters['fecha_desde'] = date('Y-m-d', strtotime('-30 days'));
    }
    if ($filters['fecha_hasta'] === '') {
        $filters['fecha_hasta'] = date('Y-m-d');
    }
    if ($filters['fecha_desde'] > $filters['fecha_hasta']) {
        [$filters['fecha_desde'], $filters['fecha_hasta']] = [$filters['fecha_hasta'], $filters['fecha_desde']];
    }

    [$embarques] = loadEmbarquesData($pdo, $filters);

    $rows = [];
    $metrics = buildEmptyMetrics();

    if ($embarques) {
        $ids = array_map('intval', array_column($embarques, 'id'));
        $stopStats = fetchStopStats($pdo, $ids);
        $pedidoStats = fetchPedidoStats($pdo, $ids);

        $metrics['total_embarques'] = count($embarques);

        $completos = 0;
        $parciales = 0;
        $fallidos = 0;
        $ontime = 0;
        $late = 0;
        $delaySum = 0.0;
        $delayCount = 0;
        $delayMax = 0;

        $itemsEsperadosUVTotal = 0;
        $itemsDespachadosUVTotal = 0;
        $itemsEsperadosUCTotal = 0;
        $itemsDespachadosUCTotal = 0;

        foreach ($embarques as $emb) {
            $id = (int)$emb['id'];
            $estadoCode = strtoupper((string)($emb['estado_code'] ?? ''));
            $stateKey = $estadoCode !== '' ? $estadoCode : 'OTRO';
            $metrics['por_estado'][$stateKey] = ($metrics['por_estado'][$stateKey] ?? 0) + 1;

            $stop = $stopStats[$id] ?? [
                'total_destinos'      => 0,
                'destinos_entregados' => 0,
                'entregas_parciales'  => 0,
                'destinos_pendientes' => 0,
                'max_delay_days'      => null,
            ];

            $pedido = $pedidoStats[$id] ?? [
                'pedidos_esperados'      => 0,
                'pedidos_despachados'    => 0,
                'items_esperados_uv'     => 0,
                'items_despachados_uv'   => 0,
                'items_esperados_uc'     => 0,
                'items_despachados_uc'   => 0,
            ];

            $totalDestinos      = (int)$stop['total_destinos'];
            $destinosEntregados = (int)$stop['destinos_entregados'];
            $entregasParciales  = (int)$stop['entregas_parciales'];
            $destinosPendientes = (int)$stop['destinos_pendientes'];

            $pedidosEsperados   = (int)$pedido['pedidos_esperados'];
            $pedidosDespachados = (int)$pedido['pedidos_despachados'];
            $itemsEsperadosUV   = (int)$pedido['items_esperados_uv'];
            $itemsDespachadosUV = (int)$pedido['items_despachados_uv'];
            $itemsEsperadosUC   = (int)$pedido['items_esperados_uc'];
            $itemsDespachadosUC = (int)$pedido['items_despachados_uc'];

            $itemsEsperadosUVTotal   += $itemsEsperadosUV;
            $itemsDespachadosUVTotal += $itemsDespachadosUV;
            $itemsEsperadosUCTotal   += $itemsEsperadosUC;
            $itemsDespachadosUCTotal += $itemsDespachadosUC;

            $fulfilled = min($pedidosDespachados, $destinosEntregados);
            $fulfillmentRate = ($pedidosEsperados > 0)
                ? ($fulfilled / $pedidosEsperados) * 100
                : 0.0;

            $delayDays = $stop['max_delay_days'];
            if ($delayDays === null && !empty($emb['salida_at']) && !empty($emb['fecha_programada'])) {
                $programada = new DateTime((string)$emb['fecha_programada']);
                $salida = new DateTime((string)$emb['salida_at']);
                $delayDays = (int)$programada->diff($salida)->format('%r%a');
            }

            if ($delayDays !== null) {
                $delayDays = max(0, (int)$delayDays);
                $delaySum += $delayDays;
                $delayCount++;
                if ($delayDays > $delayMax) {
                    $delayMax = $delayDays;
                }
            }

            $deliveryStatus = 'EN_TRANSITO';
            $isClosedState = in_array($estadoCode, ['COMPLETADO', 'ENTREGADO', 'CERRADO', 'CERR'], true);

            if ($totalDestinos > 0 && $destinosEntregados >= $totalDestinos) {
                if ($delayDays !== null && $delayDays > 1) {
                    $deliveryStatus = 'COMPLETADO_TARDE';
                    ++$late;
                } else {
                    $deliveryStatus = 'COMPLETADO';
                    ++$ontime;
                }
                ++$completos;
            } elseif ($destinosEntregados > 0) {
                $deliveryStatus = 'PARCIAL';
                ++$parciales;
                if ($delayDays !== null && $delayDays > 1) {
                    ++$late;
                }
            } elseif ($totalDestinos > 0) {
                if ($isClosedState) {
                    $deliveryStatus = 'FALLIDO';
                    ++$fallidos;
                    ++$late;
                } else {
                    $deliveryStatus = 'PENDIENTE';
                }
            } elseif ($pedidosEsperados === 0 && $estadoCode === 'CREADO') {
                $deliveryStatus = 'PENDIENTE';
            }

            $rows[] = [
                'codigo'               => $emb['codigo'],
                'fecha_embarque'       => $emb['fecha_embarque'],
                'estado_nombre'        => $emb['estado_nombre'] ?? 'Sin estado',
                'movil'                => $emb['movil'],
                'total_destinos'       => $totalDestinos,
                'pedidos_esperados'    => $pedidosEsperados,
                'pedidos_despachados'  => $pedidosDespachados,
                'destinos_entregados'  => $destinosEntregados,
                'destinos_pendientes'  => $destinosPendientes,
                'items_esperados_uv'   => $itemsEsperadosUV,
                'items_despachados_uv' => $itemsDespachadosUV,
                'items_esperados_uc'   => $itemsEsperadosUC,
                'items_despachados_uc' => $itemsDespachadosUC,
                'fulfillment_rate'     => round($fulfillmentRate, 1),
                'delay_days'           => $delayDays,
                'delivery_status'      => $deliveryStatus,
            ];
        }

        $metrics['fulfillment_rates'] = [
            'pedidos_completos'    => $completos,
            'entregas_parciales'   => $parciales,
            'entregas_fallidas'    => $fallidos,
            'rate_cumplimiento'    => $metrics['total_embarques'] > 0 ? round(($completos / $metrics['total_embarques']) * 100, 1) : 0.0,
            'rate_entrega_parcial' => $metrics['total_embarques'] > 0 ? round(($parciales / $metrics['total_embarques']) * 100, 1) : 0.0,
        ];

        $metrics['tiempos'] = [
            'embarques_a_tiempo'    => $ontime,
            'embarques_con_retraso' => $late,
            'promedio_delay_days'   => $delayCount > 0 ? round($delaySum / $delayCount, 1) : 0.0,
            'delay_maximo_days'     => $delayMax,
            'rate_puntualidad'      => $metrics['total_embarques'] > 0 ? round(($ontime / $metrics['total_embarques']) * 100, 1) : 0.0,
        ];

        $metrics['volumenes'] = [
            'items_esperados_uv'   => $itemsEsperadosUVTotal,
            'items_despachados_uv' => $itemsDespachadosUVTotal,
            'items_esperados_uc'   => $itemsEsperadosUCTotal,
            'items_despachados_uc' => $itemsDespachadosUCTotal,
            'rate_despacho_uv'     => $itemsEsperadosUVTotal > 0 ? round(($itemsDespachadosUVTotal / $itemsEsperadosUVTotal) * 100, 1) : 0.0,
            'rate_despacho_uc'     => $itemsEsperadosUCTotal > 0 ? round(($itemsDespachadosUCTotal / $itemsEsperadosUCTotal) * 100, 1) : 0.0,
        ];
    }

    $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4-L',
        'default_font' => 'sans-serif',
        'margin_left' => 10,
        'margin_right' => 10,
        'margin_top' => 30,
        'margin_bottom' => 20,
        'margin_header' => 5,
        'margin_footer' => 5,
    ]);

    $mpdf->SetTitle('Reporte de Cumplimiento de Entregas');
    $mpdf->SetAuthor('SOL System');

    $fmtNumber = static fn($value): string => number_format((float)$value, 0, ',', '.');
    $fmtPercent = static fn($value): string => number_format((float)$value, 1, ',', '.') . '%';
    $fmtDate = static fn(?string $value): string => $value ? date('d/m/Y', strtotime($value)) : '-';
    $h = static fn($value): string => htmlspecialchars((string)$value, ENT_QUOTES, 'UTF-8');

    $rateCumplimiento = $fmtPercent($metrics['fulfillment_rates']['rate_cumplimiento'] ?? 0);
    $rateCumplimientoDesc = sprintf('%s de %s embarques',
        $fmtNumber($metrics['fulfillment_rates']['pedidos_completos'] ?? 0),
        $fmtNumber($metrics['total_embarques'] ?? 0)
    );

    $rateParcial = $fmtPercent($metrics['fulfillment_rates']['rate_entrega_parcial'] ?? 0);
    $rateParcialDesc = sprintf('%s embarques con entrega parcial',
        $fmtNumber($metrics['fulfillment_rates']['entregas_parciales'] ?? 0)
    );

    $rateDespacho = $fmtPercent($metrics['volumenes']['rate_despacho_uv'] ?? 0);
    $rateDespachoDesc = sprintf('%s / %s ítems UV',
        $fmtNumber($metrics['volumenes']['items_despachados_uv'] ?? 0),
        $fmtNumber($metrics['volumenes']['items_esperados_uv'] ?? 0)
    );

    $ratePuntualidad = $fmtPercent($metrics['tiempos']['rate_puntualidad'] ?? 0);
    $ratePuntualidadDesc = sprintf('%s embarques a tiempo · %s con retraso',
        $fmtNumber($metrics['tiempos']['embarques_a_tiempo'] ?? 0),
        $fmtNumber($metrics['tiempos']['embarques_con_retraso'] ?? 0)
    );

    $badgeForRate = static function (float $rate): string {
        if ($rate >= 95.0) {
            return 'badge-success';
        }
        if ($rate >= 80.0) {
            return 'badge-warning';
        }
        if ($rate > 0) {
            return 'badge-danger';
        }
        return 'badge-info';
    };

    ob_start();
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="utf-8">
        <style>
            body { font-family: sans-serif; font-size: 9pt; color: #333; }
            .header { text-align: center; margin-bottom: 20px; }
            .header h1 { font-size: 16pt; margin: 0; color: #2c5aa0; }
            .header h2 { font-size: 12pt; margin: 5px 0; color: #666; }
            .filters { background: #f8f9fa; padding: 10px; border-radius: 4px; margin-bottom: 15px; }
            .filters strong { color: #495057; }
            .summary { margin-bottom: 20px; }
            .summary-grid { display: table; width: 100%; }
            .summary-col { display: table-cell; width: 25%; vertical-align: top; padding: 0 10px; }
            .metric-box { background: #fff; border: 1px solid #dee2e6; border-radius: 4px; padding: 10px; margin-bottom: 10px; }
            .metric-title { font-weight: bold; color: #495057; font-size: 8pt; }
            .metric-value { font-size: 14pt; font-weight: bold; color: #2c5aa0; }
            .metric-subtitle { font-size: 8pt; color: #6c757d; }
            table { width: 100%; border-collapse: collapse; margin-top: 10px; font-size: 8pt; }
            th, td { border: 1px solid #dee2e6; padding: 6px 4px; text-align: left; }
            th { background-color: #f8f9fa; font-weight: bold; color: #495057; }
            .text-center { text-align: center; }
            .badge { display: inline-block; padding: 2px 6px; border-radius: 3px; font-size: 7pt; font-weight: bold; }
            .badge-success { background-color: #d4edda; color: #155724; }
            .badge-warning { background-color: #fff3cd; color: #856404; }
            .badge-danger { background-color: #f8d7da; color: #721c24; }
            .badge-info { background-color: #d1ecf1; color: #0c5460; }
            .footer { margin-top: 20px; font-size: 7pt; color: #6c757d; text-align: center; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>SOL · Reporte de Cumplimiento de Entregas</h1>
            <h2>Análisis de Performance de Entrega</h2>
        </div>

        <div class="filters">
            <strong>Filtros aplicados:</strong>
            Período: <?= $fmtDate($filters['fecha_desde']) ?> - <?= $fmtDate($filters['fecha_hasta']) ?> ·
            Total embarques: <?= $fmtNumber($metrics['total_embarques'] ?? 0) ?>
        </div>

        <div class="summary">
            <div class="summary-grid">
                <div class="summary-col">
                    <div class="metric-box">
                        <div class="metric-title">CUMPLIMIENTO TOTAL</div>
                        <div class="metric-value"><?= $rateCumplimiento ?></div>
                        <div class="metric-subtitle"><?= $h($rateCumplimientoDesc) ?></div>
                    </div>
                </div>
                <div class="summary-col">
                    <div class="metric-box">
                        <div class="metric-title">ENTREGAS PARCIALES</div>
                        <div class="metric-value"><?= $rateParcial ?></div>
                        <div class="metric-subtitle"><?= $h($rateParcialDesc) ?></div>
                    </div>
                </div>
                <div class="summary-col">
                    <div class="metric-box">
                        <div class="metric-title">RATE DE DESPACHO</div>
                        <div class="metric-value"><?= $rateDespacho ?></div>
                        <div class="metric-subtitle"><?= $h($rateDespachoDesc) ?></div>
                    </div>
                </div>
                <div class="summary-col">
                    <div class="metric-box">
                        <div class="metric-title">PUNTUALIDAD</div>
                        <div class="metric-value"><?= $ratePuntualidad ?></div>
                        <div class="metric-subtitle"><?= $h($ratePuntualidadDesc) ?></div>
                    </div>
                </div>
            </div>
        </div>

        <table>
            <thead>
                <tr>
                    <th style="width: 10%;">Código</th>
                    <th style="width: 10%;">Fecha</th>
                    <th style="width: 12%;">Estado</th>
                    <th style="width: 10%;">Móvil</th>
                    <th style="width: 8%;" class="text-center">Destinos</th>
                    <th style="width: 8%;" class="text-center">Pedidos Esp.</th>
                    <th style="width: 8%;" class="text-center">Pedidos Desp.</th>
                    <th style="width: 8%;" class="text-center">Destinos Entr.</th>
                    <th style="width: 12%;" class="text-center">Items UV</th>
                    <th style="width: 12%;" class="text-center">Items UC</th>
                    <th style="width: 6%;" class="text-center">Rate</th>
                    <th style="width: 6%;" class="text-center">Delay</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!$rows): ?>
                    <tr>
                        <td colspan="12" class="text-center">Sin resultados para los filtros seleccionados.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($rows as $row):
                        $rate = (float)($row['fulfillment_rate'] ?? 0);
                        $badgeClass = $badgeForRate($rate);
                    ?>
                        <tr>
                            <td><?= $h($row['codigo'] ?? '-') ?></td>
                            <td class="text-center"><?= $fmtDate($row['fecha_embarque'] ?? null) ?></td>
                            <td><?= $h($row['estado_nombre'] ?? '-') ?></td>
                            <td class="text-center"><?= $h($row['movil'] ?? '-') ?></td>
                            <td class="text-center"><?= $fmtNumber($row['total_destinos'] ?? 0) ?></td>
                            <td class="text-center"><?= $fmtNumber($row['pedidos_esperados'] ?? 0) ?></td>
                            <td class="text-center"><?= $fmtNumber($row['pedidos_despachados'] ?? 0) ?></td>
                            <td class="text-center"><?= $fmtNumber($row['destinos_entregados'] ?? 0) ?></td>
                            <td class="text-center"><?= $fmtNumber($row['items_despachados_uv'] ?? 0) ?> / <?= $fmtNumber($row['items_esperados_uv'] ?? 0) ?></td>
                            <td class="text-center"><?= $fmtNumber($row['items_despachados_uc'] ?? 0) ?> / <?= $fmtNumber($row['items_esperados_uc'] ?? 0) ?></td>
                            <td class="text-center"><span class="badge <?= $badgeClass ?>"><?= $fmtPercent($rate) ?></span></td>
                            <td class="text-center"><?= $row['delay_days'] === null ? '-' : $fmtNumber($row['delay_days']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>

        <div class="footer">
            Generado el <?= date('d/m/Y H:i:s') ?> · Usuario: <?= $h($_SESSION['usuario_nombre'] ?? 'Sistema') ?>
        </div>
    </body>
    </html>
    <?php
    $html = ob_get_clean();

    $mpdf->WriteHTML($html);

    $filename = 'reporte-cumplimiento-entregas-' . date('Y-m-d') . '.pdf';
    $mpdf->Output($filename, 'I');
    exit;

} catch (Throwable $e) {
    error_log('[CumplimientoPDF] ' . $e->getMessage());
    http_response_code(500);
    $message = 'Error generando PDF';
    if (!empty($_GET['debug']) || PHP_SAPI === 'cli') {
        $message .= ': ' . $e->getMessage();
    }
    echo json_encode(['ok' => false, 'error' => $message], JSON_UNESCAPED_UNICODE);
    exit;
}

function loadEmbarquesData(PDO $pdo, array $filters): array
{
    $where = [
        'DATE(COALESCE(e.carga_fecha, e.creado_at)) >= ?',
        'DATE(COALESCE(e.carga_fecha, e.creado_at)) <= ?',
    ];

    $params = [
        $filters['fecha_desde'],
        $filters['fecha_hasta'],
    ];

    if ($filters['cliente_id'] !== '') {
        $where[] = 'EXISTS (
            SELECT 1
            FROM so_embarque_parada p
            JOIN para_destinatarios dest ON dest.id = p.destinatario_id
            WHERE p.embarque_id = e.id AND dest.cliente_id = ?
        )';
        $params[] = $filters['cliente_id'];
    }

    if ($filters['destinatario_id'] !== '') {
        $where[] = 'EXISTS (
            SELECT 1
            FROM so_embarque_parada p
            WHERE p.embarque_id = e.id AND p.destinatario_id = ?
        )';
        $params[] = $filters['destinatario_id'];
    }

    if ($filters['estado'] !== '') {
        $where[] = 'est.code = ?';
        $params[] = $filters['estado'];
    }

    if ($filters['movil_id'] !== '') {
        $where[] = 'e.movil_id = ?';
        $params[] = $filters['movil_id'];
    }

    $whereSql = implode(' AND ', $where);

    $sql = <<<SQL
        SELECT
            e.id,
            e.codigo,
            DATE(COALESCE(e.carga_fecha, e.creado_at)) AS fecha_embarque,
            COALESCE(e.carga_fecha, e.creado_at) AS fecha_programada,
            est.code AS estado_code,
            est.nombre AS estado_nombre,
            m.chapa AS movil,
            e.salida_at
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        WHERE {$whereSql}
        ORDER BY COALESCE(e.carga_fecha, e.creado_at) DESC, e.codigo
        LIMIT 500
    SQL;

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);

    return [$embarques];
}

function fetchStopStats(PDO $pdo, array $ids): array
{
    if (!$ids) {
        return [];
    }

    $placeholders = buildPlaceholders(count($ids));
    $sql = <<<SQL
        SELECT
            p.embarque_id,
            COUNT(*) AS total_destinos,
            SUM(CASE WHEN p.hora_fin_descarga IS NOT NULL OR p.hora_salida IS NOT NULL THEN 1 ELSE 0 END) AS destinos_entregados,
            SUM(CASE WHEN p.hora_fin_descarga IS NULL AND p.hora_inicio_descarga IS NOT NULL THEN 1 ELSE 0 END) AS entregas_parciales,
            SUM(CASE WHEN p.hora_inicio_descarga IS NULL THEN 1 ELSE 0 END) AS destinos_pendientes,
            MAX(CASE
                WHEN p.hora_fin_descarga IS NOT NULL THEN DATEDIFF(DATE(p.hora_fin_descarga), DATE(COALESCE(e.carga_fecha, e.creado_at)))
                WHEN p.hora_salida IS NOT NULL THEN DATEDIFF(DATE(p.hora_salida), DATE(COALESCE(e.carga_fecha, e.creado_at)))
                ELSE NULL
            END) AS max_delay_days
        FROM so_embarque_parada p
        JOIN so_embarque e ON e.id = p.embarque_id
        WHERE p.embarque_id IN ({$placeholders})
        GROUP BY p.embarque_id
    SQL;

    $stmt = $pdo->prepare($sql);
    $stmt->execute($ids);

    $result = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $result[(int)$row['embarque_id']] = [
            'total_destinos'      => (int)($row['total_destinos'] ?? 0),
            'destinos_entregados' => (int)($row['destinos_entregados'] ?? 0),
            'entregas_parciales'  => (int)($row['entregas_parciales'] ?? 0),
            'destinos_pendientes' => (int)($row['destinos_pendientes'] ?? 0),
            'max_delay_days'      => $row['max_delay_days'] !== null ? (int)$row['max_delay_days'] : null,
        ];
    }

    return $result;
}

function fetchPedidoStats(PDO $pdo, array $ids): array
{
    if (!$ids) {
        return [];
    }

    $placeholders = buildPlaceholders(count($ids));
    $sql = <<<SQL
        SELECT
            ep.embarque_id,
            COUNT(DISTINCT ep.preembarque_id) AS pedidos_esperados,
            COUNT(DISTINCT CASE WHEN sp.fin_at IS NOT NULL THEN ep.preembarque_id END) AS pedidos_despachados,
            SUM(pdi.expected_uv) AS items_esperados_uv,
            SUM(pdi.shipped_uv) AS items_despachados_uv,
            SUM(pdi.expected_uc) AS items_esperados_uc,
            SUM(pdi.shipped_uc) AS items_despachados_uc
        FROM so_embarque_pre ep
        JOIN so_preembarque sp ON sp.id = ep.preembarque_id
        JOIN so_pedido ped ON ped.id = sp.pedido_id
        JOIN so_pedido_dest pd ON pd.pedido_id = ped.id
        JOIN so_pedido_dest_item pdi ON pdi.pedido_dest_id = pd.id
        WHERE ep.embarque_id IN ({$placeholders})
        GROUP BY ep.embarque_id
    SQL;

    $stmt = $pdo->prepare($sql);
    $stmt->execute($ids);

    $result = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $result[(int)$row['embarque_id']] = [
            'pedidos_esperados'      => (int)($row['pedidos_esperados'] ?? 0),
            'pedidos_despachados'    => (int)($row['pedidos_despachados'] ?? 0),
            'items_esperados_uv'     => (int)($row['items_esperados_uv'] ?? 0),
            'items_despachados_uv'   => (int)($row['items_despachados_uv'] ?? 0),
            'items_esperados_uc'     => (int)($row['items_esperados_uc'] ?? 0),
            'items_despachados_uc'   => (int)($row['items_despachados_uc'] ?? 0),
        ];
    }

    return $result;
}

function buildPlaceholders(int $count): string
{
    return implode(',', array_fill(0, $count, '?'));
}

function buildEmptyMetrics(): array
{
    return [
        'total_embarques'   => 0,
        'por_estado'        => [],
        'fulfillment_rates' => [
            'pedidos_completos'    => 0,
            'entregas_parciales'   => 0,
            'entregas_fallidas'    => 0,
            'rate_cumplimiento'    => 0.0,
            'rate_entrega_parcial' => 0.0,
        ],
        'tiempos' => [
            'embarques_a_tiempo'    => 0,
            'embarques_con_retraso' => 0,
            'promedio_delay_days'   => 0.0,
            'delay_maximo_days'     => 0,
            'rate_puntualidad'      => 0.0,
        ],
        'volumenes' => [
            'items_esperados_uv'   => 0,
            'items_despachados_uv' => 0,
            'items_esperados_uc'   => 0,
            'items_despachados_uc' => 0,
            'rate_despacho_uv'     => 0.0,
            'rate_despacho_uc'     => 0.0,
        ],
    ];
}