<?php
/**
 * SOL · API · Reporte de Cumplimiento de Entregas (Simplified)
 * Test version with simplified queries
 */
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'No autorizado']);
    exit;
}

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Parse filters
    $fechaDesde = trim($_GET['fecha_desde'] ?? '');
    $fechaHasta = trim($_GET['fecha_hasta'] ?? '');
    
    // Default date range (last 30 days)
    if (!$fechaDesde) {
        $fechaDesde = date('Y-m-d', strtotime('-30 days'));
    }
    if (!$fechaHasta) {
        $fechaHasta = date('Y-m-d');
    }

    // Build WHERE clause
    $where = ['1=1'];
    $params = [];

    if ($fechaDesde) {
        $where[] = 'DATE(COALESCE(e.carga_fecha, e.creado_at)) >= ?';
        $params[] = $fechaDesde;
    }
    if ($fechaHasta) {
        $where[] = 'DATE(COALESCE(e.carga_fecha, e.creado_at)) <= ?';
        $params[] = $fechaHasta;
    }

    $whereClause = 'WHERE ' . implode(' AND ', $where);

    // Simplified main query
    $sql = "
        SELECT 
            e.id,
            e.codigo,
            DATE(COALESCE(e.carga_fecha, e.creado_at)) as fecha_embarque,
            est.code as estado_code,
            est.nombre as estado_nombre,
            
            -- Embarque info
            dep.code as deposito_code,
            dep.nombre as deposito_nombre,
            m.chapa as movil,
            CONCAT(COALESCE(ch.nombre, ''), ' ', COALESCE(ch.apellido, '')) as chofer,
            
            -- Timing
            e.llegada_at,
            e.carga_inicio_at,
            e.carga_fin_at,
            e.salida_at,
            
            -- Simple counts
            (SELECT COUNT(*) FROM so_embarque_parada p WHERE p.embarque_id = e.id) as total_destinos,
            (SELECT COUNT(DISTINCT ep.preembarque_id) FROM so_embarque_pre ep WHERE ep.embarque_id = e.id) as pedidos_esperados,
            
            -- Simplified metrics
            1 as pedidos_despachados,
            0 as destinos_entregados,
            1 as items_esperados_uv,
            1 as items_despachados_uv,
            0 as items_esperados_uc,
            0 as items_despachados_uc,
            0 as delay_days,
            'PENDIENTE' as delivery_status
            
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        LEFT JOIN para_choferes ch ON ch.id = e.chofer_id
        {$whereClause}
        ORDER BY COALESCE(e.carga_fecha, e.creado_at) DESC, e.codigo
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Simple metrics
    $totalEmbarques = count($embarques);
    $metrics = [
        'total_embarques' => $totalEmbarques,
        'por_estado' => [],
        'fulfillment_rates' => [
            'pedidos_completos' => $totalEmbarques,
            'entregas_parciales' => 0,
            'entregas_fallidas' => 0,
            'rate_cumplimiento' => 100.0,
            'rate_entrega_parcial' => 0.0
        ],
        'tiempos' => [
            'promedio_delay_days' => 0.0,
            'embarques_a_tiempo' => $totalEmbarques,
            'embarques_con_retraso' => 0,
            'rate_puntualidad' => 100.0
        ],
        'volumenes' => [
            'items_esperados_uv' => $totalEmbarques,
            'items_despachados_uv' => $totalEmbarques,
            'items_esperados_uc' => 0,
            'items_despachados_uc' => 0,
            'rate_despacho_uv' => 100.0,
            'rate_despacho_uc' => 0.0
        ]
    ];
    
    // Count by status
    foreach ($embarques as $emb) {
        $estado = $emb['estado_code'];
        $metrics['por_estado'][$estado] = ($metrics['por_estado'][$estado] ?? 0) + 1;
    }
    
    // Get filter options
    $estados = $pdo->query("
        SELECT code, nombre 
        FROM so_embarque_estado 
        WHERE activo = 1 
        ORDER BY orden
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $clientes = $pdo->query("
        SELECT DISTINCT c.id, c.razon_social
        FROM para_clientes c
        WHERE c.deleted_at IS NULL
        ORDER BY c.razon_social
        LIMIT 20
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $destinatarios = $pdo->query("
        SELECT id, cod, nombre, cliente_id
        FROM para_destinatarios 
        WHERE deleted_at IS NULL 
        ORDER BY nombre
        LIMIT 50
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $moviles = $pdo->query("
        SELECT id, chapa, 
               CONCAT(COALESCE(chapa, ''), ' (ID: ', id, ')') as label
        FROM para_moviles 
        WHERE activo = 1 
        ORDER BY chapa
    ")->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'ok' => true,
        'data' => $embarques,
        'metrics' => $metrics,
        'filters' => [
            'fecha_desde' => $fechaDesde,
            'fecha_hasta' => $fechaHasta
        ],
        'filter_options' => [
            'estados' => $estados,
            'clientes' => $clientes,
            'destinatarios' => $destinatarios,
            'moviles' => $moviles
        ]
    ], JSON_UNESCAPED_UNICODE);

} catch (Exception $e) {
    error_log("Error en reporte de cumplimiento: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'ok' => false, 
        'error' => 'Error interno: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ], JSON_UNESCAPED_UNICODE);
}