<?php
// api/reportes/embarque_monitoreo.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = getPDO();
    
    // Get filters from request
    $fechaDesde = $_GET['fecha_desde'] ?? null;
    $fechaHasta = $_GET['fecha_hasta'] ?? null;
    $estado = $_GET['estado'] ?? null;
    $movilId = isset($_GET['movil_id']) && $_GET['movil_id'] !== '' ? (int)$_GET['movil_id'] : null;
    $choferId = isset($_GET['chofer_id']) && $_GET['chofer_id'] !== '' ? (int)$_GET['chofer_id'] : null;
    $depositoId = isset($_GET['deposito_id']) && $_GET['deposito_id'] !== '' ? (int)$_GET['deposito_id'] : null;
    
    // Build WHERE conditions
    $whereConditions = [];
    $params = [];
    
    if ($fechaDesde) {
        $whereConditions[] = "DATE(e.creado_at) >= ?";
        $params[] = $fechaDesde;
    }
    
    if ($fechaHasta) {
        $whereConditions[] = "DATE(e.creado_at) <= ?";
        $params[] = $fechaHasta;
    }
    
    if ($estado) {
        $whereConditions[] = "est.code = ?";
        $params[] = $estado;
    }
    
    if ($movilId) {
        $whereConditions[] = "e.movil_id = ?";
        $params[] = $movilId;
    }
    
    if ($choferId) {
        $whereConditions[] = "e.chofer_id = ?";
        $params[] = $choferId;
    }
    
    if ($depositoId) {
        $whereConditions[] = "e.deposito_id = ?";
        $params[] = $depositoId;
    }
    
    $whereClause = $whereConditions ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
    
    // Main query for embarques data
    $sql = "
        SELECT 
            e.id,
            e.codigo,
            e.creado_at,
            DATE(e.creado_at) as fecha_embarque,
            est.code as estado_code,
            est.nombre as estado,
            dep.code as deposito_code,
            dep.nombre as deposito,
            m.chapa as movil,
            m.id as movil_id,
            ch.nombre as chofer,
            ch.id as chofer_id,
            e.llegada_at,
            e.carga_inicio_at,
            e.carga_fin_at,
            e.salida_at,
            e.ayudantes_cant,
            e.temp_salida_c,
            e.km_inicial,
            e.observacion,
            
            -- Metrics calculation
            CASE 
                WHEN e.llegada_at IS NOT NULL AND e.carga_inicio_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.carga_inicio_at)
                ELSE NULL 
            END as tiempo_espera_min,
            
            CASE 
                WHEN e.carga_inicio_at IS NOT NULL AND e.carga_fin_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.carga_inicio_at, e.carga_fin_at)
                ELSE NULL 
            END as tiempo_carga_min,
            
            CASE 
                WHEN e.llegada_at IS NOT NULL AND e.salida_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.salida_at)
                ELSE NULL 
            END as tiempo_total_planta_min,
            
            -- Count destinations/stops
            (SELECT COUNT(*) FROM so_embarque_parada p WHERE p.embarque_id = e.id) as total_destinos,
            
            -- Count preembarques attached
            (SELECT COUNT(*) FROM so_embarque_pre ep WHERE ep.embarque_id = e.id) as total_preembarques
            
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        LEFT JOIN para_choferes ch ON ch.id = e.chofer_id
        {$whereClause}
        ORDER BY e.creado_at DESC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Summary statistics
    $totalEmbarques = count($embarques);
    $porEstado = [];
    $totalDestinos = 0;
    $tiemposEspera = [];
    $tiemposCarga = [];
    $tiemposPlanta = [];
    
    foreach ($embarques as $emb) {
        $estado = $emb['estado_code'];
        $porEstado[$estado] = ($porEstado[$estado] ?? 0) + 1;
        $totalDestinos += (int)$emb['total_destinos'];
        
        if ($emb['tiempo_espera_min'] !== null) {
            $tiemposEspera[] = (int)$emb['tiempo_espera_min'];
        }
        if ($emb['tiempo_carga_min'] !== null) {
            $tiemposCarga[] = (int)$emb['tiempo_carga_min'];
        }
        if ($emb['tiempo_total_planta_min'] !== null) {
            $tiemposPlanta[] = (int)$emb['tiempo_total_planta_min'];
        }
    }
    
    $promedioEspera = $tiemposEspera ? round(array_sum($tiemposEspera) / count($tiemposEspera), 1) : null;
    $promedioCarga = $tiemposCarga ? round(array_sum($tiemposCarga) / count($tiemposCarga), 1) : null;
    $promedioPlanta = $tiemposPlanta ? round(array_sum($tiemposPlanta) / count($tiemposPlanta), 1) : null;
    
    // Get filter options for frontend
    $estados = $pdo->query("
        SELECT code, nombre 
        FROM so_embarque_estado 
        WHERE activo = 1 
        ORDER BY orden
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $moviles = $pdo->query("
        SELECT id, chapa, 
               CONCAT(COALESCE(chapa, ''), ' (ID: ', id, ')') as label
        FROM para_moviles 
        WHERE activo = 1 
        ORDER BY chapa
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $choferes = $pdo->query("
        SELECT id, nombre 
        FROM para_choferes 
        WHERE activo = 1 
        ORDER BY nombre
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $depositos = $pdo->query("
        SELECT id, code, nombre,
               CONCAT(code, ' - ', nombre) as label
        FROM wh_deposito 
        ORDER BY code
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $response = [
        'ok' => true,
        'data' => $embarques,
        'summary' => [
            'total_embarques' => $totalEmbarques,
            'total_destinos' => $totalDestinos,
            'por_estado' => $porEstado,
            'tiempos_promedio' => [
                'espera_min' => $promedioEspera,
                'carga_min' => $promedioCarga,
                'planta_min' => $promedioPlanta
            ]
        ],
        'filters' => [
            'estados' => $estados,
            'moviles' => $moviles,
            'choferes' => $choferes,
            'depositos' => $depositos
        ]
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    error_log("Error in embarque_monitoreo.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error interno del servidor'
    ], JSON_UNESCAPED_UNICODE);
}