<?php
// api/reportes/embarque_tiempos.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ini_set('display_errors', '0');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = getPDO();
    
    // Get filters from request
    $fechaDesde = $_GET['fecha_desde'] ?? null;
    $fechaHasta = $_GET['fecha_hasta'] ?? null;
    $estado = $_GET['estado'] ?? null;
    $movilId = isset($_GET['movil_id']) && $_GET['movil_id'] !== '' ? (int)$_GET['movil_id'] : null;
    $choferId = isset($_GET['chofer_id']) && $_GET['chofer_id'] !== '' ? (int)$_GET['chofer_id'] : null;
    $depositoId = isset($_GET['deposito_id']) && $_GET['deposito_id'] !== '' ? (int)$_GET['deposito_id'] : null;
    
    // Build WHERE conditions
    $whereConditions = [];
    $params = [];
    
    if ($fechaDesde) {
        $whereConditions[] = "DATE(e.creado_at) >= ?";
        $params[] = $fechaDesde;
    }
    
    if ($fechaHasta) {
        $whereConditions[] = "DATE(e.creado_at) <= ?";
        $params[] = $fechaHasta;
    }
    
    if ($estado) {
        $whereConditions[] = "est.code = ?";
        $params[] = $estado;
    }
    
    if ($movilId) {
        $whereConditions[] = "e.movil_id = ?";
        $params[] = $movilId;
    }
    
    if ($choferId) {
        $whereConditions[] = "e.chofer_id = ?";
        $params[] = $choferId;
    }
    
    if ($depositoId) {
        $whereConditions[] = "e.deposito_id = ?";
        $params[] = $depositoId;
    }
    
    $whereClause = $whereConditions ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
    
    // Main query for detailed timing analysis
    $sql = "
        SELECT 
            e.id,
            e.codigo,
            e.creado_at,
            DATE(e.creado_at) as fecha_embarque,
            est.code as estado_code,
            est.nombre as estado,
            dep.code as deposito_code,
            dep.nombre as deposito,
            m.chapa as movil,
            ch.nombre as chofer,
            
            -- Timestamps
            e.llegada_at,
            e.carga_inicio_at,
            e.carga_fin_at,
            e.salida_at,
            
            -- Timing calculations (in minutes)
            CASE 
                WHEN e.llegada_at IS NOT NULL AND e.carga_inicio_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.carga_inicio_at)
                ELSE NULL 
            END as tiempo_espera_min,
            
            CASE 
                WHEN e.carga_inicio_at IS NOT NULL AND e.carga_fin_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.carga_inicio_at, e.carga_fin_at)
                ELSE NULL 
            END as tiempo_carga_min,
            
            CASE 
                WHEN e.llegada_at IS NOT NULL AND e.salida_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.salida_at)
                ELSE NULL 
            END as tiempo_total_planta_min,
            
            -- Additional metrics
            e.ayudantes_cant,
            (SELECT COUNT(*) FROM so_embarque_parada p WHERE p.embarque_id = e.id) as total_destinos,
            
            -- Route timing analysis (average time per destination)
            CASE 
                WHEN e.salida_at IS NOT NULL AND EXISTS(
                    SELECT 1 FROM so_embarque_parada p WHERE p.embarque_id = e.id AND p.hora_llegada IS NOT NULL
                ) THEN (
                    SELECT TIMESTAMPDIFF(MINUTE, e.salida_at, MAX(p.hora_llegada)) 
                    FROM so_embarque_parada p 
                    WHERE p.embarque_id = e.id AND p.hora_llegada IS NOT NULL
                )
                ELSE NULL 
            END as tiempo_ruta_total_min,
            
            -- Efficiency metrics
            CASE 
                WHEN e.carga_inicio_at IS NOT NULL AND e.carga_fin_at IS NOT NULL AND (SELECT COUNT(*) FROM so_embarque_parada p WHERE p.embarque_id = e.id) > 0
                THEN TIMESTAMPDIFF(MINUTE, e.carga_inicio_at, e.carga_fin_at) / (SELECT COUNT(*) FROM so_embarque_parada p WHERE p.embarque_id = e.id)
                ELSE NULL 
            END as tiempo_carga_por_destino_min
            
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        LEFT JOIN para_choferes ch ON ch.id = e.chofer_id
        {$whereClause}
        ORDER BY e.creado_at DESC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate comprehensive statistics
    $stats = [
        'total_embarques' => count($embarques),
        'embarques_con_tiempos' => 0,
        'tiempo_espera' => ['valores' => [], 'promedio' => null, 'minimo' => null, 'maximo' => null],
        'tiempo_carga' => ['valores' => [], 'promedio' => null, 'minimo' => null, 'maximo' => null],
        'tiempo_planta' => ['valores' => [], 'promedio' => null, 'minimo' => null, 'maximo' => null],
        'tiempo_ruta' => ['valores' => [], 'promedio' => null, 'minimo' => null, 'maximo' => null],
        'eficiencia_carga' => ['valores' => [], 'promedio' => null, 'minimo' => null, 'maximo' => null],
        'distribucion_ayudantes' => [],
        'analisis_por_movil' => [],
        'analisis_por_deposito' => []
    ];
    
    foreach ($embarques as $emb) {
        $hasTimings = false;
        
        // Collect timing data for statistics
        if ($emb['tiempo_espera_min'] !== null) {
            $stats['tiempo_espera']['valores'][] = (int)$emb['tiempo_espera_min'];
            $hasTimings = true;
        }
        if ($emb['tiempo_carga_min'] !== null) {
            $stats['tiempo_carga']['valores'][] = (int)$emb['tiempo_carga_min'];
            $hasTimings = true;
        }
        if ($emb['tiempo_total_planta_min'] !== null) {
            $stats['tiempo_planta']['valores'][] = (int)$emb['tiempo_total_planta_min'];
            $hasTimings = true;
        }
        if ($emb['tiempo_ruta_total_min'] !== null) {
            $stats['tiempo_ruta']['valores'][] = (int)$emb['tiempo_ruta_total_min'];
            $hasTimings = true;
        }
        if ($emb['tiempo_carga_por_destino_min'] !== null) {
            $stats['eficiencia_carga']['valores'][] = (float)$emb['tiempo_carga_por_destino_min'];
            $hasTimings = true;
        }
        
        if ($hasTimings) {
            $stats['embarques_con_tiempos']++;
        }
        
        // Ayudantes distribution
        $ayudantes = (int)($emb['ayudantes_cant'] ?: 0);
        $stats['distribucion_ayudantes'][$ayudantes] = ($stats['distribucion_ayudantes'][$ayudantes] ?? 0) + 1;
        
        // Análisis por móvil
        $movil = $emb['movil'] ?: 'Sin móvil';
        if (!isset($stats['analisis_por_movil'][$movil])) {
            $stats['analisis_por_movil'][$movil] = [
                'total_embarques' => 0,
                'tiempo_promedio_carga' => [],
                'tiempo_promedio_planta' => []
            ];
        }
        $stats['analisis_por_movil'][$movil]['total_embarques']++;
        if ($emb['tiempo_carga_min'] !== null) {
            $stats['analisis_por_movil'][$movil]['tiempo_promedio_carga'][] = (int)$emb['tiempo_carga_min'];
        }
        if ($emb['tiempo_total_planta_min'] !== null) {
            $stats['analisis_por_movil'][$movil]['tiempo_promedio_planta'][] = (int)$emb['tiempo_total_planta_min'];
        }
        
        // Análisis por depósito
        $deposito = $emb['deposito_code'] ?: 'Sin depósito';
        if (!isset($stats['analisis_por_deposito'][$deposito])) {
            $stats['analisis_por_deposito'][$deposito] = [
                'total_embarques' => 0,
                'tiempo_promedio_espera' => [],
                'tiempo_promedio_carga' => []
            ];
        }
        $stats['analisis_por_deposito'][$deposito]['total_embarques']++;
        if ($emb['tiempo_espera_min'] !== null) {
            $stats['analisis_por_deposito'][$deposito]['tiempo_promedio_espera'][] = (int)$emb['tiempo_espera_min'];
        }
        if ($emb['tiempo_carga_min'] !== null) {
            $stats['analisis_por_deposito'][$deposito]['tiempo_promedio_carga'][] = (int)$emb['tiempo_carga_min'];
        }
    }
    
    // Calculate summary statistics
    function calcStats($valores) {
        if (empty($valores)) return ['promedio' => null, 'minimo' => null, 'maximo' => null];
        return [
            'promedio' => round(array_sum($valores) / count($valores), 1),
            'minimo' => min($valores),
            'maximo' => max($valores)
        ];
    }
    
    $stats['tiempo_espera'] = array_merge($stats['tiempo_espera'], calcStats($stats['tiempo_espera']['valores']));
    $stats['tiempo_carga'] = array_merge($stats['tiempo_carga'], calcStats($stats['tiempo_carga']['valores']));
    $stats['tiempo_planta'] = array_merge($stats['tiempo_planta'], calcStats($stats['tiempo_planta']['valores']));
    $stats['tiempo_ruta'] = array_merge($stats['tiempo_ruta'], calcStats($stats['tiempo_ruta']['valores']));
    $stats['eficiencia_carga'] = array_merge($stats['eficiencia_carga'], calcStats($stats['eficiencia_carga']['valores']));
    
    // Process analysis by móvil and depósito
    foreach ($stats['analisis_por_movil'] as $movil => &$data) {
        $data['tiempo_promedio_carga'] = !empty($data['tiempo_promedio_carga']) && is_array($data['tiempo_promedio_carga']) 
            ? round(array_sum($data['tiempo_promedio_carga']) / count($data['tiempo_promedio_carga']), 1) : null;
        $data['tiempo_promedio_planta'] = !empty($data['tiempo_promedio_planta']) && is_array($data['tiempo_promedio_planta']) 
            ? round(array_sum($data['tiempo_promedio_planta']) / count($data['tiempo_promedio_planta']), 1) : null;
    }
    
    foreach ($stats['analisis_por_deposito'] as $deposito => &$data) {
        $data['tiempo_promedio_espera'] = !empty($data['tiempo_promedio_espera']) && is_array($data['tiempo_promedio_espera']) 
            ? round(array_sum($data['tiempo_promedio_espera']) / count($data['tiempo_promedio_espera']), 1) : null;
        $data['tiempo_promedio_carga'] = !empty($data['tiempo_promedio_carga']) && is_array($data['tiempo_promedio_carga']) 
            ? round(array_sum($data['tiempo_promedio_carga']) / count($data['tiempo_promedio_carga']), 1) : null;
    }
    
    // Remove raw values arrays for cleaner response
    unset($stats['tiempo_espera']['valores']);
    unset($stats['tiempo_carga']['valores']);
    unset($stats['tiempo_planta']['valores']);
    unset($stats['tiempo_ruta']['valores']);
    unset($stats['eficiencia_carga']['valores']);
    
    // Get filter options
    $estados = $pdo->query("
        SELECT code, nombre 
        FROM so_embarque_estado 
        WHERE activo = 1 
        ORDER BY orden
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $moviles = $pdo->query("
        SELECT id, chapa, 
               CONCAT(COALESCE(chapa, ''), ' (ID: ', id, ')') as label
        FROM para_moviles 
        WHERE activo = 1 
        ORDER BY chapa
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $choferes = $pdo->query("
        SELECT id, nombre 
        FROM para_choferes 
        WHERE activo = 1 
        ORDER BY nombre
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $depositos = $pdo->query("
        SELECT id, code, nombre,
               CONCAT(code, ' - ', nombre) as label
        FROM wh_deposito 
        ORDER BY code
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    $response = [
        'ok' => true,
        'data' => $embarques,
        'statistics' => $stats,
        'filters' => [
            'estados' => $estados,
            'moviles' => $moviles,
            'choferes' => $choferes,
            'depositos' => $depositos
        ]
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    error_log("Error in embarque_tiempos.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error interno del servidor'
    ], JSON_UNESCAPED_UNICODE);
}