<?php
// api/reportes/embarque_tiempos_pdf.php
declare(strict_types=1);

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = getPDO();
    
    // Get the same parameters as the main API
    $fechaDesde = $_GET['fecha_desde'] ?? null;
    $fechaHasta = $_GET['fecha_hasta'] ?? null;
    $estado = $_GET['estado'] ?? null;
    $movilId = isset($_GET['movil_id']) && $_GET['movil_id'] !== '' ? (int)$_GET['movil_id'] : null;
    $choferId = isset($_GET['chofer_id']) && $_GET['chofer_id'] !== '' ? (int)$_GET['chofer_id'] : null;
    $depositoId = isset($_GET['deposito_id']) && $_GET['deposito_id'] !== '' ? (int)$_GET['deposito_id'] : null;
    
    // Build the same query structure
    $whereConditions = [];
    $params = [];
    
    if ($fechaDesde) {
        $whereConditions[] = "DATE(e.creado_at) >= ?";
        $params[] = $fechaDesde;
    }
    
    if ($fechaHasta) {
        $whereConditions[] = "DATE(e.creado_at) <= ?";
        $params[] = $fechaHasta;
    }
    
    if ($estado) {
        $whereConditions[] = "est.code = ?";
        $params[] = $estado;
    }
    
    if ($movilId) {
        $whereConditions[] = "e.movil_id = ?";
        $params[] = $movilId;
    }
    
    if ($choferId) {
        $whereConditions[] = "e.chofer_id = ?";
        $params[] = $choferId;
    }
    
    if ($depositoId) {
        $whereConditions[] = "e.deposito_id = ?";
        $params[] = $depositoId;
    }
    
    $whereClause = $whereConditions ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
    
    $sql = "
        SELECT 
            e.codigo,
            DATE(e.creado_at) as fecha_embarque,
            est.nombre as estado,
            dep.code as deposito,
            COALESCE(m.chapa, '-') as movil,
            COALESCE(ch.nombre, '-') as chofer,
            e.ayudantes_cant,
            TIME(e.llegada_at) as llegada,
            TIME(e.carga_inicio_at) as inicio_carga,
            TIME(e.carga_fin_at) as fin_carga,
            TIME(e.salida_at) as salida,
            CASE 
                WHEN e.llegada_at IS NOT NULL AND e.carga_inicio_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.carga_inicio_at)
                ELSE NULL 
            END as tiempo_espera_min,
            CASE 
                WHEN e.carga_inicio_at IS NOT NULL AND e.carga_fin_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.carga_inicio_at, e.carga_fin_at)
                ELSE NULL 
            END as tiempo_carga_min,
            CASE 
                WHEN e.llegada_at IS NOT NULL AND e.salida_at IS NOT NULL 
                THEN TIMESTAMPDIFF(MINUTE, e.llegada_at, e.salida_at)
                ELSE NULL 
            END as tiempo_total_planta_min,
            (SELECT COUNT(*) FROM so_embarque_parada p WHERE p.embarque_id = e.id) as total_destinos
            
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        LEFT JOIN para_choferes ch ON ch.id = e.chofer_id
        {$whereClause}
        ORDER BY e.creado_at DESC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate summary statistics
    $totalEmbarques = count($embarques);
    $tiemposEspera = array_filter(array_column($embarques, 'tiempo_espera_min'));
    $tiemposCarga = array_filter(array_column($embarques, 'tiempo_carga_min'));
    $tiemposPlanta = array_filter(array_column($embarques, 'tiempo_total_planta_min'));
    
    $statsEspera = $tiemposEspera ? [
        'promedio' => round(array_sum($tiemposEspera) / count($tiemposEspera), 1),
        'minimo' => min($tiemposEspera),
        'maximo' => max($tiemposEspera)
    ] : ['promedio' => null, 'minimo' => null, 'maximo' => null];
    
    $statsCarga = $tiemposCarga ? [
        'promedio' => round(array_sum($tiemposCarga) / count($tiemposCarga), 1),
        'minimo' => min($tiemposCarga),
        'maximo' => max($tiemposCarga)
    ] : ['promedio' => null, 'minimo' => null, 'maximo' => null];
    
    $statsPlanta = $tiemposPlanta ? [
        'promedio' => round(array_sum($tiemposPlanta) / count($tiemposPlanta), 1),
        'minimo' => min($tiemposPlanta),
        'maximo' => max($tiemposPlanta)
    ] : ['promedio' => null, 'minimo' => null, 'maximo' => null];
    
    // Format filters for display
    $filtrosTexto = [];
    if ($fechaDesde) $filtrosTexto[] = "Desde: " . date('d/m/Y', strtotime($fechaDesde));
    if ($fechaHasta) $filtrosTexto[] = "Hasta: " . date('d/m/Y', strtotime($fechaHasta));
    if ($estado) $filtrosTexto[] = "Estado: $estado";
    if ($movilId) {
        $movil = $pdo->prepare("SELECT chapa FROM para_moviles WHERE id = ?")->execute([$movilId]);
        $movilNombre = $pdo->fetchColumn() ?: "ID $movilId";
        $filtrosTexto[] = "Móvil: $movilNombre";
    }
    if ($depositoId) {
        $deposito = $pdo->prepare("SELECT code FROM wh_deposito WHERE id = ?")->execute([$depositoId]);
        $depositoNombre = $pdo->fetchColumn() ?: "ID $depositoId";
        $filtrosTexto[] = "Depósito: $depositoNombre";
    }
    
    function formatMinutos($mins) {
        if (!$mins) return '-';
        if ($mins >= 60) {
            $h = intval($mins / 60);
            $m = $mins % 60;
            return "{$h}h {$m}m";
        }
        return "{$mins}m";
    }
    
} catch (Exception $e) {
    error_log("Error in embarque_tiempos_pdf.php: " . $e->getMessage());
    die("Error al generar el reporte");
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reporte de Análisis de Tiempos de Embarques</title>
    <style>
        @page { margin: 20mm; }
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; font-size: 9px; line-height: 1.3; color: #333; }
        .header { text-align: center; border-bottom: 2px solid #333; padding-bottom: 10px; margin-bottom: 15px; }
        .header h1 { font-size: 16px; margin-bottom: 5px; }
        .header p { font-size: 10px; color: #666; }
        .summary { margin-bottom: 15px; }
        .stats-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 10px; margin-bottom: 15px; }
        .stat-card { padding: 8px; background: #f5f5f5; border-radius: 3px; text-align: center; }
        .stat-card h4 { font-size: 10px; margin-bottom: 5px; color: #2c5aa0; }
        .stat-values { display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 5px; text-align: center; }
        .stat-values > div { font-size: 8px; }
        .stat-values .value { font-weight: bold; color: #333; }
        .stat-values .label { color: #666; }
        .filters { margin-bottom: 15px; padding: 8px; background: #f9f9f9; border-radius: 3px; }
        .filters strong { color: #333; }
        table { width: 100%; border-collapse: collapse; margin-top: 10px; font-size: 7px; }
        th { background: #2c5aa0; color: white; padding: 5px 3px; text-align: left; font-weight: bold; }
        td { padding: 3px; border-bottom: 1px solid #ddd; }
        tr:nth-child(even) { background: #f9f9f9; }
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .badge { padding: 2px 4px; border-radius: 3px; color: white; font-size: 6px; font-weight: bold; }
        .badge-success { background: #28a745; }
        .badge-warning { background: #ffc107; color: #333; }
        .badge-danger { background: #dc3545; }
        .badge-info { background: #17a2b8; }
        .badge-secondary { background: #6c757d; }
        .badge-primary { background: #007bff; }
        .badge-dark { background: #343a40; }
        .footer { margin-top: 20px; text-align: center; font-size: 8px; color: #666; border-top: 1px solid #ddd; padding-top: 10px; }
        .no-break { page-break-inside: avoid; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Reporte de Análisis de Tiempos de Embarques</h1>
        <p>Generado el <?= date('d/m/Y H:i:s') ?></p>
    </div>

    <?php if ($filtrosTexto): ?>
    <div class="filters">
        <strong>Filtros aplicados:</strong> <?= implode(' | ', $filtrosTexto) ?>
    </div>
    <?php endif; ?>

    <div class="summary no-break">
        <h3 style="margin-bottom: 10px;">Estadísticas de Tiempos</h3>
        <div class="stats-grid">
            <div class="stat-card">
                <h4>Tiempo de Espera</h4>
                <div class="stat-values">
                    <div>
                        <div class="label">Promedio</div>
                        <div class="value"><?= formatMinutos($statsEspera['promedio']) ?></div>
                    </div>
                    <div>
                        <div class="label">Mínimo</div>
                        <div class="value"><?= formatMinutos($statsEspera['minimo']) ?></div>
                    </div>
                    <div>
                        <div class="label">Máximo</div>
                        <div class="value"><?= formatMinutos($statsEspera['maximo']) ?></div>
                    </div>
                </div>
            </div>
            <div class="stat-card">
                <h4>Tiempo de Carga</h4>
                <div class="stat-values">
                    <div>
                        <div class="label">Promedio</div>
                        <div class="value"><?= formatMinutos($statsCarga['promedio']) ?></div>
                    </div>
                    <div>
                        <div class="label">Mínimo</div>
                        <div class="value"><?= formatMinutos($statsCarga['minimo']) ?></div>
                    </div>
                    <div>
                        <div class="label">Máximo</div>
                        <div class="value"><?= formatMinutos($statsCarga['maximo']) ?></div>
                    </div>
                </div>
            </div>
            <div class="stat-card">
                <h4>Total en Planta</h4>
                <div class="stat-values">
                    <div>
                        <div class="label">Promedio</div>
                        <div class="value"><?= formatMinutos($statsPlanta['promedio']) ?></div>
                    </div>
                    <div>
                        <div class="label">Mínimo</div>
                        <div class="value"><?= formatMinutos($statsPlanta['minimo']) ?></div>
                    </div>
                    <div>
                        <div class="label">Máximo</div>
                        <div class="value"><?= formatMinutos($statsPlanta['maximo']) ?></div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <h3 style="margin-bottom: 10px;">Detalle de Embarques</h3>
    <table>
        <thead>
            <tr>
                <th style="width: 12%;">Código</th>
                <th style="width: 8%;">Fecha</th>
                <th style="width: 10%;">Estado</th>
                <th style="width: 8%;">Móvil</th>
                <th style="width: 8%;">Dep.</th>
                <th style="width: 6%;">Dest.</th>
                <th style="width: 6%;">Ayud.</th>
                <th style="width: 6%;">Lleg.</th>
                <th style="width: 6%;">I.Carga</th>
                <th style="width: 6%;">F.Carga</th>
                <th style="width: 6%;">Salida</th>
                <th style="width: 6%;">T.Esp.</th>
                <th style="width: 6%;">T.Carga</th>
                <th style="width: 6%;">T.Planta</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($embarques as $emb): ?>
            <tr>
                <td><?= htmlspecialchars($emb['codigo']) ?></td>
                <td class="text-center"><?= $emb['fecha_embarque'] ? date('d/m/Y', strtotime($emb['fecha_embarque'])) : '-' ?></td>
                <td>
                    <?php
                    $estadoClass = 'badge-secondary';
                    switch($emb['estado']) {
                        case 'EN_COLA': $estadoClass = 'badge-secondary'; break;
                        case 'EN_CARGA': $estadoClass = 'badge-warning'; break;
                        case 'CARGADO': $estadoClass = 'badge-primary'; break;
                        case 'EN_RUTA': $estadoClass = 'badge-info'; break;
                        case 'ENTREGADO': $estadoClass = 'badge-success'; break;
                        case 'FINALIZADO': $estadoClass = 'badge-dark'; break;
                        case 'CANCELADO': $estadoClass = 'badge-danger'; break;
                    }
                    ?>
                    <span class="badge <?= $estadoClass ?>"><?= htmlspecialchars($emb['estado']) ?></span>
                </td>
                <td><?= htmlspecialchars($emb['movil']) ?></td>
                <td><?= htmlspecialchars($emb['deposito']) ?></td>
                <td class="text-center"><?= $emb['total_destinos'] ?></td>
                <td class="text-center"><?= $emb['ayudantes_cant'] ?: '-' ?></td>
                <td class="text-center"><?= $emb['llegada'] ?: '-' ?></td>
                <td class="text-center"><?= $emb['inicio_carga'] ?: '-' ?></td>
                <td class="text-center"><?= $emb['fin_carga'] ?: '-' ?></td>
                <td class="text-center"><?= $emb['salida'] ?: '-' ?></td>
                <td class="text-center"><?= formatMinutos($emb['tiempo_espera_min']) ?></td>
                <td class="text-center"><?= formatMinutos($emb['tiempo_carga_min']) ?></td>
                <td class="text-center"><?= formatMinutos($emb['tiempo_total_planta_min']) ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <?php if (empty($embarques)): ?>
    <div style="text-align: center; margin: 30px 0; color: #666;">
        <p>No se encontraron embarques con los filtros aplicados.</p>
    </div>
    <?php endif; ?>

    <div class="footer">
        <p>Sistema de Operaciones Logísticas (SOL) | Reporte de análisis de tiempos generado automáticamente</p>
    </div>

    <script>
        // Auto-print and close
        window.onload = function() {
            window.print();
            setTimeout(function() {
                window.close();
            }, 1000);
        };
    </script>
</body>
</html>