<?php
/**
 * SOL · API · Reporte de Devoluciones y Retornos
 * Ruta: api/reportes/embarques_devoluciones.php
 */
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

const REPORT_MAX_ROWS = 1000;

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'No autorizado'], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $filters = buildFilters($_GET ?? []);

    if (hasSimulatedDevoluciones($pdo)) {
        $rawRows = fetchSimulatedRows($pdo, $filters);
    } else {
        $rawRows = fetchLegacyRows($pdo, $filters);
    }

    $rows = array_map('formatRow', $rawRows);
    $summary = buildSummary($rows, $filters);
    $aggregates = [
        'motivos' => aggregateBy($rows, static function (array $row): array {
            return [
                'key' => (string) ($row['motivo_id'] ?? 0),
                'label' => $row['motivo_nombre'] ?? 'Sin motivo',
                'code' => $row['motivo_code'] ?? null,
                'cost' => $row['costo_total_gs'] ?? 0,
            ];
        }),
        'destinatarios' => aggregateBy($rows, static function (array $row): array {
            $destId = $row['destinatario_id'] ?? null;
            return [
                'key' => $destId !== null ? (string) $destId : '0',
                'label' => $row['destinatario_label'] ?? 'Sin destinatario',
                'cliente' => $row['cliente_nombre'] ?? null,
                'cost' => $row['costo_total_gs'] ?? 0,
            ];
        }),
        'productos' => aggregateBy($rows, static function (array $row): array {
            $prodId = $row['producto_id'] ?? null;
            return [
                'key' => $prodId !== null ? (string) $prodId : '0',
                'label' => $row['producto_label'] ?? 'Sin producto',
                'sku' => $row['producto_sku'] ?? null,
                'cost' => $row['costo_total_gs'] ?? 0,
            ];
        }, 12),
    ];

    $filterOptions = loadFilterOptions($pdo);

    echo json_encode([
        'ok' => true,
        'filters' => $filters,
        'data' => $rows,
        'summary' => $summary,
        'aggregates' => $aggregates,
        'filter_options' => $filterOptions,
        'limits' => [
            'max_rows' => REPORT_MAX_ROWS,
            'returned_rows' => count($rows),
        ],
    ], JSON_UNESCAPED_UNICODE);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error de base de datos',
        'message' => $e->getMessage(),
    ], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * @param array<string, mixed> $input
 * @return array<string, mixed>
 */
function buildFilters(array $input): array
{
    $defaultDesde = date('Y-m-d', strtotime('-30 days'));
    $defaultHasta = date('Y-m-d');

    $fechaDesde = sanitizeDate($input['fecha_desde'] ?? null, $defaultDesde);
    $fechaHasta = sanitizeDate($input['fecha_hasta'] ?? null, $defaultHasta);

    if ($fechaDesde > $fechaHasta) {
        [$fechaDesde, $fechaHasta] = [$fechaHasta, $fechaDesde];
    }

    return [
        'fecha_desde' => $fechaDesde,
        'fecha_hasta' => $fechaHasta,
        'deposito_id' => sanitizeNumeric($input['deposito_id'] ?? null),
        'motivo_id' => sanitizeNumeric($input['motivo_id'] ?? null),
        'cliente_id' => sanitizeNumeric($input['cliente_id'] ?? null),
        'destinatario_id' => sanitizeNumeric($input['destinatario_id'] ?? null),
        'movil_id' => sanitizeNumeric($input['movil_id'] ?? null),
        'sku' => sanitizeString($input['sku'] ?? '', 80),
        'producto_text' => sanitizeString($input['producto_text'] ?? '', 120),
        'embarque_codigo' => sanitizeString($input['embarque_codigo'] ?? '', 80),
        'solo_uc' => !empty($input['solo_uc']),
    ];
}

function sanitizeDate(?string $value, string $fallback): string
{
    $value = trim((string) $value);
    if (preg_match('/^20\\d{2}-[01]\\d-[0-3]\\d$/', $value)) {
        return $value;
    }
    return $fallback;
}

function sanitizeNumeric($value): string
{
    if ($value === null || $value === '') {
        return '';
    }
    if (!is_numeric($value)) {
        return '';
    }
    $intVal = (int) $value;
    return $intVal > 0 ? (string) $intVal : '';
}

function sanitizeString(?string $value, int $maxLength): string
{
    $value = trim((string) $value);
    if ($value === '') {
        return '';
    }
    if (strlen($value) > $maxLength) {
        $value = substr($value, 0, $maxLength);
    }
    return $value;
}

/**
 * @return array<string, array<int, array<string, mixed>>>
 */
function loadFilterOptions(PDO $pdo): array
{
    $depositos = $pdo->query('SELECT id, code, nombre FROM wh_deposito ORDER BY nombre')
        ->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $motivos = $pdo->query('SELECT id, code, nombre FROM para_devolucion_motivo WHERE activo = 1 ORDER BY nombre')
        ->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $clientes = $pdo->query('SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social LIMIT 200')
        ->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $destinatarios = $pdo->query('SELECT id, cod, nombre, cliente_id FROM para_destinatarios WHERE deleted_at IS NULL ORDER BY nombre LIMIT 300')
        ->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $moviles = $pdo->query('SELECT id, chapa FROM para_moviles WHERE activo = 1 ORDER BY chapa')
        ->fetchAll(PDO::FETCH_ASSOC) ?: [];

    return [
        'depositos' => array_map(static function (array $row): array {
            return [
                'id' => (int) $row['id'],
                'nombre' => trim((string) ($row['nombre'] ?? $row['code'] ?? '')) ?: 'Sin nombre',
                'code' => $row['code'] ?? null,
            ];
        }, $depositos),
        'motivos' => array_map(static function (array $row): array {
            return [
                'id' => (int) $row['id'],
                'code' => $row['code'] ?? null,
                'nombre' => $row['nombre'] ?? 'Sin motivo',
            ];
        }, $motivos),
        'clientes' => array_map(static function (array $row): array {
            return [
                'id' => (int) $row['id'],
                'nombre' => $row['razon_social'] ?? 'Sin cliente',
            ];
        }, $clientes),
        'destinatarios' => array_map(static function (array $row): array {
            $codigo = trim((string) ($row['cod'] ?? ''));
            $nombre = trim((string) ($row['nombre'] ?? ''));
            $label = $nombre !== '' ? $nombre : 'Sin destinatario';
            if ($codigo !== '') {
                $label = $codigo . ' · ' . $label;
            }
            return [
                'id' => (int) $row['id'],
                'cliente_id' => isset($row['cliente_id']) ? (int) $row['cliente_id'] : null,
                'label' => $label,
            ];
        }, $destinatarios),
        'moviles' => array_map(static function (array $row): array {
            return [
                'id' => (int) $row['id'],
                'label' => $row['chapa'] ?? ('ID ' . (int) $row['id']),
            ];
        }, $moviles),
    ];
}

function hasSimulatedDevoluciones(PDO $pdo): bool
{
    static $cache = null;
    if ($cache !== null) {
        return $cache;
    }
    try {
        $stmt = $pdo->query('SELECT 1 FROM so_embarque_devoluciones LIMIT 1');
        $cache = $stmt !== false && $stmt->fetchColumn() !== false;
    } catch (Throwable $e) {
        $cache = false;
    }
    return $cache;
}

/**
 * @param array<string, mixed> $filters
 * @return array<int, array<string, mixed>>
 */
function fetchLegacyRows(PDO $pdo, array $filters): array
{
    $where = [];
    $params = [];

    $where[] = 'DATE(COALESCE(r.llegada_at, d.created_at)) >= :fecha_desde';
    $where[] = 'DATE(COALESCE(r.llegada_at, d.created_at)) <= :fecha_hasta';
    $params[':fecha_desde'] = $filters['fecha_desde'];
    $params[':fecha_hasta'] = $filters['fecha_hasta'];

    if ($filters['deposito_id'] !== '') {
        $where[] = 'e.deposito_id = :deposito_id';
        $params[':deposito_id'] = (int) $filters['deposito_id'];
    }

    if ($filters['motivo_id'] !== '') {
        $where[] = 'd.motivo_id = :motivo_id';
        $params[':motivo_id'] = (int) $filters['motivo_id'];
    }

    if ($filters['cliente_id'] !== '') {
        $where[] = 'dest.cliente_id = :cliente_id';
        $params[':cliente_id'] = (int) $filters['cliente_id'];
    }

    if ($filters['destinatario_id'] !== '') {
        $where[] = 'd.destinatario_id = :destinatario_id';
        $params[':destinatario_id'] = (int) $filters['destinatario_id'];
    }

    if ($filters['movil_id'] !== '') {
        $where[] = 'e.movil_id = :movil_id';
        $params[':movil_id'] = (int) $filters['movil_id'];
    }

    if ($filters['sku'] !== '') {
        $where[] = 'prod.sku LIKE :sku';
        $params[':sku'] = '%' . $filters['sku'] . '%';
    }

    if ($filters['producto_text'] !== '') {
        $where[] = 'prod.denominacion LIKE :producto_text';
        $params[':producto_text'] = '%' . $filters['producto_text'] . '%';
    }

    if ($filters['embarque_codigo'] !== '') {
        $where[] = 'e.codigo LIKE :embarque_codigo';
        $params[':embarque_codigo'] = '%' . $filters['embarque_codigo'] . '%';
    }

    if ($filters['solo_uc']) {
        $where[] = 'd.qty_uc > 0';
    }

    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';
    $sql = "
        SELECT
            d.id AS devolucion_id,
            d.retorno_id,
            COALESCE(r.llegada_at, d.created_at) AS retorno_at,
            r.llegada_at,
            r.observacion AS retorno_observacion,
            r.created_at AS retorno_creado,
            e.id AS embarque_id,
            e.codigo AS embarque_codigo,
            e.observacion AS embarque_observacion,
            e.deposito_id,
            dep.nombre AS deposito_nombre,
            dep.code AS deposito_code,
            e.movil_id,
            mov.chapa AS movil_chapa,
            e.chofer_id,
            cho.nombre AS chofer_nombre,
            dest.id AS destinatario_id,
            dest.cod AS destinatario_codigo,
            dest.nombre AS destinatario_nombre,
            cli.id AS cliente_id,
            cli.razon_social AS cliente_nombre,
            d.doc_tipo,
            d.doc_numero,
            d.motivo_id,
            mot.code AS motivo_code,
            mot.nombre AS motivo_nombre,
            d.producto_id,
            prod.sku AS producto_sku,
            prod.denominacion AS producto_nombre,
            prod.precio_unit_efectivo_gs AS producto_precio_unitario,
            d.lote_id,
            lote.codigo AS lote_codigo,
            d.qty_pallets,
            d.qty_uv,
            d.qty_uc,
            d.responsable_user_id,
            usr.full_name AS responsable_nombre,
            d.created_at AS devolucion_creada
        FROM so_devolucion d
        JOIN so_retorno r ON r.id = d.retorno_id
        JOIN so_embarque e ON e.id = r.embarque_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_destinatarios dest ON dest.id = d.destinatario_id
        LEFT JOIN para_clientes cli ON cli.id = dest.cliente_id
        LEFT JOIN para_devolucion_motivo mot ON mot.id = d.motivo_id
        LEFT JOIN para_productos prod ON prod.id = d.producto_id
        LEFT JOIN wh_lote lote ON lote.id = d.lote_id
        LEFT JOIN para_moviles mov ON mov.id = e.movil_id
        LEFT JOIN para_choferes cho ON cho.id = e.chofer_id
        LEFT JOIN sys_users usr ON usr.id = d.responsable_user_id
        {$whereSql}
        ORDER BY COALESCE(r.llegada_at, d.created_at) DESC, d.id DESC
        LIMIT " . REPORT_MAX_ROWS . "
    ";

    $stmt = $pdo->prepare($sql);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();

    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

/**
 * @param array<string, mixed> $filters
 * @return array<int, array<string, mixed>>
 */
function fetchSimulatedRows(PDO $pdo, array $filters): array
{
    $dateExpr = 'COALESCE(r.llegada_at, e.llegada_at, e.salida_at, e.updated_at, e.creado_at, d.created_at)';

    $where = [];
    $params = [];

    $where[] = "DATE({$dateExpr}) >= :fecha_desde";
    $where[] = "DATE({$dateExpr}) <= :fecha_hasta";
    $params[':fecha_desde'] = $filters['fecha_desde'];
    $params[':fecha_hasta'] = $filters['fecha_hasta'];

    if ($filters['deposito_id'] !== '') {
        $where[] = 'e.deposito_id = :deposito_id';
        $params[':deposito_id'] = (int) $filters['deposito_id'];
    }

    if ($filters['motivo_id'] !== '') {
        $where[] = 'mot.id = :motivo_id';
        $params[':motivo_id'] = (int) $filters['motivo_id'];
    }

    if ($filters['cliente_id'] !== '') {
        $where[] = 'map.cliente_id = :cliente_id';
        $params[':cliente_id'] = (int) $filters['cliente_id'];
    }

    if ($filters['destinatario_id'] !== '') {
        $where[] = 'map.destinatario_id = :destinatario_id';
        $params[':destinatario_id'] = (int) $filters['destinatario_id'];
    }

    if ($filters['movil_id'] !== '') {
        $where[] = 'e.movil_id = :movil_id';
        $params[':movil_id'] = (int) $filters['movil_id'];
    }

    if ($filters['sku'] !== '') {
        $where[] = 'UPPER(d.sku) LIKE :sku';
        $params[':sku'] = '%' . strtoupper($filters['sku']) . '%';
    }

    if ($filters['producto_text'] !== '') {
        $where[] = 'prod.denominacion LIKE :producto_text';
        $params[':producto_text'] = '%' . $filters['producto_text'] . '%';
    }

    if ($filters['embarque_codigo'] !== '') {
        $where[] = 'e.codigo LIKE :embarque_codigo';
        $params[':embarque_codigo'] = '%' . $filters['embarque_codigo'] . '%';
    }

    if ($filters['solo_uc']) {
        $where[] = 'd.unidades_sueltas > 0';
    }

    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

    $destMapSql = "
        SELECT
            e.id AS embarque_id,
            UPPER(pr.sku) AS sku,
            MIN(dest.id) AS destinatario_id,
            MIN(dest.cod) AS destinatario_codigo,
            MIN(dest.nombre) AS destinatario_nombre,
            MIN(cli.id) AS cliente_id,
            MIN(cli.razon_social) AS cliente_nombre
        FROM so_embarque e
        JOIN so_embarque_pre ep ON ep.embarque_id = e.id
        JOIN so_preembarque pre ON pre.id = ep.preembarque_id
        JOIN so_pedido ped ON ped.id = pre.pedido_id
        JOIN so_pedido_dest pd ON pd.pedido_id = ped.id
        JOIN para_destinatarios dest ON dest.id = pd.destinatario_id
        JOIN para_clientes cli ON cli.id = dest.cliente_id
        JOIN so_pedido_dest_item di ON di.pedido_dest_id = pd.id
        JOIN para_productos pr ON pr.id = di.producto_id
        GROUP BY e.id, UPPER(pr.sku)
    ";

    $sql = "
        SELECT
            d.id AS devolucion_id,
            r.id AS retorno_id,
            {$dateExpr} AS retorno_at,
            r.llegada_at,
            r.observacion AS retorno_observacion,
            r.created_at AS retorno_creado,
            e.id AS embarque_id,
            e.codigo AS embarque_codigo,
            e.observacion AS embarque_observacion,
            e.deposito_id,
            dep.nombre AS deposito_nombre,
            dep.code AS deposito_code,
            e.movil_id,
            mov.chapa AS movil_chapa,
            e.chofer_id,
            cho.nombre AS chofer_nombre,
            map.destinatario_id,
            map.destinatario_codigo,
            map.destinatario_nombre,
            map.cliente_id,
            map.cliente_nombre,
            NULL AS doc_tipo,
            NULL AS doc_numero,
            mot.id AS motivo_id,
            mot.code AS motivo_code,
            COALESCE(mot.nombre, d.motivo, 'Sin motivo') AS motivo_nombre,
            prod.id AS producto_id,
            UPPER(d.sku) AS producto_sku,
            COALESCE(prod.denominacion, CONCAT('SKU ', UPPER(d.sku))) AS producto_nombre,
            prod.precio_unit_efectivo_gs AS producto_precio_unitario,
            NULL AS lote_id,
            d.lote AS lote_codigo,
            d.pallets AS qty_pallets,
            d.cajas_sueltas AS qty_uv,
            d.unidades_sueltas AS qty_uc,
            NULL AS responsable_user_id,
            NULL AS responsable_nombre,
            d.created_at AS devolucion_creada
        FROM so_embarque_devoluciones d
        JOIN so_embarque e ON e.id = d.embarque_id
        LEFT JOIN (
            SELECT sr.*
            FROM so_retorno sr
            JOIN (
                SELECT embarque_id, MAX(llegada_at) AS max_llegada
                FROM so_retorno
                GROUP BY embarque_id
            ) latest ON latest.embarque_id = sr.embarque_id AND latest.max_llegada = sr.llegada_at
        ) r ON r.embarque_id = d.embarque_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_moviles mov ON mov.id = e.movil_id
        LEFT JOIN para_choferes cho ON cho.id = e.chofer_id
    LEFT JOIN ({$destMapSql}) map ON map.embarque_id = d.embarque_id AND BINARY map.sku = BINARY UPPER(d.sku)
    LEFT JOIN para_productos prod ON BINARY UPPER(prod.sku) = BINARY UPPER(d.sku)
        LEFT JOIN para_devolucion_motivo mot ON (
            CONVERT(mot.nombre USING utf8mb4) COLLATE utf8mb4_unicode_ci = CONVERT(d.motivo USING utf8mb4) COLLATE utf8mb4_unicode_ci
            OR CONVERT(mot.code USING utf8mb4) COLLATE utf8mb4_unicode_ci = CONVERT(d.motivo USING utf8mb4) COLLATE utf8mb4_unicode_ci
        )
        {$whereSql}
        ORDER BY {$dateExpr} DESC, d.id DESC
        LIMIT " . REPORT_MAX_ROWS . "
    ";

    $stmt = $pdo->prepare($sql);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();

    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

/**
 * @param array<string, scalar|null> $row
 * @return array<string, mixed>
 */
function formatRow(array $row): array
{
    $retornoAt = $row['retorno_at'] ?? $row['devolucion_creada'] ?? null;
    $retornoAt = $retornoAt !== null ? (string) $retornoAt : null;
    $retornoFecha = $retornoAt ? substr($retornoAt, 0, 10) : null;
    $retornoHora = $retornoAt && strlen($retornoAt) >= 16 ? substr($retornoAt, 11, 5) : null;

    $qtyUc = (int) ($row['qty_uc'] ?? 0);
    $precioUnitario = null;
    if (isset($row['producto_precio_unitario']) && $row['producto_precio_unitario'] !== '' && $row['producto_precio_unitario'] !== null) {
        $precioUnitario = (float) $row['producto_precio_unitario'];
    }
    $costoTotal = $precioUnitario !== null ? $precioUnitario * $qtyUc : 0.0;

    $destCodigo = trim((string) ($row['destinatario_codigo'] ?? ''));
    $destNombre = trim((string) ($row['destinatario_nombre'] ?? ''));
    $destLabel = $destNombre !== '' ? $destNombre : 'Sin destinatario';
    if ($destCodigo !== '') {
        $destLabel = $destCodigo . ' · ' . $destLabel;
    }

    $productoSku = trim((string) ($row['producto_sku'] ?? ''));
    $productoNombre = trim((string) ($row['producto_nombre'] ?? ''));
    $productoLabel = $productoNombre !== '' ? $productoNombre : 'Sin producto';
    if ($productoSku !== '') {
        $productoLabel = $productoSku . ($productoNombre !== '' ? ' · ' . $productoNombre : '');
    }

    $docTipo = trim((string) ($row['doc_tipo'] ?? ''));
    $docNumero = trim((string) ($row['doc_numero'] ?? ''));
    $docRef = '-';
    if ($docTipo !== '' && $docNumero !== '') {
        $docRef = $docTipo . ' ' . $docNumero;
    } elseif ($docNumero !== '') {
        $docRef = $docNumero;
    } elseif ($docTipo !== '') {
        $docRef = $docTipo;
    }

    $responsableNombre = trim((string) ($row['responsable_nombre'] ?? ''));
    if ($responsableNombre === '') {
        $responsableNombre = '-';
    }

    $observacion = trim((string) ($row['retorno_observacion'] ?? ''));
    if ($observacion === '' && !empty($row['embarque_observacion'])) {
        $observacion = trim((string) $row['embarque_observacion']);
    }
    if ($observacion === '') {
        $observacion = '-';
    }

    return [
        'devolucion_id' => (int) $row['devolucion_id'],
        'retorno_id' => (int) $row['retorno_id'],
        'embarque_id' => (int) $row['embarque_id'],
        'retorno_at' => $retornoAt,
        'retorno_fecha' => $retornoFecha,
        'retorno_hora' => $retornoHora,
        'retorno_observacion' => $observacion,
        'embarque_codigo' => $row['embarque_codigo'] ?? '',
        'deposito_id' => isset($row['deposito_id']) ? (int) $row['deposito_id'] : null,
        'deposito_code' => $row['deposito_code'] ?? null,
        'deposito_nombre' => $row['deposito_nombre'] ?? ($row['deposito_code'] ?? '-'),
        'cliente_id' => isset($row['cliente_id']) ? (int) $row['cliente_id'] : null,
        'cliente_nombre' => $row['cliente_nombre'] ?? '-',
        'destinatario_id' => isset($row['destinatario_id']) ? (int) $row['destinatario_id'] : null,
        'destinatario_label' => $destLabel,
        'motivo_id' => isset($row['motivo_id']) ? (int) $row['motivo_id'] : null,
        'motivo_code' => $row['motivo_code'] ?? null,
        'motivo_nombre' => $row['motivo_nombre'] ?? 'Sin motivo',
        'producto_id' => isset($row['producto_id']) ? (int) $row['producto_id'] : null,
        'producto_sku' => $productoSku,
        'producto_nombre' => $productoNombre,
        'producto_label' => $productoLabel,
        'lote_codigo' => $row['lote_codigo'] ?? '-',
        'qty_pallets' => (int) ($row['qty_pallets'] ?? 0),
        'qty_uv' => (int) ($row['qty_uv'] ?? 0),
    'qty_uc' => $qtyUc,
    'producto_precio_unitario_gs' => $precioUnitario,
    'costo_total_gs' => $costoTotal,
        'doc_tipo' => $docTipo,
        'doc_numero' => $docNumero,
        'doc_referencia' => $docRef,
        'responsable_id' => isset($row['responsable_user_id']) ? (int) $row['responsable_user_id'] : null,
        'responsable_nombre' => $responsableNombre,
        'movil_id' => isset($row['movil_id']) ? (int) $row['movil_id'] : null,
        'movil_chapa' => $row['movil_chapa'] ?? null,
        'chofer_nombre' => $row['chofer_nombre'] ?? null,
        'created_at' => $row['devolucion_creada'] ?? null,
        'embarque_observacion' => $row['embarque_observacion'] ?? null,
    ];
}

/**
 * @param array<int, array<string, mixed>> $rows
 * @return array<string, mixed>
 */
function buildSummary(array $rows, array $filters): array
{
    if (!$rows) {
        return [
            'total_registros' => 0,
            'total_embarques' => 0,
            'total_destinatarios' => 0,
            'total_clientes' => 0,
            'total_moviles' => 0,
            'total_pallets' => 0,
            'total_uv' => 0,
            'total_uc' => 0,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }

    $embarques = [];
    $destinatarios = [];
    $clientes = [];
    $moviles = [];
    $pallets = 0;
    $uv = 0;
    $uc = 0;

    foreach ($rows as $row) {
        if (!empty($row['embarque_id'])) {
            $embarques[(int) $row['embarque_id']] = true;
        }
        if (!empty($row['destinatario_id'])) {
            $destinatarios[(int) $row['destinatario_id']] = true;
        }
        if (!empty($row['cliente_id'])) {
            $clientes[(int) $row['cliente_id']] = true;
        }
        if (!empty($row['movil_id'])) {
            $moviles[(int) $row['movil_id']] = true;
        }
        $pallets += (int) ($row['qty_pallets'] ?? 0);
        $uv += (int) ($row['qty_uv'] ?? 0);
        $uc += (int) ($row['qty_uc'] ?? 0);
    }

    return [
        'total_registros' => count($rows),
        'total_embarques' => count($embarques),
        'total_destinatarios' => count($destinatarios),
        'total_clientes' => count($clientes),
        'total_moviles' => count($moviles),
        'total_pallets' => $pallets,
        'total_uv' => $uv,
        'total_uc' => $uc,
        'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
    ];
}

/**
 * @param array<int, array<string, mixed>> $rows
 * @param callable $selector
 * @return array<int, array<string, mixed>>
 */
function aggregateBy(array $rows, callable $selector, int $limit = 10): array
{
    $buckets = [];

    foreach ($rows as $row) {
        $info = $selector($row);
        if (!is_array($info) || empty($info['key'])) {
            continue;
        }
        $key = (string) $info['key'];
        if (!isset($buckets[$key])) {
            $bucket = [
                'label' => $info['label'] ?? $key,
                'rows' => 0,
                'pallets' => 0,
                'uv' => 0,
                'uc' => 0,
                'cost' => 0.0,
            ];
            foreach ($info as $infoKey => $value) {
                if ($infoKey === 'key' || $infoKey === 'label' || $infoKey === 'cost') {
                    continue;
                }
                $bucket[$infoKey] = $value;
            }
            $buckets[$key] = $bucket;
        }
        $buckets[$key]['rows']++;
        $buckets[$key]['pallets'] += (int) ($row['qty_pallets'] ?? 0);
        $buckets[$key]['uv'] += (int) ($row['qty_uv'] ?? 0);
        $buckets[$key]['uc'] += (int) ($row['qty_uc'] ?? 0);
        if (isset($info['cost'])) {
            $buckets[$key]['cost'] += (float) $info['cost'];
        } elseif (isset($row['costo_total_gs'])) {
            $buckets[$key]['cost'] += (float) $row['costo_total_gs'];
        }
    }

    if (!$buckets) {
        return [];
    }

    $list = array_values($buckets);
    usort($list, static function (array $a, array $b): int {
        $cmp = $b['uc'] <=> $a['uc'];
        if ($cmp !== 0) {
            return $cmp;
        }
        $cmp = $b['uv'] <=> $a['uv'];
        if ($cmp !== 0) {
            return $cmp;
        }
        $cmp = $b['rows'] <=> $a['rows'];
        if ($cmp !== 0) {
            return $cmp;
        }
        return strcmp((string) $a['label'], (string) $b['label']);
    });

    $list = array_slice($list, 0, $limit);

    return array_map(static function (array $row): array {
        return array_filter(
            $row,
            static function ($value) {
                return $value !== null && $value !== '';
            }
        );
    }, $list);
}
