<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

// Seguridad mínima (protegido por sesión en tu router)
if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']);
  exit;
}

try {
  $pdo = get_pdo(); // de config/db.php

  // Filtros
  $clienteId       = trim($_GET['cliente_id'] ?? '');
  $operativaId     = trim($_GET['operativa_id'] ?? '');
  $qProducto       = trim($_GET['q_producto'] ?? '');
  $lote            = trim($_GET['lote'] ?? '');
  $bucketSel       = trim($_GET['bucket'] ?? '30'); // 30|60|90|all
  $incluirVencidos = (($_GET['incluir_vencidos'] ?? 'no') === 'si');

  // Rango de días
  $whereBucket = [];
  if ($incluirVencidos) {
    $whereBucket[] = "TRUE"; // dejamos que entren días <= 0
  } else {
    $whereBucket[] = "DATEDIFF(i.fecha_vencimiento, CURDATE()) > 0";
  }

  if ($bucketSel === '30') {
    $whereBucket[] = "DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 0 AND 30";
  } elseif ($bucketSel === '60') {
    $whereBucket[] = "DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 31 AND 60";
  } elseif ($bucketSel === '90') {
    $whereBucket[] = "DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 61 AND 90";
  } else { // 'all' → 0..90
    $whereBucket[] = "DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 0 AND 90";
  }

  // Filtros adicionales
  $params = [];
  $where  = ["i.fecha_vencimiento IS NOT NULL"];
  $where[] = implode(' AND ', $whereBucket);

  if ($clienteId !== '') {
    $where[] = "p.cliente_id = :cliente_id";
    $params[':cliente_id'] = $clienteId;
  }
  if ($operativaId !== '') {
    $where[] = "p.operativa_id = :operativa_id";
    $params[':operativa_id'] = $operativaId;
  }
  if ($qProducto !== '') {
    $where[] = "(p.sku LIKE :q OR p.denominacion LIKE :q)";
    $params[':q'] = "%{$qProducto}%";
  }
  if ($lote !== '') {
    $where[] = "i.lote LIKE :lote";
    $params[':lote'] = "%{$lote}%";
  }

  // Consulta
  // - Origen de stock: wh_pallet_items (uc_total_cache, fecha_vencimiento, lote)
  // - Join producto/cliente/operativa para descripciones
  // - Group por producto + lote + fecha_vencimiento
  $sql = "
    SELECT
      c.razon_social      AS cliente,
      o.nombre            AS operativa,
      p.sku,
      p.denominacion,
      i.lote,
      i.fecha_vencimiento AS fecha_venc,
      DATEDIFF(i.fecha_vencimiento, CURDATE()) AS vence_en_dias,
      SUM(i.uc_total_cache) AS uc_total,
      CASE
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) <= 0 THEN 'VENCIDO'
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 1 AND 30 THEN '0-30'
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 31 AND 60 THEN '31-60'
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 61 AND 90 THEN '61-90'
        ELSE '>90'
      END AS bucket,
      CASE
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) < 0  THEN 'NEGRO'
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 0 AND 30 THEN 'ROJO'
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) BETWEEN 31 AND 60 THEN 'AMARILLO'
        WHEN DATEDIFF(i.fecha_vencimiento, CURDATE()) >= 90 THEN 'VERDE'
        ELSE 'NEUTRO'
      END AS color      
    FROM wh_pallet_items i
    JOIN para_productos p   ON p.id = i.producto_id
    LEFT JOIN para_clientes c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas o ON o.id = p.operativa_id
    WHERE " . implode(' AND ', $where) . "
    GROUP BY
      p.id, i.lote, i.fecha_vencimiento, c.id, o.id
    HAVING
      -- coherencia con bucket seleccionado (ya limitado en WHERE, pero mantenemos por seguridad)
      (CASE
        WHEN :incl_venc = 1 THEN 1
        ELSE (DATEDIFF(i.fecha_vencimiento, CURDATE()) > 0)
      END) = 1
    ORDER BY vence_en_dias ASC, p.sku ASC, i.lote ASC
  ";

  $stmt = $pdo->prepare($sql);
  $stmt->bindValue(':incl_venc', $incluirVencidos ? 1 : 0, PDO::PARAM_INT);
  foreach ($params as $k => $v) $stmt->bindValue($k, $v);
  $stmt->execute();

  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  echo json_encode([
    'ok'   => true,
    'data' => $rows,
  ]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Error cargando reporte', 'message' => $e->getMessage()]);
}
