<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']); exit;
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $hasTbl = function(PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t,$cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };
  $usePlural = $hasTbl($pdo,'wh_moves');
  $useSing   = !$usePlural && $hasTbl($pdo,'wh_move');

  // -------- Filtros --------
  $desde        = trim((string)($_GET['desde'] ?? ''));
  $hasta        = trim((string)($_GET['hasta'] ?? ''));
  $cliente_id   = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q            = trim((string)($_GET['q'] ?? ''));
  $sentido      = (string)($_GET['sentido'] ?? 'todos'); // mayor|menor|todos
  $top          = (int)($_GET['top'] ?? 0);
  $min_ops      = (int)($_GET['min_ops'] ?? 0);

  // -------- WHERE dinámico (sin m.producto_id) --------
  $where = [];
  $params = [];

  if ($desde !== '') {
    $where[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) >= :desde";
    $params[':desde'] = $desde;
  }
  if ($hasta !== '') {
    $where[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) <= :hasta";
    $params[':hasta'] = $hasta;
  }
  if ($cliente_id !== '') {
    $where[] = "p.cliente_id = :cliente";
    $params[':cliente'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $where[] = "p.operativa_id = :operativa";
    $params[':operativa'] = $operativa_id;
  }
  if ($q !== '') {
    $where[] = "(p.sku LIKE :q OR p.denominacion LIKE :q)";
    $params[':q'] = "%{$q}%";
  }

  $whereSql = $where ? ("WHERE " . implode(' AND ', $where)) : "";

  // -------- Consulta agregada --------
  $sql = $usePlural ? "
    SELECT
      c.razon_social        AS cliente,
      o.nombre              AS operativa,
      p.id                  AS producto_id,
      p.sku,
      p.denominacion,

      /* Totales IN/OUT por unidades (tomamos uc_unidades; si viniera NULL, caemos a uv_cajas) */
      SUM(CASE WHEN m.move_type = 'INBOUND'
               THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
               ELSE 0 END) AS mov_in,
      SUM(CASE WHEN m.move_type = 'OUTBOUND'
               THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
               ELSE 0 END) AS mov_out,

      ( SUM(CASE WHEN m.move_type = 'INBOUND'
                 THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
                 ELSE 0 END)
      - SUM(CASE WHEN m.move_type = 'OUTBOUND'
                 THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
                 ELSE 0 END) ) AS mov_net,

      COUNT(*) AS mov_ops,
      MAX(DATE_FORMAT(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at), '%Y-%m-%d %H:%i')) AS ultimo_mov

    FROM wh_moves m
    JOIN wh_move_items   mi ON mi.move_id = m.id
    JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
    JOIN para_productos   p ON p.id = pi.producto_id
    LEFT JOIN para_clientes  c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas o ON o.id = p.operativa_id

    $whereSql

    GROUP BY c.id, o.id, p.id
    HAVING mov_ops >= :min_ops
  " : "
    SELECT
      c.razon_social        AS cliente,
      o.nombre              AS operativa,
      p.id                  AS producto_id,
      p.sku,
      p.denominacion,

      SUM(CASE WHEN m.tipo = 'IN'
               THEN COALESCE(m.delta_uc, m.delta_uv, 0)
               ELSE 0 END) AS mov_in,
      SUM(CASE WHEN m.tipo = 'OUT'
               THEN COALESCE(m.delta_uc, m.delta_uv, 0)
               ELSE 0 END) AS mov_out,

      ( SUM(CASE WHEN m.tipo = 'IN'
                 THEN COALESCE(m.delta_uc, m.delta_uv, 0)
                 ELSE 0 END)
      - SUM(CASE WHEN m.tipo = 'OUT'
                 THEN COALESCE(m.delta_uc, m.delta_uv, 0)
                 ELSE 0 END) ) AS mov_net,

      COUNT(*) AS mov_ops,
      MAX(DATE_FORMAT(m.created_at, '%Y-%m-%d %H:%i')) AS ultimo_mov

    FROM wh_move m
    JOIN para_productos   p ON p.id = m.producto_id
    LEFT JOIN para_clientes  c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas o ON o.id = p.operativa_id

    $whereSql

    GROUP BY c.id, o.id, p.id
    HAVING mov_ops >= :min_ops
  ";

  // Orden por “sentido”
  $order = " ORDER BY mov_net DESC, mov_in DESC, mov_out ASC, p.sku ASC ";
  if ($sentido === 'menor') {
    $order = " ORDER BY mov_net ASC, mov_out DESC, mov_in ASC, p.sku ASC ";
  }

  $limit = $top > 0 ? (" LIMIT " . (int)$top) : "";

  $sql .= $order . $limit;

  $st = $pdo->prepare($sql);
  $st->bindValue(':min_ops', $min_ops, PDO::PARAM_INT);
  foreach ($params as $k => $v) $st->bindValue($k, $v);
  $st->execute();

  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  echo json_encode(['ok' => true, 'data' => $rows], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Error cargando reporte', 'message' => $e->getMessage()]);
}
