<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 2);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

/* ======================
 * Dependencias (mPDF)
 * ====================== */
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Ejecuta: composer install";
  exit;
}
require_once $aut;
if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

/* ======================
 * Helpers
 * ====================== */
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}
function pillColorByNet($net): array {
  $v = (float)$net;
  if ($v > 0)  return ['#198754', '#FFFFFF', 'POS']; // verde
  if ($v < 0)  return ['#DC3545', '#FFFFFF', 'NEG']; // rojo
  return ['#6C757D', '#FFFFFF', 'NEUTRO'];           // gris
}

/* ======================
 * Main
 * ====================== */
try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $hasTbl = function(PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t,$cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };
  $usePlural = $hasTbl($pdo,'wh_moves');
  $useSing   = !$usePlural && $hasTbl($pdo,'wh_move');

  // -------- Filtros (igual que API JSON) --------
  $desde        = trim((string)($_GET['desde'] ?? ''));
  $hasta        = trim((string)($_GET['hasta'] ?? ''));
  $cliente_id   = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id = trim((string)($_GET['operativa_id'] ?? ''));
  $q            = trim((string)($_GET['q'] ?? ''));
  $sentido      = (string)($_GET['sentido'] ?? 'todos'); // mayor|menor|todos
  $top          = (int)($_GET['top'] ?? 0);
  $min_ops      = (int)($_GET['min_ops'] ?? 0);

  $where = [];
  $params = [];

  if ($desde !== '') {
    $where[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) >= :desde";
    $params[':desde'] = $desde;
  }
  if ($hasta !== '') {
    $where[] = "DATE(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) <= :hasta";
    $params[':hasta'] = $hasta;
  }
  if ($cliente_id !== '') {
    $where[] = "p.cliente_id = :cliente";
    $params[':cliente'] = $cliente_id;
  }
  if ($operativa_id !== '') {
    $where[] = "p.operativa_id = :operativa";
    $params[':operativa'] = $operativa_id;
  }
  if ($q !== '') {
    $where[] = "(p.sku LIKE :q OR p.denominacion LIKE :q)";
    $params[':q'] = "%{$q}%";
  }
  $whereSql = $where ? ("WHERE " . implode(' AND ', $where)) : "";

  // Labels para encabezado
  $clienteLabel   = $cliente_id
    ? ($pdo->query("SELECT razon_social FROM para_clientes WHERE id=".(int)$cliente_id)->fetchColumn() ?: '#'.$cliente_id)
    : '(Todos)';
  $operativaLabel = $operativa_id
    ? ($pdo->query("SELECT nombre FROM sys_operativas WHERE id=".(int)$operativa_id)->fetchColumn() ?: '#'.$operativa_id)
    : '(Todas)';

  // -------- SQL principal (igual que API, sin LIMIT salvo que llegue "top") --------
  $sql = $usePlural ? "
    SELECT
      c.razon_social        AS cliente,
      o.nombre              AS operativa,
      p.id                  AS producto_id,
      p.sku,
      p.denominacion,

      /* Totales IN/OUT por unidades */
      SUM(CASE WHEN m.move_type = 'INBOUND'
               THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
               ELSE 0 END) AS mov_in,
      SUM(CASE WHEN m.move_type = 'OUTBOUND'
               THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
               ELSE 0 END) AS mov_out,

      ( SUM(CASE WHEN m.move_type = 'INBOUND'
                 THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
                 ELSE 0 END)
      - SUM(CASE WHEN m.move_type = 'OUTBOUND'
                 THEN COALESCE(mi.uc_unidades, mi.uv_cajas, 0)
                 ELSE 0 END) ) AS mov_net,

      COUNT(*) AS mov_ops,
      MAX(DATE_FORMAT(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at), '%Y-%m-%d %H:%i')) AS ultimo_mov

  FROM wh_moves m
    JOIN wh_move_items   mi ON mi.move_id = m.id
    JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
    JOIN para_productos   p ON p.id = pi.producto_id
    LEFT JOIN para_clientes  c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas o ON o.id = p.operativa_id

    $whereSql

    GROUP BY c.id, o.id, p.id
    HAVING mov_ops >= :min_ops
  " : "
    SELECT
      c.razon_social        AS cliente,
      o.nombre              AS operativa,
      p.id                  AS producto_id,
      p.sku,
      p.denominacion,

      SUM(CASE WHEN m.tipo = 'IN'
               THEN COALESCE(m.delta_uc, m.delta_uv, 0)
               ELSE 0 END) AS mov_in,
      SUM(CASE WHEN m.tipo = 'OUT'
               THEN COALESCE(m.delta_uc, m.delta_uv, 0)
               ELSE 0 END) AS mov_out,

      ( SUM(CASE WHEN m.tipo = 'IN'
                 THEN COALESCE(m.delta_uc, m.delta_uv, 0)
                 ELSE 0 END)
      - SUM(CASE WHEN m.tipo = 'OUT'
                 THEN COALESCE(m.delta_uc, m.delta_uv, 0)
                 ELSE 0 END) ) AS mov_net,

      COUNT(*) AS mov_ops,
      MAX(DATE_FORMAT(m.created_at, '%Y-%m-%d %H:%i')) AS ultimo_mov

    FROM wh_move m
    JOIN para_productos   p ON p.id = m.producto_id
    LEFT JOIN para_clientes  c ON c.id = p.cliente_id
    LEFT JOIN sys_operativas o ON o.id = p.operativa_id

    $whereSql

    GROUP BY c.id, o.id, p.id
    HAVING mov_ops >= :min_ops
  ";

  // Orden por “sentido”
  $order = " ORDER BY mov_net DESC, mov_in DESC, mov_out ASC, p.sku ASC ";
  if ($sentido === 'menor') {
    $order = " ORDER BY mov_net ASC, mov_out DESC, mov_in ASC, p.sku ASC ";
  }

  $limit = $top > 0 ? (" LIMIT " . (int)$top) : "";

  $sql .= $order . $limit;

  $st = $pdo->prepare($sql);
  $st->bindValue(':min_ops', $min_ops, PDO::PARAM_INT);
  foreach ($params as $k => $v) $st->bindValue($k, $v);
  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  // Totales
  $tot = ['mov_in'=>0,'mov_out'=>0,'mov_net'=>0,'mov_ops'=>0];
  foreach ($rows as $r) {
    $tot['mov_in']  += (float)($r['mov_in']  ?? 0);
    $tot['mov_out'] += (float)($r['mov_out'] ?? 0);
    $tot['mov_net'] += (float)($r['mov_net'] ?? 0);
    $tot['mov_ops'] += (float)($r['mov_ops'] ?? 0);
  }

  // ---------- PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left' => 10, 'margin_right' => 10,
    'margin_top'  => 14, 'margin_bottom' => 12
  ]);

  $title = "Rotación · Productos con mayor y menor movimiento";
  $styles = '<style>
    body{font-family:DejaVu Sans,Arial,sans-serif;font-size:11px;}
    h1{font-size:18px;margin:0 0 6px 0;}
    .muted{color:#666;}
    .kv{display:flex;flex-wrap:wrap;gap:12px;margin:6px 0 8px 0;}
    .kv div{min-width:180px;}
    .grid{width:100%;border-collapse:collapse;margin-top:6px;}
    .grid th,.grid td{border:0.6px solid #999;padding:4px 6px;}
    .grid th{background:#f1f1f1;}
    .right{text-align:right;}
    .pill{display:inline-block;padding:2px 8px;border-radius:999px;font-weight:600;}
    .legend{margin:8px 0 0 0;font-size:10px;color:#555;}
    .legend .dot{display:inline-block;width:10px;height:10px;border-radius:50%;margin-right:4px;vertical-align:middle;}
  </style>';

  ob_start(); ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">
    Generado: <?= date('Y-m-d H:i') ?> · Sistema: <?= htmlspecialchars((string)env('APP_NAME','SOL')) ?>
  </div>

  <div class="kv">
    <div><b>Cliente:</b> <?= htmlspecialchars($clienteLabel) ?></div>
    <div><b>Operativa:</b> <?= htmlspecialchars($operativaLabel) ?></div>
    <div><b>Rango:</b> <?= htmlspecialchars($desde ?: '—') ?> → <?= htmlspecialchars($hasta ?: '—') ?></div>
    <div><b>Filtro producto:</b> <?= htmlspecialchars($q ?: '—') ?></div>
    <div><b>Sentido:</b> <?= htmlspecialchars($sentido) ?></div>
    <div><b>Top:</b> <?= $top > 0 ? (int)$top : '—' ?></div>
    <div><b>Mín. operaciones:</b> <?= (int)$min_ops ?></div>
    <div><b>Filas:</b> <?= count($rows) ?></div>
  </div>

  <div class="legend">
    <span class="dot" style="background:#198754"></span> Neto &gt; 0 (Verde) &nbsp;·&nbsp;
    <span class="dot" style="background:#DC3545"></span> Neto &lt; 0 (Rojo) &nbsp;·&nbsp;
    <span class="dot" style="background:#6C757D"></span> Neto = 0 (Gris)
  </div>

  <table class="grid">
    <thead>
      <tr>
        <th>Cliente</th>
        <th>Operativa</th>
        <th>SKU</th>
        <th>Producto</th>
        <th class="right">IN (uc)</th>
        <th class="right">OUT (uc)</th>
        <th class="right">NETO (uc)</th>
        <th class="right">Operaciones</th>
        <th>Último mov.</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($rows as $r):
        $net = (float)($r['mov_net'] ?? 0);
        [$bg, $fg, $lbl] = pillColorByNet($net);
      ?>
      <tr>
        <td><?= htmlspecialchars((string)$r['cliente']) ?></td>
        <td><?= htmlspecialchars((string)$r['operativa']) ?></td>
        <td><?= htmlspecialchars((string)$r['sku']) ?></td>
        <td><?= htmlspecialchars((string)$r['denominacion']) ?></td>
        <td class="right"><?= nf($r['mov_in']) ?></td>
        <td class="right"><?= nf($r['mov_out']) ?></td>
        <td class="right"><span class="pill" style="background:<?= $bg ?>;color:<?= $fg ?>"><?= nf($net) ?></span></td>
        <td class="right"><?= nf($r['mov_ops']) ?></td>
        <td><?= htmlspecialchars((string)$r['ultimo_mov']) ?></td>
      </tr>
      <?php endforeach; ?>
    </tbody>
    <tfoot>
      <tr>
        <th colspan="4" class="right">Totales</th>
        <th class="right"><?= nf($tot['mov_in']) ?></th>
        <th class="right"><?= nf($tot['mov_out']) ?></th>
        <th class="right">
          <?php [$bgT,$fgT] = pillColorByNet($tot['mov_net']); ?>
          <span class="pill" style="background:<?= $bgT ?>;color:<?= $fgT ?>"><?= nf($tot['mov_net']) ?></span>
        </th>
        <th class="right"><?= nf($tot['mov_ops']) ?></th>
        <th></th>
      </tr>
    </tfoot>
  </table>
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output('rotacion_minmax.pdf', \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local') ? ("Error generando PDF: ".$e->getMessage()) : "No se pudo generar el PDF";
}
