<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';

if (empty($_SESSION['usuario_id'])) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'No autorizado']); exit;
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  // Table detection
  $hasTbl = function(PDO $pdo, string $t): bool {
    static $cache = [];
    if (array_key_exists($t,$cache)) return $cache[$t];
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=?");
    $st->execute([$t]);
    return $cache[$t] = ((int)$st->fetchColumn() > 0);
  };
  $usePlural = $hasTbl($pdo,'wh_moves');
  $useSing   = !$usePlural && $hasTbl($pdo,'wh_move');

  // -------- Filtros --------
  $desde            = trim((string)($_GET['desde'] ?? ''));
  $hasta            = trim((string)($_GET['hasta'] ?? ''));
  $cliente_id       = trim((string)($_GET['cliente_id'] ?? ''));
  $operativa_id     = trim((string)($_GET['operativa_id'] ?? ''));
  $q                = trim((string)($_GET['q'] ?? ''));
  $incluir_abiertos = (($_GET['incluir_abiertos'] ?? 'no') === 'si');
  $min_items        = (int)($_GET['min_items'] ?? 1);

  // -------- Base de fechas --------
  $tz = new DateTimeZone((string)env('TIMEZONE', 'UTC'));
  $today = (new DateTime('now', $tz))->format('Y-m-d');

  // -------- Subconsultas: IN y OUT por pallet_item --------
  if ($usePlural) {
    $sqlIn = "
      SELECT mi.pallet_item_id, MIN(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) AS in_at
      FROM wh_moves m
      JOIN wh_move_items mi ON mi.move_id = m.id
      WHERE m.move_type = 'INBOUND'
      GROUP BY mi.pallet_item_id
    ";
    $sqlOut = "
      SELECT mi.pallet_item_id, MAX(COALESCE(m.finalizado_at, m.iniciado_at, m.created_at)) AS out_at
      FROM wh_moves m
      JOIN wh_move_items mi ON mi.move_id = m.id
      WHERE m.move_type = 'OUTBOUND'
      GROUP BY mi.pallet_item_id
    ";
  } elseif ($useSing) {
    $sqlIn = "
      SELECT m.pallet_id AS pallet_item_id, MIN(m.created_at) AS in_at
      FROM wh_move m
      WHERE m.tipo = 'IN'
      GROUP BY m.pallet_id
    ";
    $sqlOut = "
      SELECT m.pallet_id AS pallet_item_id, MAX(m.created_at) AS out_at
      FROM wh_move m
      WHERE m.tipo = 'OUT'
      GROUP BY m.pallet_id
    ";
  } else {
    throw new RuntimeException('No existe tabla de movimientos');
  }

  // -------- WHERE dinámico --------
  $where  = [];
  $params = [];

  if ($cliente_id !== '')    { $where[] = "p.cliente_id = :cliente";          $params[':cliente']   = $cliente_id; }
  if ($operativa_id !== '')  { $where[] = "p.operativa_id = :operativa";      $params[':operativa'] = $operativa_id; }
  if ($q !== '')             { $where[] = "(p.sku LIKE :q OR p.denominacion LIKE :q)"; $params[':q'] = "%{$q}%"; }

  if ($desde !== '' || $hasta !== '') {
    if ($incluir_abiertos) {
      if ($desde !== '') { $where[] = "DATE(COALESCE(o.out_at, i.in_at)) >= :desde"; $params[':desde'] = $desde; }
      if ($hasta !== '') { $where[] = "DATE(COALESCE(o.out_at, i.in_at)) <= :hasta"; $params[':hasta'] = $hasta; }
    } else {
      if ($desde !== '') { $where[] = "DATE(o.out_at) >= :desde"; $params[':desde'] = $desde; }
      if ($hasta !== '') { $where[] = "DATE(o.out_at) <= :hasta"; $params[':hasta'] = $hasta; }
    }
  }

  $whereSql = $where ? ("WHERE " . implode(' AND ', $where)) : "";

  // -------- Fracciones condicionales para abiertos (evita repetir el mismo placeholder) --------
  $caseOpenAvg = $incluir_abiertos ? "WHEN i.in_at IS NOT NULL AND o.out_at IS NULL THEN DATEDIFF(:today1, DATE(i.in_at))" : "";
  $caseOpenMin = $incluir_abiertos ? "WHEN i.in_at IS NOT NULL AND o.out_at IS NULL THEN DATEDIFF(:today2, DATE(i.in_at))" : "";
  $caseOpenMax = $incluir_abiertos ? "WHEN i.in_at IS NOT NULL AND o.out_at IS NULL THEN DATEDIFF(:today3, DATE(i.in_at))" : "";

  // -------- SQL principal --------
  $sql = "
    WITH i AS ($sqlIn),
         o AS ($sqlOut)
    SELECT
      c.razon_social        AS cliente,
      ovr.nombre            AS operativa,
      p.id                  AS producto_id,
      p.sku,
      p.denominacion,

      -- Contamos items cerrados
      SUM(CASE
            WHEN i.in_at IS NOT NULL AND o.out_at IS NOT NULL AND o.out_at >= i.in_at
            THEN 1 ELSE 0
          END) AS items,

      AVG(CASE
            WHEN i.in_at IS NOT NULL AND o.out_at IS NOT NULL AND o.out_at >= i.in_at
              THEN DATEDIFF(DATE(o.out_at), DATE(i.in_at))
            $caseOpenAvg
            ELSE NULL
          END) AS prom_dias,

      MIN(CASE
            WHEN i.in_at IS NOT NULL AND o.out_at IS NOT NULL AND o.out_at >= i.in_at
              THEN DATEDIFF(DATE(o.out_at), DATE(i.in_at))
            $caseOpenMin
            ELSE NULL
          END) AS min_dias,

      MAX(CASE
            WHEN i.in_at IS NOT NULL AND o.out_at IS NOT NULL AND o.out_at >= i.in_at
              THEN DATEDIFF(DATE(o.out_at), DATE(i.in_at))
            $caseOpenMax
            ELSE NULL
          END) AS max_dias,

      MAX(DATE_FORMAT(o.out_at, '%Y-%m-%d %H:%i')) AS ultimo_out

    FROM wh_pallet_items pi
    JOIN i ON i.pallet_item_id = pi.id
    LEFT JOIN o ON o.pallet_item_id = pi.id
    JOIN para_productos    p   ON p.id  = pi.producto_id
    LEFT JOIN para_clientes c   ON c.id  = p.cliente_id
    LEFT JOIN sys_operativas ovr ON ovr.id = p.operativa_id

    $whereSql

    GROUP BY c.id, ovr.id, p.id
    HAVING (items >= :min_items)
    ORDER BY prom_dias DESC, p.sku ASC
  ";

  $st = $pdo->prepare($sql);

  // Bind de parámetros
  $st->bindValue(':min_items', $min_items, PDO::PARAM_INT);
  foreach ($params as $k => $v) { $st->bindValue($k, $v); }

  // Si incluimos abiertos, bindear placeholders duplicados
  if ($incluir_abiertos) {
    $st->bindValue(':today1', $today);
    $st->bindValue(':today2', $today);
    $st->bindValue(':today3', $today);
  }

  $st->execute();
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

  echo json_encode(['ok' => true, 'data' => $rows], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Error cargando reporte', 'message' => $e->getMessage()]);
}
