<?php
/**
 * SOL · Paso 6 — Endpoints Warehouse
 * Ruta: public/api/wh.php
 *
 * Operaciones (action):
 * - ingreso        → CALL sp_wh_ingreso_create
 * - reposicion     → CALL sp_wh_reposicion
 * - salida_total   → CALL sp_wh_salida_total
 * - salida_parcial → CALL sp_wh_salida_parcial
 *
 * Método: POST (JSON o x-www-form-urlencoded)
 * Respuesta: JSON { ok: bool, data|error, meta }
 */

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

$BASE = dirname(__DIR__, 2);
require_once $BASE . '/config/config.php';

$startHr = hrtime(true);

try {
    // Carga PDO desde tu config
    require_once $BASE . '/config/db.php';
    $pdo = getPDO();

    // Detectar acción
    $action = $_GET['action'] ?? $_POST['action'] ?? null;
    if (!$action) {
        respond(400, ['ok' => false, 'error' => 'Falta parámetro action. Valores: ingreso | reposicion | salida_total | salida_parcial']);
    }

    // Leer body (JSON o form)
    $raw = file_get_contents('php://input');
    $isJson = (isset($_SERVER['CONTENT_TYPE']) && stripos($_SERVER['CONTENT_TYPE'], 'application/json') !== false);
    $payload = [];
    if ($isJson && $raw) {
        $payload = json_decode($raw, true);
        if (!is_array($payload)) {
            respond(400, ['ok' => false, 'error' => 'JSON inválido']);
        }
    } else {
        // Mezcla POST + GET (sin action)
        $payload = array_merge($_POST, $_GET);
        unset($payload['action']);
    }

    // Normalizador helper
    $str = fn($k, $def = null) => isset($payload[$k]) ? trim((string)$payload[$k]) : $def;
    $int = fn($k, $def = null) => isset($payload[$k]) && $payload[$k] !== '' ? (int)$payload[$k] : $def;

    switch ($action) {

        case 'ingreso':
            /**
             * Espera:
             *  deposito_code   (string, req)
             *  pallet_codigo   (string, req)
             *  pos_code        (string, opcional)  → si null, el SP elige CUARENTENA
             *  estado_code     (string, opcional)  → default 'CUARENTENA'
             *  items           (array JSON de objetos)
             *  overwrite       (0/1) → si el pallet existe
             */
            $deposito = $str('deposito_code');
            $pallet   = $str('pallet_codigo');
            $posCode  = $payload['pos_code']    ?? null;
            $estado   = $payload['estado_code'] ?? null;
            $items    = $payload['items']       ?? null;
            $overwrite= (int)($payload['overwrite'] ?? 0);

            if (!$deposito || !$pallet) {
                respond(422, ['ok' => false, 'error' => 'deposito_code y pallet_codigo son requeridos.']);
            }
            if (!is_array($items) || count($items) === 0) {
                respond(422, ['ok' => false, 'error' => 'items debe ser un array con al menos un elemento.']);
            }

            // El SP recibe JSON; encodemos de forma estable
            $itemsJson = json_encode($items, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

            $stmt = $pdo->prepare("CALL sp_wh_ingreso_create(?,?,?,?,?,?)");
            $stmt->execute([
                $deposito,
                $pallet,
                $posCode,
                $estado,
                $itemsJson,
                $overwrite
            ]);
            $data = fetchAllResultsets($stmt);
            respond(200, ['ok' => true, 'data' => $data, 'meta' => meta($action, $startHr)]);
            break;

        case 'reposicion':
            /**
             * Espera:
             *  deposito_code (string, req)
             *  pallet_codigo (string, req)
             *  to_pos_code   (string, req)
             *  motivo        (string, opcional)  → default 'UBICACION'
             *  estado_dest   (string, opcional)  → default POS_ENTERO / POS_PICKEADO según pickeado
             */
            $deposito = $str('deposito_code');
            $pallet   = $str('pallet_codigo');
            $toPos    = $str('to_pos_code');
            $motivo   = $payload['motivo'] ?? null;
            $estado   = $payload['estado_dest'] ?? null;

            if (!$deposito || !$pallet || !$toPos) {
                respond(422, ['ok' => false, 'error' => 'deposito_code, pallet_codigo y to_pos_code son requeridos.']);
            }

            $stmt = $pdo->prepare("CALL sp_wh_reposicion(?,?,?,?,?)");
            $stmt->execute([$deposito, $pallet, $toPos, $motivo, $estado]);
            $data = fetchAllResultsets($stmt);
            respond(200, ['ok' => true, 'data' => $data, 'meta' => meta($action, $startHr)]);
            break;

        case 'salida_total':
            /**
             * Espera:
             *  deposito_code (string, req)
             *  pallet_codigo (string, req)
             *  referencia    (string, opcional) → default 'DESPACHO'
             */
            $deposito = $str('deposito_code');
            $pallet   = $str('pallet_codigo');
            $ref      = $payload['referencia'] ?? null;

            if (!$deposito || !$pallet) {
                respond(422, ['ok' => false, 'error' => 'deposito_code y pallet_codigo son requeridos.']);
            }

            $stmt = $pdo->prepare("CALL sp_wh_salida_total(?,?,?)");
            $stmt->execute([$deposito, $pallet, $ref]);
            $data = fetchAllResultsets($stmt);
            respond(200, ['ok' => true, 'data' => $data, 'meta' => meta($action, $startHr)]);
            break;

        case 'salida_parcial':
            /**
             * Espera:
             *  deposito_code (string, req)
             *  pallet_codigo (string, req)
             *  producto_id   (int, req)
             *  lote_codigo   (string, req)
             *  delta_uv      (int, req >=0)
             *  delta_uc      (int, req >=0)
             *  referencia    (string, opcional) → default 'PICKING'
             */
            $deposito  = $str('deposito_code');
            $pallet    = $str('pallet_codigo');
            $producto  = $int('producto_id');
            $loteCode  = $str('lote_codigo');
            $deltaUv   = $int('delta_uv', 0);
            $deltaUc   = $int('delta_uc', 0);
            $ref       = $payload['referencia'] ?? null;

            if (!$deposito || !$pallet || !$producto || !$loteCode) {
                respond(422, ['ok' => false, 'error' => 'deposito_code, pallet_codigo, producto_id y lote_codigo son requeridos.']);
            }
            if (($deltaUv ?? 0) < 0 || ($deltaUc ?? 0) < 0 || (($deltaUv ?? 0) === 0 && ($deltaUc ?? 0) === 0)) {
                respond(422, ['ok' => false, 'error' => 'delta_uv/delta_uc deben ser ≥ 0 y al menos uno > 0.']);
            }

            $stmt = $pdo->prepare("CALL sp_wh_salida_parcial(?,?,?,?,?,?,?)");
            $stmt->execute([$deposito, $pallet, $producto, $loteCode, $deltaUv, $deltaUc, $ref]);
            $data = fetchAllResultsets($stmt);
            respond(200, ['ok' => true, 'data' => $data, 'meta' => meta($action, $startHr)]);
            break;

        default:
            respond(400, ['ok' => false, 'error' => 'action desconocida']);
    }

} catch (Throwable $e) {
    $code = (int)($e->getCode() ?: 500);
    // Sanitiza mensaje para no filtrar info sensible
    $msg = $e->getMessage();
    respond(($code >= 400 && $code < 600) ? $code : 500, [
        'ok'    => false,
        'error' => $msg,
        'meta'  => meta('error', $startHr)
    ]);
}

/* =================== Helpers =================== */

function fetchAllResultsets(PDOStatement $stmt): array {
    // Devuelve todos los resultsets que generó el CALL
    $all = [];
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && count($rows)) {
            $all[] = $rows;
        }
    } while ($stmt->nextRowset());
    // Si sólo hay un set, devolverlo directamente
    if (count($all) === 1) {
        return $all[0];
    }
    return $all;
}

function meta(string $action, int $startHr): array {
    $ms = (hrtime(true) - $startHr) / 1e6;
    return [
        'action' => $action,
        'took_ms' => round($ms, 2),
        'ts' => date('c')
    ];
}

function respond(int $httpCode, array $payload): void {
    http_response_code($httpCode);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}
