<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Helpers para APIs (JSON, introspección de esquema, utilidades).
 *
 * NOTA:
 * - No configuramos headers ni error_reporting aquí (hazlo en cada endpoint).
 * - Todos los helpers son independientes y reutilizables.
 */

/* ================================================================
 * JSON helpers
 * ================================================================ */

/**
 * Respuesta JSON OK y termina la ejecución.
 *
 * @param mixed $data
 * @return never
 */
function json_ok($data = null): void
{
    echo json_encode(['ok' => true, 'data' => $data], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Respuesta JSON de error con código HTTP y termina la ejecución.
 *
 * @param string $msg   Mensaje de error para UI
 * @param mixed  $extra (opcional) Detalles de depuración
 * @param int    $code  Código HTTP (por defecto 400)
 * @return never
 */
function json_err(string $msg, $extra = null, int $code = 400): void
{
    http_response_code($code);
    $o = ['ok' => false, 'error' => $msg];
    if ($extra !== null) {
        $o['debug'] = $extra;
    }
    echo json_encode($o, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Lee y decodifica el cuerpo JSON de la solicitud.
 *
 * @return array
 */
function getJsonInput(): array
{
    $raw = file_get_contents('php://input');
    if (!$raw) return [];
    $j = json_decode($raw, true);
    return is_array($j) ? $j : [];
}


/* ================================================================
 * Introspección de esquema (MySQL)
 * ================================================================ */

/**
 * Verifica si existe una tabla en la BD actual.
 *
 * @param PDO    $pdo
 * @param string $t   Nombre de la tabla
 * @return bool
 */
function tableExists(PDO $pdo, string $t): bool
{
    $st = $pdo->prepare(
        "SELECT 1
           FROM INFORMATION_SCHEMA.TABLES
          WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME   = ?"
    );
    $st->execute([$t]);
    return (bool) $st->fetchColumn();
}

/**
 * Verifica si existe una columna en una tabla de la BD actual.
 *
 * @param PDO    $pdo
 * @param string $t   Nombre de la tabla
 * @param string $c   Nombre de la columna
 * @return bool
 */
function columnExists(PDO $pdo, string $t, string $c): bool
{
    $st = $pdo->prepare(
        "SELECT 1
           FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME   = ?
            AND COLUMN_NAME  = ?"
    );
    $st->execute([$t, $c]);
    return (bool) $st->fetchColumn();
}

/**
 * Retorna la primera tabla existente de una lista de candidatas.
 *
 * @param PDO    $pdo
 * @param array  $cands Lista de nombres de tabla
 * @return string|null  Nombre encontrado o null
 */
function pickExistingTable(PDO $pdo, array $cands): ?string
{
    foreach ($cands as $t) {
        if (tableExists($pdo, $t)) return $t;
    }
    return null;
}

/**
 * Indica si una columna es generada (STORED/VIRTUAL).
 *
 * @param PDO    $pdo
 * @param string $table
 * @param string $column
 * @return bool
 */
function isGeneratedColumn(PDO $pdo, string $table, string $column): bool
{
    $st = $pdo->prepare(
        "SELECT EXTRA, GENERATION_EXPRESSION
           FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME   = ?
            AND COLUMN_NAME  = ?"
    );
    $st->execute([$table, $column]);
    $row = $st->fetch();
    if (!$row) return false;

    $extra = strtoupper((string)($row['EXTRA'] ?? ''));
    $gen   = (string)($row['GENERATION_EXPRESSION'] ?? '');
    return (strpos($extra, 'GENERATED') !== false) || ($gen !== '');
}


/* ================================================================
 * Utilidades de negocio (genéricas)
 * ================================================================ */

/**
 * Devuelve la moda (valor más frecuente) de una lista de enteros positivos.
 * Si no hay valores > 0, retorna null.
 *
 * @param array<int|numeric-string> $vals
 * @return int|null
 */
function mode_numeric(array $vals): ?int
{
    $cnt = [];
    foreach ($vals as $v) {
        $n = (int) $v;
        if ($n <= 0) continue;
        $cnt[$n] = ($cnt[$n] ?? 0) + 1;
    }
    if (!$cnt) return null;
    arsort($cnt);
    return (int) array_key_first($cnt);
}


/* ================================================================
 * Manejo de errores fatales (formato JSON)
 * ================================================================ */

/**
 * Captura errores fatales y devuelve JSON legible en lugar de página en blanco.
 * No interfiere con exceptions normales (estas deben capturarse en el endpoint).
 */
register_shutdown_function(function (): void {
    $err = error_get_last();
    if ($err && in_array($err['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
        if (!headers_sent()) {
            header('Content-Type: application/json; charset=utf-8', true, 500);
        }
        echo json_encode([
            'ok'    => false,
            'error' => 'Fatal error en API',
            'debug' => [
                'type'    => $err['type'],
                'message' => $err['message'],
                'file'    => $err['file'],
                'line'    => $err['line'],
            ],
        ], JSON_UNESCAPED_UNICODE);
    }
});
