<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Soporte: app/Support/Assets.php
 *
 * Helpers para gestionar assets (CSS/JS) desde las vistas sin repetir lógica.
 * Depende de app/Support/ViewHelpers.php (sol_section, sol_append_section, project_path, url).
 */

require_once __DIR__ . '/ViewHelpers.php';

// -------------------------------------------------------------
// Utilitarios internos
// -------------------------------------------------------------
if (!function_exists('sol_assets_normalize_list')) {
  /**
   * Normaliza una lista de assets que puede venir como:
   *  - ['https://cdn/archivo.css', ...]
   *  - [['path' => 'https://cdn/archivo.css'], ...]
   */
  function sol_assets_normalize_list(array $items): array
  {
    $out = [];
    foreach ($items as $it) {
      if (is_string($it) && $it !== '') {
        $out[] = $it;
      } elseif (is_array($it) && !empty($it['path'])) {
        $out[] = (string) $it['path'];
      }
    }
    return $out;
  }
}

if (!function_exists('sol_assets_push')) {
  /**
   * Empuja CSS y JS al layout (secciones: page_css, page_js).
   *
   * @param array $css
   * @param array $js
   */
  function sol_assets_push(array $css = [], array $js = []): void
  {
    if (!empty($css)) {
      sol_append_section('page_css', $css);
    }
    if (!empty($js)) {
      sol_append_section('page_js', $js);
    }
  }
}

// -------------------------------------------------------------
// DataTables bundle
// -------------------------------------------------------------
if (!function_exists('sol_dt_assets')) {
  /**
   * Carga los assets de DataTables (desde app/config/dataTablesConfig.php) + dataTablesBase.js
   * y luego agrega el script específico de la página.
   *
   * @param string $pageScriptRel  Path del JS de la página (relativo a BASE_URL), ej:
   *                               '/page-scripts/pages/parametros/productos_scripts.js'
   * @param array  $extraCss       CSS extra a sumar (opcional)
   * @param array  $extraJs        JS extra a sumar (después del pageScript) (opcional)
   */
  function sol_dt_assets(string $pageScriptRel, array $extraCss = [], array $extraJs = []): void
  {
    // 1) Cargar config DT si la función aún no existe
    if (!function_exists('getDataTablesConfig')) {
      $cfgFile = project_path('app/config/dataTablesConfig.php');
      if (is_file($cfgFile)) {
        require_once $cfgFile;
      }
    }

    // 2) Obtener listado de assets desde la config (o arrays vacíos)
    $cfg = function_exists('getDataTablesConfig')
      ? (array) getDataTablesConfig()
      : ['scripts' => [], 'styles' => []];

    $dtCss = sol_assets_normalize_list($cfg['styles']  ?? []);
    $dtJs  = sol_assets_normalize_list($cfg['scripts'] ?? []);

    // 3) Bundle ordenado:
    //    - Primero CSS de DT
    //    - Luego JS de DT
    //    - Luego base común de la app para DataTables
    //    - Luego el script específico de la página
    //    - Finalmente extras si los hay
    $pageScript = url($pageScriptRel);

    $css = array_merge($dtCss, $extraCss);
    $js  = array_merge($dtJs, [
      url('/page-scripts/global/dataTablesBase.js'),
      $pageScript,
    ], $extraJs);

    sol_assets_push($css, $js);
  }
}
