<?php
declare(strict_types=1);

if (!function_exists('embarque_table_exists')) {
    function embarque_table_exists(PDO $pdo, string $table): bool {
        static $cache = [];
        $dbKey = spl_object_id($pdo);
        $tableKey = $dbKey . ':' . strtolower($table);
        if (array_key_exists($tableKey, $cache)) {
            return (bool)$cache[$tableKey];
        }
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
        $stmt->execute([$table]);
        $cache[$tableKey] = ((int)$stmt->fetchColumn()) > 0;
        return (bool)$cache[$tableKey];
    }
}

if (!function_exists('embarque_column_exists')) {
    function embarque_column_exists(PDO $pdo, string $table, string $column): bool {
        static $cache = [];
        $dbKey = spl_object_id($pdo);
        $colKey = $dbKey . ':' . strtolower($table) . ':' . strtolower($column);
        if (array_key_exists($colKey, $cache)) {
            return (bool)$cache[$colKey];
        }
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
        $stmt->execute([$table, $column]);
        $cache[$colKey] = ((int)$stmt->fetchColumn()) > 0;
        return (bool)$cache[$colKey];
    }
}

if (!function_exists('embarque_detect_column')) {
    function embarque_detect_column(PDO $pdo, string $table, array $candidates): ?string {
        foreach ($candidates as $candidate) {
            if ($candidate && embarque_column_exists($pdo, $table, $candidate)) {
                return $candidate;
            }
        }

        return null;
    }
}

if (!function_exists('embarque_fetch_pedido_meta')) {
    function embarque_fetch_pedido_meta(PDO $pdo, array $pedidoDestIds): array {
        $pedidoDestIds = array_values(array_unique(array_filter(array_map('intval', $pedidoDestIds))));
        if (empty($pedidoDestIds)) {
            return [];
        }
        if (!embarque_table_exists($pdo, 'so_pedido_dest')) {
            return [];
        }
        $placeholders = implode(',', array_fill(0, count($pedidoDestIds), '?'));
        $sql = "SELECT d.id, d.destinatario_id, dest.razon_social
                  FROM so_pedido_dest d
             LEFT JOIN para_destinatarios dest ON dest.id = d.destinatario_id
                 WHERE d.id IN ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($pedidoDestIds);
        $meta = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $pid = (int)($row['id'] ?? 0);
            if ($pid <= 0) {
                continue;
            }
            $meta[$pid] = [
                'destinatario_id' => isset($row['destinatario_id']) ? (int)$row['destinatario_id'] : null,
                'destinatario' => isset($row['razon_social']) ? (string)$row['razon_social'] : null,
            ];
        }
        return $meta;
    }
}

if (!function_exists('embarque_collect_pedido_docs')) {
    function embarque_collect_pedido_docs(PDO $pdo, array $pedidoDestIds): array {
        $pedidoDestIds = array_values(array_unique(array_filter(array_map('intval', $pedidoDestIds))));
        if (empty($pedidoDestIds)) {
            return [];
        }
        $docsByPid = [];
        $seenByPid = [];

        $registerDoc = static function (int $pid, ?string $tipo, ?string $numero, string $label, string $source) use (&$docsByPid, &$seenByPid): void {
            $tipoNorm = $tipo !== null ? strtoupper(trim($tipo)) : null;
            $numeroNorm = $numero !== null ? trim($numero) : null;
            if ($numeroNorm === '') {
                $numeroNorm = null;
            }
            $tipoNorm = ($tipoNorm === '') ? null : $tipoNorm;
            $label = trim($label);
            if ($label === '' && $numeroNorm !== null) {
                $label = $numeroNorm;
            }
            if ($label === '') {
                return;
            }
            $key = ($tipoNorm ?? '') . '|' . ($numeroNorm ?? '') . '|' . $label;
            if (!isset($seenByPid[$pid])) {
                $seenByPid[$pid] = [];
            }
            if (isset($seenByPid[$pid][$key])) {
                return;
            }
            $seenByPid[$pid][$key] = true;
            $docsByPid[$pid][] = [
                'doc_tipo' => $tipoNorm,
                'doc_numero' => $numeroNorm,
                'label' => $label,
                'factura' => ($tipoNorm !== null && strpos($tipoNorm, 'FACT') === 0) ? $numeroNorm : null,
                'source' => $source,
            ];
        };

        if (embarque_table_exists($pdo, 'so_pedido_dest_doc') && embarque_column_exists($pdo, 'so_pedido_dest_doc', 'pedido_dest_id')) {
            $placeholders = implode(',', array_fill(0, count($pedidoDestIds), '?'));
            $sql = "SELECT pedido_dest_id, doc_tipo, doc_numero FROM so_pedido_dest_doc WHERE pedido_dest_id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($pedidoDestIds);
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $pid = (int)($row['pedido_dest_id'] ?? 0);
                if ($pid <= 0) {
                    continue;
                }
                $tipo = isset($row['doc_tipo']) ? (string)$row['doc_tipo'] : null;
                $numero = isset($row['doc_numero']) ? (string)$row['doc_numero'] : null;
                if (($tipo === null || trim($tipo) === '') && ($numero === null || trim($numero) === '')) {
                    continue;
                }
                $label = '';
                if ($tipo !== null && trim($tipo) !== '') {
                    $label .= trim($tipo) . ': ';
                }
                if ($numero !== null) {
                    $label .= trim($numero);
                }
                $registerDoc($pid, $tipo, $numero, $label, 'doc_table');
            }
        }

        if (embarque_table_exists($pdo, 'so_pedido_dest')) {
            $hasFactura = embarque_column_exists($pdo, 'so_pedido_dest', 'factura');
            $hasDocNum = embarque_column_exists($pdo, 'so_pedido_dest', 'doc_numero');
            if ($hasFactura || $hasDocNum) {
                $placeholders = implode(',', array_fill(0, count($pedidoDestIds), '?'));
                $cols = ['id'];
                if ($hasFactura) {
                    $cols[] = 'factura';
                }
                if ($hasDocNum) {
                    $cols[] = 'doc_numero';
                }
                $sql = 'SELECT ' . implode(',', $cols) . ' FROM so_pedido_dest WHERE id IN (' . $placeholders . ')';
                $stmt = $pdo->prepare($sql);
                $stmt->execute($pedidoDestIds);
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $pid = (int)($row['id'] ?? 0);
                    if ($pid <= 0) {
                        continue;
                    }
                    if ($hasFactura && isset($row['factura'])) {
                        $fact = trim((string)$row['factura']);
                        if ($fact !== '') {
                            $registerDoc($pid, 'FACTURA', $fact, 'Factura: ' . $fact, 'pedido_dest.factura');
                        }
                    }
                    if ($hasDocNum && isset($row['doc_numero'])) {
                        $doc = trim((string)$row['doc_numero']);
                        if ($doc !== '') {
                            $registerDoc($pid, 'DOC', $doc, 'Doc: ' . $doc, 'pedido_dest.doc_numero');
                        }
                    }
                }
            }
        }

        return $docsByPid;
    }
}

if (!function_exists('embarque_collect_pedido_items')) {
    function embarque_collect_pedido_items(PDO $pdo, array $pedidoDestIds): array {
        $pedidoDestIds = array_values(array_unique(array_filter(array_map('intval', $pedidoDestIds))));
        if (empty($pedidoDestIds)) {
            return [];
        }
        if (!embarque_table_exists($pdo, 'so_pedido_dest_item')) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($pedidoDestIds), '?'));
        $select = ['i.id AS item_id', 'i.pedido_dest_id'];

        $productoIdExists = embarque_column_exists($pdo, 'so_pedido_dest_item', 'producto_id');
        if ($productoIdExists) {
            $select[] = 'i.producto_id';
        }

        $itemSkuCol = embarque_detect_column($pdo, 'so_pedido_dest_item', ['sku', 'sku_cliente', 'producto_codigo', 'codigo_producto', 'codigo']);
        if ($itemSkuCol) {
            $select[] = 'i.`' . $itemSkuCol . '` AS item_sku';
        }

        $itemDescCol = embarque_detect_column($pdo, 'so_pedido_dest_item', ['descripcion', 'producto_txt', 'producto_nombre', 'denominacion']);
        if ($itemDescCol) {
            $select[] = 'i.`' . $itemDescCol . '` AS item_descripcion';
        }

        $joins = [];
        $productSkuCol = null;
        $productDescCol = null;

        if ($productoIdExists && embarque_table_exists($pdo, 'para_productos')) {
            $joins[] = 'LEFT JOIN para_productos p ON p.id = i.producto_id';
            $productSkuCol = embarque_detect_column($pdo, 'para_productos', ['sku', 'codigo', 'cod', 'sku_cliente']);
            if ($productSkuCol) {
                $select[] = 'p.`' . $productSkuCol . '` AS producto_sku';
            }
            $productDescCol = embarque_detect_column($pdo, 'para_productos', ['denominacion', 'descripcion', 'nombre', 'titulo']);
            if ($productDescCol) {
                $select[] = 'p.`' . $productDescCol . '` AS producto_nombre';
            }
        }

        $sql = 'SELECT ' . implode(', ', $select) . ' FROM so_pedido_dest_item i ' . implode(' ', $joins)
             . ' WHERE i.pedido_dest_id IN (' . $placeholders . ') ORDER BY i.pedido_dest_id, i.id';

        $stmt = $pdo->prepare($sql);
        $stmt->execute($pedidoDestIds);

        $map = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $pedidoId = isset($row['pedido_dest_id']) ? (int)$row['pedido_dest_id'] : 0;
            if ($pedidoId <= 0) {
                continue;
            }
            $sku = trim((string)($row['item_sku'] ?? $row['producto_sku'] ?? ''));
            $descripcion = trim((string)($row['item_descripcion'] ?? $row['producto_nombre'] ?? ''));
            $productoId = isset($row['producto_id']) ? (int)$row['producto_id'] : null;
            $itemId = isset($row['item_id']) ? (int)$row['item_id'] : 0;

            $labelParts = [];
            if ($sku !== '') {
                $labelParts[] = $sku;
            }
            if ($descripcion !== '') {
                $labelParts[] = $descripcion;
            }
            $label = $labelParts ? implode(' · ', $labelParts) : ($sku !== '' ? $sku : ($descripcion !== '' ? $descripcion : ('Item #' . ($itemId ?: $pedidoId))));

            $entry = [
                'item_id' => $itemId,
                'pedido_dest_id' => $pedidoId,
                'producto_id' => $productoId,
                'sku' => $sku,
                'descripcion' => $descripcion,
                'label' => $label,
            ];

            if (!isset($map[$pedidoId])) {
                $map[$pedidoId] = [];
            }
            $map[$pedidoId][] = $entry;
        }

        return $map;
    }
}

if (!function_exists('embarque_collect_destinatarios_docs')) {
    function embarque_collect_destinatarios_docs(PDO $pdo, int $embarqueId): array {
        $byDest = [];
        $pedidoDestIds = [];

        $sqlPre = "SELECT pre.id
                     FROM so_embarque_pre ep
                     JOIN so_preembarque pre ON pre.id = ep.preembarque_id
                    WHERE ep.embarque_id = ?
                 ORDER BY pre.id";
        $stPre = $pdo->prepare($sqlPre);
        $stPre->execute([$embarqueId]);
        $preIds = $stPre->fetchAll(PDO::FETCH_COLUMN, 0) ?: [];

        if (!empty($preIds)) {
            $sqlRows = "
                SELECT d.id AS pedido_dest_id,
                       d.destinatario_id,
                       dest.razon_social AS destinatario,
                       sp.uv_cajas,
                       sp.uc_unidades
                  FROM so_pre_pick sp
                  JOIN so_pedido_dest_item i ON i.id = sp.pedido_dest_item_id
                  JOIN so_pedido_dest d ON d.id = i.pedido_dest_id
             LEFT JOIN para_destinatarios dest ON dest.id = d.destinatario_id
                 WHERE sp.preembarque_id = ?
            ";
            $stRows = $pdo->prepare($sqlRows);
            foreach ($preIds as $preId) {
                $stRows->execute([(int)$preId]);
                while ($row = $stRows->fetch(PDO::FETCH_ASSOC)) {
                    $destId = isset($row['destinatario_id']) ? (int)$row['destinatario_id'] : 0;
                    $pedidoId = isset($row['pedido_dest_id']) ? (int)$row['pedido_dest_id'] : 0;
                    if ($pedidoId > 0) {
                        $pedidoDestIds[$pedidoId] = true;
                    }
                    if ($destId <= 0) {
                        continue;
                    }
                    if (!isset($byDest[$destId])) {
                        $byDest[$destId] = [
                            'destinatario_id' => $destId,
                            'destinatario' => isset($row['destinatario']) ? (string)$row['destinatario'] : null,
                            'uv' => 0,
                            'uc' => 0,
                            'pedido_dest_ids' => [],
                        ];
                    }
                    $byDest[$destId]['uv'] += (int)($row['uv_cajas'] ?? 0);
                    $byDest[$destId]['uc'] += (int)($row['uc_unidades'] ?? 0);
                    if ($pedidoId > 0) {
                        $byDest[$destId]['pedido_dest_ids'][$pedidoId] = true;
                    }
                }
            }
        }

        // Meta por pedido_dest para asegurar asociación con destinatario
        $pedidoMeta = embarque_fetch_pedido_meta($pdo, array_keys($pedidoDestIds));
        foreach ($pedidoMeta as $pid => $meta) {
            $destId = $meta['destinatario_id'] ?? 0;
            if ($destId > 0) {
                if (!isset($byDest[$destId])) {
                    $byDest[$destId] = [
                        'destinatario_id' => $destId,
                        'destinatario' => $meta['destinatario'] ?? null,
                        'uv' => 0,
                        'uc' => 0,
                        'pedido_dest_ids' => [],
                    ];
                } elseif (empty($byDest[$destId]['destinatario'])) {
                    $byDest[$destId]['destinatario'] = $meta['destinatario'] ?? null;
                }
                $byDest[$destId]['pedido_dest_ids'][$pid] = true;
            }
        }

        if (empty($byDest)) {
            try {
                $fallback = $pdo->prepare("SELECT s.destinatario_id, d.razon_social FROM so_embarque_seguimiento_dest s JOIN para_destinatarios d ON d.id = s.destinatario_id WHERE s.embarque_id = ? GROUP BY s.destinatario_id, d.razon_social");
                $fallback->execute([$embarqueId]);
                while ($row = $fallback->fetch(PDO::FETCH_ASSOC)) {
                    $destId = isset($row['destinatario_id']) ? (int)$row['destinatario_id'] : 0;
                    if ($destId <= 0) {
                        continue;
                    }
                    if (!isset($byDest[$destId])) {
                        $byDest[$destId] = [
                            'destinatario_id' => $destId,
                            'destinatario' => isset($row['razon_social']) ? (string)$row['razon_social'] : null,
                            'uv' => 0,
                            'uc' => 0,
                            'pedido_dest_ids' => [],
                        ];
                    }
                }
            } catch (Throwable $ignored) {
                // Tabla puede no existir aún; ignorar
            }
        }

        $pedidoIdList = array_keys($pedidoDestIds);
        $docsByPedido = embarque_collect_pedido_docs($pdo, $pedidoIdList);
        $pedidoItems = embarque_collect_pedido_items($pdo, $pedidoIdList);

        return [
            'destinatarios' => $byDest,
            'pedido_docs' => $docsByPedido,
            'pedido_meta' => $pedidoMeta,
            'pedido_items' => $pedidoItems,
        ];
    }
}
