<?php

declare(strict_types=1);

if (!function_exists('inventario_is_local_env')) {
    function inventario_is_local_env(): bool
    {
        $env = $_ENV['APP_ENV'] ?? getenv('APP_ENV');
        if (!is_string($env)) {
            return false;
        }

        $env = strtolower(trim($env));
        return in_array($env, ['local', 'development', 'dev'], true);
    }
}

if (!function_exists('inventario_has_table')) {
    function inventario_has_table(PDO $pdo, string $name): bool
    {
        static $cache = [];
        if (array_key_exists($name, $cache)) {
            return $cache[$name];
        }

        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$name]);
        $cache[$name] = ((int) $stmt->fetchColumn() > 0);

        return $cache[$name];
    }
}

if (!function_exists('inventario_has_column')) {
    function inventario_has_column(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = $table . ':' . $column;

        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }

        if (!inventario_has_table($pdo, $table)) {
            return $cache[$key] = false;
        }

        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        $cache[$key] = ((int) $stmt->fetchColumn() > 0);

        return $cache[$key];
    }
}

if (!function_exists('inventario_detect_column')) {
    function inventario_detect_column(PDO $pdo, string $table, array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            if (inventario_has_column($pdo, $table, $candidate)) {
                return $candidate;
            }
        }

        return null;
    }
}

if (!function_exists('inventario_normalize_row')) {
    function inventario_normalize_row(array $row): array
    {
        $row['producto_id'] = isset($row['producto_id']) ? (int) $row['producto_id'] : (isset($row['product_id']) ? (int) $row['product_id'] : 0);
        $cliente = trim((string) ($row['cliente'] ?? ''));
        $operativa = trim((string) ($row['operativa'] ?? ''));

        $row['cliente'] = $cliente !== '' ? $cliente : '—';
        $row['operativa'] = $operativa !== '' ? $operativa : '—';
        $row['sku'] = trim((string) ($row['sku'] ?? ''));
        $row['denominacion'] = trim((string) ($row['denominacion'] ?? ($row['descripcion'] ?? '')));

        $row['pallets'] = (float) ($row['pallets'] ?? 0);
        $row['cajas'] = (float) ($row['cajas'] ?? 0);

        $sueltas = 0.0;
        if (array_key_exists('unidades_sueltas', $row)) {
            $sueltas = (float) $row['unidades_sueltas'];
        } elseif (array_key_exists('uc', $row)) {
            $sueltas = (float) $row['uc'];
        } elseif (array_key_exists('uc_sueltas', $row)) {
            $sueltas = (float) $row['uc_sueltas'];
        } elseif (array_key_exists('uc_unidades', $row)) {
            $sueltas = (float) $row['uc_unidades'];
        }

        $row['unidades_sueltas'] = $sueltas;
        $row['uc'] = $sueltas;

        $row['unidades'] = (float) ($row['unidades'] ?? ($row['stock'] ?? 0));
        if ($row['unidades'] <= 0 && isset($row['stock'])) {
            $row['unidades'] = (float) $row['stock'];
        }

        $row['stock'] = (float) ($row['stock'] ?? $row['unidades']);
        $row['reservados'] = (float) ($row['reservados'] ?? 0);

        if (!array_key_exists('disponibles', $row)) {
            $row['disponibles'] = $row['stock'] - $row['reservados'];
        }
        $row['disponibles'] = max(0.0, (float) ($row['disponibles'] ?? ($row['stock'] - $row['reservados'])));

        return $row;
    }
}

if (!function_exists('inventario_compute_totals')) {
    function inventario_compute_totals(array $rows): array
    {
        $totals = [
            'productos'   => 0,
            'clientes'    => 0,
            'operativas'  => 0,
            'pallets'          => 0.0,
            'cajas'            => 0.0,
            'unidades_sueltas' => 0.0,
            'unidades'         => 0.0,
            'stock'            => 0.0,
            'reservados'       => 0.0,
            'disponibles'      => 0.0,
        ];

        if (!$rows) {
            return $totals;
        }

        $clientes = [];
        $operativas = [];

        foreach ($rows as $row) {
            $cliente = (string) ($row['cliente'] ?? '');
            $operativa = (string) ($row['operativa'] ?? '');

            if ($cliente !== '' && $cliente !== '—') {
                $clientes[$cliente] = true;
            }
            if ($operativa !== '' && $operativa !== '—') {
                $operativas[$operativa] = true;
            }

            $totals['pallets']          += (float) ($row['pallets'] ?? 0);
            $totals['cajas']            += (float) ($row['cajas'] ?? 0);
            $totals['unidades_sueltas'] += (float) ($row['unidades_sueltas'] ?? ($row['uc'] ?? 0));
            $totals['unidades']         += (float) ($row['unidades'] ?? ($row['stock'] ?? 0));
            $totals['stock']            += (float) ($row['stock'] ?? 0);
            $totals['reservados']       += (float) ($row['reservados'] ?? 0);
            $totals['disponibles']      += (float) ($row['disponibles'] ?? (($row['stock'] ?? 0) - ($row['reservados'] ?? 0)));
        }

        $totals['productos']   = count($rows);
        $totals['clientes']    = count($clientes);
        $totals['operativas']  = count($operativas);

        return $totals;
    }
}

if (!function_exists('inventario_filter_rows')) {
    function inventario_filter_rows(array $rows, string $searchValue, array $fields): array
    {
        $searchValue = trim($searchValue);
        if ($searchValue === '') {
            return array_values($rows);
        }

        return array_values(array_filter($rows, static function (array $row) use ($fields, $searchValue): bool {
            foreach ($fields as $field) {
                if (!array_key_exists($field, $row)) {
                    continue;
                }

                $value = (string) $row[$field];
                if ($value === '') {
                    continue;
                }

                if (function_exists('mb_stripos')) {
                    if (mb_stripos($value, $searchValue) !== false) {
                        return true;
                    }
                } elseif (stripos($value, $searchValue) !== false) {
                    return true;
                }
            }

            return false;
        }));
    }
}

if (!function_exists('inventario_order_rows')) {
    function inventario_order_rows(array $rows, array $orders, array $columnMap, array $numericFields = []): array
    {
        if (!$orders) {
            return array_values($rows);
        }

        $criteria = [];
        foreach ($orders as $order) {
            $index = isset($order['column']) ? (int) $order['column'] : null;
            if ($index === null || !isset($columnMap[$index])) {
                continue;
            }

            $field = $columnMap[$index];
            if ($field === null) {
                continue;
            }

            $dir = strtolower((string) ($order['dir'] ?? 'asc')) === 'desc' ? 'desc' : 'asc';
            $criteria[] = ['field' => $field, 'dir' => $dir];
        }

        if (!$criteria) {
            return array_values($rows);
        }

        $numericLookup = array_flip($numericFields);

        usort($rows, static function (array $a, array $b) use ($criteria, $numericLookup): int {
            foreach ($criteria as $criterion) {
                $field = $criterion['field'];
                $dir = $criterion['dir'];

                $va = $a[$field] ?? null;
                $vb = $b[$field] ?? null;

                if (isset($numericLookup[$field])) {
                    $va = (float) $va;
                    $vb = (float) $vb;
                } else {
                    $va = mb_strtolower((string) $va);
                    $vb = mb_strtolower((string) $vb);
                }

                if ($va < $vb) {
                    return $dir === 'desc' ? 1 : -1;
                }
                if ($va > $vb) {
                    return $dir === 'desc' ? -1 : 1;
                }
            }

            return 0;
        });

        return array_values($rows);
    }
}

if (!function_exists('inventario_slice_rows')) {
    function inventario_slice_rows(array $rows, int $start, int $length): array
    {
        if ($length < 0) {
            return array_values($rows);
        }

        return array_slice(array_values($rows), $start, $length);
    }
}

if (!function_exists('inventario_existencias_dataset')) {
    function inventario_existencias_dataset(PDO $pdo): array
    {
        $useNewSchema = inventario_has_table($pdo, 'wh_pallets') && inventario_has_table($pdo, 'wh_pallet_items');

        if ($useNewSchema) {
            $hasUcTotal    = inventario_has_column($pdo, 'wh_pallet_items', 'uc_total_cache');
            $hasUcPorCaja  = inventario_has_column($pdo, 'wh_pallet_items', 'uc_por_caja');
            $hasUcSueltas  = inventario_has_column($pdo, 'wh_pallet_items', 'uc_sueltas');
            $hasUcUnidades = inventario_has_column($pdo, 'wh_pallet_items', 'uc_unidades');
            $hasUvCajas    = inventario_has_column($pdo, 'wh_pallet_items', 'uv_cajas');
            $hasEstado     = inventario_has_column($pdo, 'wh_pallet_items', 'estado');
            $hasReservaFlg = inventario_has_column($pdo, 'wh_pallets', 'reservado');

            if ($hasUcTotal) {
                $fallbackParts = [];
                if ($hasUcPorCaja && $hasUvCajas) {
                    $fallbackParts[] = 'COALESCE(it.uc_por_caja,0) * COALESCE(it.uv_cajas,0)';
                }
                if ($hasUcSueltas) {
                    $fallbackParts[] = 'COALESCE(it.uc_sueltas,0)';
                }
                if (!$fallbackParts && $hasUcUnidades) {
                    $fallbackParts[] = 'COALESCE(it.uc_unidades,0)';
                }
                if (!$fallbackParts) {
                    $fallbackParts[] = '0';
                }
                $unitExpr = 'COALESCE(it.uc_total_cache, ' . implode(' + ', $fallbackParts) . ')';
            } elseif ($hasUcUnidades) {
                $unitExpr = 'COALESCE(it.uc_unidades,0)';
            } else {
                $parts = [];
                if ($hasUcPorCaja && $hasUvCajas) {
                    $parts[] = 'COALESCE(it.uc_por_caja,0) * COALESCE(it.uv_cajas,0)';
                }
                if ($hasUcSueltas) {
                    $parts[] = 'COALESCE(it.uc_sueltas,0)';
                }
                $unitExpr = $parts ? implode(' + ', $parts) : '0';
            }

            $cajasExpr = $hasUvCajas ? 'COALESCE(it.uv_cajas,0)' : '0';

            $sueltasExpr = '0';
            if ($hasUcSueltas) {
                $sueltasExpr = 'COALESCE(it.uc_sueltas,0)';
            } elseif ($hasUcUnidades) {
                $sueltasExpr = 'COALESCE(it.uc_unidades,0)';
            }

            $reservadoExpr = '0';
            if ($hasEstado) {
                $reservadoExpr = "CASE WHEN it.estado = 'RESERVADO' THEN {$unitExpr} ELSE 0 END";
            } elseif ($hasReservaFlg) {
                $reservadoExpr = "CASE WHEN pa.reservado = 1 THEN {$unitExpr} ELSE 0 END";
            }

            $sqlParts = [
                'SELECT',
                '    pr.id                        AS producto_id,',
                '    pr.sku,',
                '    pr.denominacion,',
                "    COALESCE(c.razon_social, '') AS cliente,",
                "    COALESCE(opv.nombre, '')     AS operativa,",
                '    COUNT(DISTINCT pa.id)        AS pallets,',
                "    SUM({$cajasExpr})            AS cajas,",
                "    SUM({$sueltasExpr})          AS unidades_sueltas,",
                "    SUM({$unitExpr})             AS unidades,",
                "    SUM({$unitExpr})             AS stock,",
                "    SUM({$reservadoExpr})        AS reservados,",
                "    SUM({$unitExpr}) - SUM({$reservadoExpr}) AS disponibles",
                'FROM wh_pallets pa',
                'JOIN wh_pallet_items it ON it.pallet_id = pa.id',
                'JOIN para_productos pr  ON pr.id = it.producto_id',
                'LEFT JOIN para_clientes c    ON c.id = pr.cliente_id',
                'LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id',
            ];

            if (inventario_has_table($pdo, 'wh_lote')) {
                $sqlParts[] = 'LEFT JOIN wh_lote l ON l.id = it.lote_id';
            }

            $sqlParts[] = 'WHERE 1=1';
            $sqlParts[] = 'GROUP BY pr.id, pr.sku, pr.denominacion, cliente, operativa';

            $sql = implode("\n", $sqlParts);
        } else {
            $stockExpr = 'COALESCE(s.qty_uc,0) + COALESCE(s.qty_uv,0)';
            $packJoin = '';

            if (inventario_has_table($pdo, 'para_producto_pack') && inventario_has_column($pdo, 'para_producto_pack', 'unidades_por_uv')) {
                $packJoin = implode("\n", [
                    'LEFT JOIN (',
                    '    SELECT producto_id, MAX(unidades_por_uv) AS unidades_por_uv',
                    '    FROM para_producto_pack',
                    '    GROUP BY producto_id',
                    ') ppu ON ppu.producto_id = pr.id',
                ]);
                $stockExpr = 'COALESCE(s.qty_uc,0) + COALESCE(s.qty_uv,0) * COALESCE(ppu.unidades_por_uv,0)';
            }

            $sqlParts = [
                'SELECT',
                '    pr.id                        AS producto_id,',
                '    pr.sku,',
                '    pr.denominacion,',
                "    COALESCE(c.razon_social, '') AS cliente,",
                "    COALESCE(opv.nombre, '')     AS operativa,",
                '    COUNT(DISTINCT s.pallet_id)  AS pallets,',
                '    SUM(COALESCE(s.qty_uv,0))    AS cajas,',
                '    SUM(COALESCE(s.qty_uc,0))    AS unidades_sueltas,',
                "    SUM({$stockExpr})            AS unidades,",
                "    SUM({$stockExpr})            AS stock,",
                '    0                             AS reservados,',
                "    SUM({$stockExpr})            AS disponibles",
                'FROM wh_stock s',
                'JOIN para_productos pr  ON pr.id = s.producto_id',
                'LEFT JOIN para_clientes c    ON c.id = pr.cliente_id',
                'LEFT JOIN sys_operativas opv ON opv.id = pr.operativa_id',
            ];

            if ($packJoin !== '') {
                $sqlParts[] = $packJoin;
            }

            $sqlParts[] = 'GROUP BY pr.id, pr.sku, pr.denominacion, cliente, operativa';

            $sql = implode("\n", $sqlParts);
        }

        try {
            $stmt = $pdo->query($sql);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        } catch (Throwable $e) {
            return [
                'ok'     => false,
                'schema' => $useNewSchema ? 'new' : 'legacy',
                'error'  => $e->getMessage(),
            ];
        }

        $normalized = array_map('inventario_normalize_row', $rows);

        return [
            'ok'     => true,
            'schema' => $useNewSchema ? 'new' : 'legacy',
            'rows'   => $normalized,
            'totals' => inventario_compute_totals($normalized),
        ];
    }
}

if (!function_exists('inventario_sanitize_date')) {
    function inventario_sanitize_date(?string $value): string
    {
        if ($value === null) {
            return '';
        }

        $value = trim($value);
        if ($value === '') {
            return '';
        }

        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return '';
        }

        return $value;
    }
}

if (!function_exists('inventario_movimientos_compute_totals')) {
    function inventario_movimientos_compute_totals(array $rows): array
    {
        $totals = [
            'productos'                        => 0,
            'saldo_inicial'                    => 0.0,
            'saldo_inicial_pallets'            => 0.0,
            'saldo_inicial_cajas'              => 0.0,
            'saldo_inicial_unidades_sueltas'   => 0.0,
            'ingresos'                         => 0.0,
            'ingresos_pallets'                 => 0.0,
            'ingresos_cajas'                   => 0.0,
            'ingresos_unidades_sueltas'        => 0.0,
            'salidas'                          => 0.0,
            'salidas_pallets'                  => 0.0,
            'salidas_cajas'                    => 0.0,
            'salidas_unidades_sueltas'         => 0.0,
            'saldo_final'                      => 0.0,
            'saldo_final_pallets'              => 0.0,
            'saldo_final_cajas'                => 0.0,
            'saldo_final_unidades_sueltas'     => 0.0,
            'inventario_fisico'                => 0.0,
            'inventario_fisico_pallets'        => 0.0,
            'inventario_fisico_cajas'          => 0.0,
            'inventario_fisico_unidades_sueltas' => 0.0,
            'diferencia_unidades'              => 0.0,
            'valor_unidades'                   => 0.0,
            'valor_diferencia'                 => 0.0,
        ];

        $optionalFields = [
            'saldo_inicial_pallets',
            'saldo_inicial_cajas',
            'saldo_inicial_unidades_sueltas',
            'ingresos_pallets',
            'ingresos_cajas',
            'ingresos_unidades_sueltas',
            'salidas_pallets',
            'salidas_cajas',
            'salidas_unidades_sueltas',
            'saldo_final_pallets',
            'saldo_final_cajas',
            'saldo_final_unidades_sueltas',
            'inventario_fisico',
            'inventario_fisico_pallets',
            'inventario_fisico_cajas',
            'inventario_fisico_unidades_sueltas',
            'diferencia_unidades',
            'valor_diferencia',
        ];
        if (!$rows) {
            foreach ($optionalFields as $field) {
                $totals[$field] = null;
            }
            return $totals;
        }

        $totals['productos'] = count($rows);
        $optionalPresence = array_fill_keys($optionalFields, false);

        foreach ($rows as $row) {
            $totals['saldo_inicial'] += (float) ($row['saldo_inicial'] ?? 0);
            $totals['ingresos']      += (float) ($row['ingresos'] ?? 0);
            $totals['salidas']       += (float) ($row['salidas'] ?? 0);
            $totals['saldo_final']   += (float) ($row['saldo_final'] ?? 0);

            foreach (['saldo_inicial_pallets', 'saldo_inicial_cajas', 'saldo_inicial_unidades_sueltas', 'ingresos_pallets', 'ingresos_cajas', 'ingresos_unidades_sueltas', 'salidas_pallets', 'salidas_cajas', 'salidas_unidades_sueltas', 'saldo_final_pallets', 'saldo_final_cajas', 'saldo_final_unidades_sueltas', 'inventario_fisico', 'inventario_fisico_pallets', 'inventario_fisico_cajas', 'inventario_fisico_unidades_sueltas', 'diferencia_unidades', 'valor_diferencia'] as $field) {
                if (array_key_exists($field, $row) && $row[$field] !== null) {
                    $totals[$field] += (float) $row[$field];
                    $optionalPresence[$field] = true;
                }
            }

            if ($row['valor_unidades'] !== null) {
                $totals['valor_unidades'] += (float) $row['valor_unidades'];
            }
        }

        foreach ($optionalPresence as $field => $present) {
            if (!$present) {
                $totals[$field] = null;
            }
        }

        return $totals;
    }
}

if (!function_exists('inventario_fetch_units_from_pallet_items')) {
    function inventario_fetch_units_from_pallet_items(PDO $pdo, array $productIds): array
    {
        $productIds = array_values(array_unique(array_map('intval', $productIds)));
        if (!$productIds) {
            return [];
        }

        if (!inventario_has_table($pdo, 'wh_pallet_items') || !inventario_has_column($pdo, 'wh_pallet_items', 'uc_por_caja')) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($productIds), '?'));
        $sql = "SELECT producto_id, MAX(COALESCE(uc_por_caja,0)) AS unidades FROM wh_pallet_items WHERE producto_id IN ($placeholders) GROUP BY producto_id";
        $stmt = $pdo->prepare($sql);
        foreach ($productIds as $index => $pid) {
            $stmt->bindValue($index + 1, $pid, PDO::PARAM_INT);
        }
        $stmt->execute();

        $result = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
            $pid = (int) ($row['producto_id'] ?? 0);
            if ($pid <= 0) {
                continue;
            }
            $result[$pid] = (float) ($row['unidades'] ?? 0);
        }

        return $result;
    }
}

if (!function_exists('inventario_fetch_cases_per_pallet_from_pallet_items')) {
    function inventario_fetch_cases_per_pallet_from_pallet_items(PDO $pdo, array $productIds): array
    {
        $productIds = array_values(array_unique(array_map('intval', $productIds)));
        if (!$productIds) {
            return [];
        }

        if (!inventario_has_table($pdo, 'wh_pallet_items') || !inventario_has_column($pdo, 'wh_pallet_items', 'uv_cajas')) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($productIds), '?'));
        $sql = "SELECT producto_id, MAX(COALESCE(uv_cajas,0)) AS cajas FROM wh_pallet_items WHERE producto_id IN ($placeholders) GROUP BY producto_id";
        $stmt = $pdo->prepare($sql);
        foreach ($productIds as $index => $pid) {
            $stmt->bindValue($index + 1, $pid, PDO::PARAM_INT);
        }
        $stmt->execute();

        $result = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
            $pid = (int) ($row['producto_id'] ?? 0);
            if ($pid <= 0) {
                continue;
            }
            $result[$pid] = (float) ($row['cajas'] ?? 0);
        }

        return $result;
    }
}

if (!function_exists('inventario_fetch_product_info')) {
    function inventario_fetch_product_info(PDO $pdo, array $productIds): array
    {
        $productIds = array_values(array_unique(array_map('intval', $productIds)));
        if (!$productIds) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($productIds), '?'));

        $skuCol = inventario_detect_column($pdo, 'para_productos', ['sku', 'codigo', 'cod', 'sku_cliente']);
        $descCol = inventario_detect_column($pdo, 'para_productos', ['denominacion', 'descripcion', 'nombre', 'detalle']);
        $unitsCol = inventario_detect_column($pdo, 'para_productos', ['uc_por_caja', 'unidades_por_caja', 'units_per_case', 'units_per_box']);
        $casesCol = inventario_detect_column($pdo, 'para_productos', ['cajas_por_pallet', 'uv_por_pallet', 'cajas_pallet', 'cajas_x_pallet', 'cajasxpallet', 'cajas_por_paleta', 'caja_por_pallet']);
        $costCol = inventario_detect_column($pdo, 'para_productos', ['costo_unitario', 'costo_promedio', 'valor_unitario', 'precio_unitario', 'costo']);

        $skuExpr = $skuCol ? sprintf('pr.`%s`', $skuCol) : 'CAST(pr.id AS CHAR)';
        $descExpr = $descCol ? sprintf('pr.`%s`', $descCol) : $skuExpr;
        $unitsExpr = $unitsCol ? sprintf('pr.`%s`', $unitsCol) : null;
        $casesExpr = $casesCol ? sprintf('pr.`%s`', $casesCol) : null;
        $costExpr = $costCol ? sprintf('pr.`%s`', $costCol) : null;

        $select = [
            'pr.id AS producto_id',
            $skuExpr . ' AS sku',
            $descExpr . ' AS descripcion',
            $unitsExpr ? $unitsExpr . ' AS unidades_por_caja' : 'NULL AS unidades_por_caja',
            $casesExpr ? $casesExpr . ' AS cajas_por_pallet' : 'NULL AS cajas_por_pallet',
            $costExpr ? $costExpr . ' AS costo_unitario' : 'NULL AS costo_unitario',
        ];

        $sql = 'SELECT ' . implode(', ', $select) . ' FROM para_productos pr WHERE pr.id IN (' . $placeholders . ')';
        $stmt = $pdo->prepare($sql);
        foreach ($productIds as $index => $pid) {
            $stmt->bindValue($index + 1, $pid, PDO::PARAM_INT);
        }
        $stmt->execute();

        $info = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
            $pid = (int) ($row['producto_id'] ?? 0);
            if ($pid <= 0) {
                continue;
            }
            $info[$pid] = [
                'sku' => (string) ($row['sku'] ?? ''),
                'descripcion' => (string) ($row['descripcion'] ?? ''),
                'unidades_por_caja' => $row['unidades_por_caja'] !== null ? (float) $row['unidades_por_caja'] : null,
                'cajas_por_pallet' => $row['cajas_por_pallet'] !== null ? (float) $row['cajas_por_pallet'] : null,
                'costo_unitario' => $row['costo_unitario'] !== null ? (float) $row['costo_unitario'] : null,
            ];
        }

        $missingUnits = [];
        $missingCases = [];
        foreach ($productIds as $pid) {
            if (!isset($info[$pid])) {
                $info[$pid] = [
                    'sku' => '',
                    'descripcion' => '',
                    'unidades_por_caja' => null,
                    'cajas_por_pallet' => null,
                    'costo_unitario' => null,
                ];
            }
            if ($info[$pid]['unidades_por_caja'] === null) {
                $missingUnits[] = $pid;
            }
            if ($info[$pid]['cajas_por_pallet'] === null) {
                $missingCases[] = $pid;
            }
        }

        if ($missingUnits) {
            $units = inventario_fetch_units_from_pallet_items($pdo, $missingUnits);
            foreach ($missingUnits as $pid) {
                if (isset($units[$pid]) && $units[$pid] > 0) {
                    $info[$pid]['unidades_por_caja'] = $units[$pid];
                }
            }
        }

        if ($missingCases) {
            $cases = inventario_fetch_cases_per_pallet_from_pallet_items($pdo, $missingCases);
            foreach ($missingCases as $pid) {
                if (isset($cases[$pid]) && $cases[$pid] > 0) {
                    $info[$pid]['cajas_por_pallet'] = $cases[$pid];
                }
            }
        }

        return $info;
    }
}

if (!function_exists('inventario_breakdown_units')) {
    function inventario_breakdown_units(float $units, ?float $cajasPorPallet, ?float $unidadesPorCaja): array
    {
        $total = (float) $units;
        $sign = $total < 0 ? -1 : 1;
        $remaining = abs($total);

        $pallets = null;
        $cajas = null;

        if ($cajasPorPallet !== null && $cajasPorPallet > 0 && $unidadesPorCaja !== null && $unidadesPorCaja > 0) {
            $unitsPerPallet = $cajasPorPallet * $unidadesPorCaja;
            if ($unitsPerPallet > 0) {
                $palletCount = (int) floor(($remaining / $unitsPerPallet) + 1e-9);
                $remaining -= $palletCount * $unitsPerPallet;
                $pallets = $sign * $palletCount;
            }
        }

        if ($unidadesPorCaja !== null && $unidadesPorCaja > 0) {
            $cajaCount = (int) floor(($remaining / $unidadesPorCaja) + 1e-9);
            $remaining -= $cajaCount * $unidadesPorCaja;
            $cajas = $sign * $cajaCount;
        }

        $unidadesSueltas = $sign * round($remaining);

        if ($pallets !== null && abs($pallets) < 1e-9) {
            $pallets = 0.0;
        }
        if ($cajas !== null && abs($cajas) < 1e-9) {
            $cajas = 0.0;
        }

        if ($unidadesPorCaja === null || $unidadesPorCaja <= 0) {
            $unidadesSueltas = $total * 1.0;
        }

        if (abs($unidadesSueltas) < 1e-9) {
            $unidadesSueltas = 0.0;
        }

        return [
            'pallets' => $pallets,
            'cajas' => $cajas,
            'unidades_sueltas' => $unidadesSueltas,
            'total_unidades' => $total,
        ];
    }
}

if (!function_exists('inventario_movements_map')) {
    function inventario_movements_map(PDO $pdo, string $desde, string $hasta): array
    {
        $map = [];
        $params = [];
        $where = [];

        $timeColumns = [];
        foreach (['finalizado_at', 'iniciado_at', 'asignado_at', 'created_at'] as $col) {
            if (inventario_has_column($pdo, 'wh_moves', $col)) {
                $timeColumns[] = "m.`$col`";
            }
        }
        $timestampExpr = $timeColumns ? ('COALESCE(' . implode(', ', $timeColumns) . ')') : 'm.created_at';

        if ($desde !== '') {
            $where[] = 'DATE(' . $timestampExpr . ') >= :desde';
            $params[':desde'] = $desde;
        }
        if ($hasta !== '') {
            $where[] = 'DATE(' . $timestampExpr . ') <= :hasta';
            $params[':hasta'] = $hasta;
        }
        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $hasPluralMoves = inventario_has_table($pdo, 'wh_moves') && inventario_has_table($pdo, 'wh_move_items') && inventario_has_table($pdo, 'wh_pallet_items');
        $hasLegacyMoves = inventario_has_table($pdo, 'wh_move');

        if ($hasPluralMoves) {
            $hasMiUc = inventario_has_column($pdo, 'wh_move_items', 'uc_unidades');
            $hasMiUv = inventario_has_column($pdo, 'wh_move_items', 'uv_cajas');
            $hasPiPor = inventario_has_column($pdo, 'wh_pallet_items', 'uc_por_caja');
            $hasPiTotal = inventario_has_column($pdo, 'wh_pallet_items', 'uc_total_cache');
            $hasPiSueltas = inventario_has_column($pdo, 'wh_pallet_items', 'uc_sueltas');

            $unitFallback = [];
            if ($hasPiPor && $hasMiUv) {
                $unitFallback[] = 'COALESCE(mi.uv_cajas,0) * COALESCE(pi.uc_por_caja,0)';
            } elseif ($hasMiUv) {
                $unitFallback[] = 'COALESCE(mi.uv_cajas,0)';
            }
            if ($hasPiTotal) {
                $unitFallback[] = 'COALESCE(pi.uc_total_cache,0)';
            } elseif ($hasPiSueltas) {
                $unitFallback[] = 'COALESCE(pi.uc_sueltas,0)';
            }

            $unitExprParts = [];
            if ($hasMiUc) {
                $unitExprParts[] = 'mi.uc_unidades';
            }
            $unitExprParts = array_merge($unitExprParts, $unitFallback);
            if (!$unitExprParts) {
                $unitExprParts[] = '0';
            }
            $unitExpr = 'COALESCE(' . implode(', ', $unitExprParts) . ', 0)';

            $inboundConditions = [];
            if (inventario_has_column($pdo, 'wh_moves', 'move_type')) {
                $inboundConditions[] = "m.move_type IN ('INBOUND','IN','ADJUSTMENT_IN','ADJUSTMENT')";
            }
            if (inventario_has_column($pdo, 'wh_moves', 'tipo')) {
                $inboundConditions[] = "m.tipo IN ('INGRESO','ENTRADA','AJUSTE_IN','AJUSTE_ALTA','AJUSTE')";
            }
                $inboundConditions[] = '(m.desde_position_id IS NULL AND m.hasta_position_id IS NOT NULL)';
                $inboundConditions[] = '(m.tipo = \'AJUSTE\' AND m.hasta_position_id IS NOT NULL AND m.desde_position_id IS NULL)';
            $inboundCondition = '(' . implode(' OR ', $inboundConditions) . ')';

            $outboundConditions = [];
            if (inventario_has_column($pdo, 'wh_moves', 'move_type')) {
                $outboundConditions[] = "m.move_type IN ('OUTBOUND','OUT','ADJUSTMENT_OUT')";
            }
            if (inventario_has_column($pdo, 'wh_moves', 'tipo')) {
                $outboundConditions[] = "m.tipo IN ('BAJA','SALIDA','AJUSTE_OUT','AJUSTE_BAJA','AJUSTE')";
            }
                $outboundConditions[] = '(m.desde_position_id IS NOT NULL AND m.hasta_position_id IS NULL)';
                $outboundConditions[] = '(m.tipo = \'AJUSTE\' AND m.desde_position_id IS NOT NULL AND m.hasta_position_id IS NULL)';
            $outboundCondition = '(' . implode(' OR ', $outboundConditions) . ')';

            $sql = "
                SELECT
                    pi.producto_id AS producto_id,
                    SUM(CASE WHEN {$inboundCondition} THEN {$unitExpr} ELSE 0 END) AS ingresos,
                    SUM(CASE WHEN {$outboundCondition} THEN {$unitExpr} ELSE 0 END) AS salidas
                FROM wh_moves m
                JOIN wh_move_items mi ON mi.move_id = m.id
                JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
                {$whereSql}
                GROUP BY pi.producto_id
            ";

            $stmt = $pdo->prepare($sql);
            foreach ($params as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            $stmt->execute();
            foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
                $pid = (int) ($row['producto_id'] ?? 0);
                if ($pid <= 0) {
                    continue;
                }
                $map[$pid] = [
                    'ingresos' => (float) ($row['ingresos'] ?? 0),
                    'salidas'  => (float) ($row['salidas'] ?? 0),
                ];
            }
        } elseif ($hasLegacyMoves) {
            $primaryExpr = inventario_has_column($pdo, 'wh_move', 'delta_uc') ? 'COALESCE(m.delta_uc,0)' : '0';
            $fallbackExpr = inventario_has_column($pdo, 'wh_move', 'delta_uv') ? 'COALESCE(m.delta_uv,0)' : '0';

            $positiveExpr = "CASE WHEN {$primaryExpr} > 0 THEN {$primaryExpr} WHEN {$primaryExpr} = 0 AND {$fallbackExpr} > 0 THEN {$fallbackExpr} ELSE 0 END";
            $negativeExpr = "CASE WHEN {$primaryExpr} < 0 THEN ABS({$primaryExpr}) WHEN {$primaryExpr} = 0 AND {$fallbackExpr} < 0 THEN ABS({$fallbackExpr}) ELSE 0 END";

            $ingExpr = "CASE WHEN m.tipo = 'IN' THEN ABS({$primaryExpr}) ELSE {$positiveExpr} END";
            $outExpr = "CASE WHEN m.tipo = 'OUT' THEN ABS({$primaryExpr}) ELSE {$negativeExpr} END";

            $timeExpr = inventario_has_column($pdo, 'wh_move', 'created_at') ? 'DATE(m.created_at)' : null;
            $whereLegacy = [];
            $legacyParams = [];
            if ($timeExpr !== null) {
                if ($desde !== '') {
                    $whereLegacy[] = $timeExpr . ' >= :desde';
                    $legacyParams[':desde'] = $desde;
                }
                if ($hasta !== '') {
                    $whereLegacy[] = $timeExpr . ' <= :hasta';
                    $legacyParams[':hasta'] = $hasta;
                }
            }
            $whereLegacySql = $whereLegacy ? ('WHERE ' . implode(' AND ', $whereLegacy)) : '';

            $sql = "
                SELECT
                    m.producto_id AS producto_id,
                    SUM({$ingExpr}) AS ingresos,
                    SUM({$outExpr}) AS salidas
                FROM wh_move m
                {$whereLegacySql}
                GROUP BY m.producto_id
            ";

            $stmt = $pdo->prepare($sql);
            foreach ($legacyParams as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            $stmt->execute();

            foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
                $pid = (int) ($row['producto_id'] ?? 0);
                if ($pid <= 0) {
                    continue;
                }
                $map[$pid] = [
                    'ingresos' => (float) ($row['ingresos'] ?? 0),
                    'salidas'  => (float) ($row['salidas'] ?? 0),
                ];
            }
        }

        return $map;
    }
}

if (!function_exists('inventario_movimientos_dataset')) {
    function inventario_movimientos_dataset(PDO $pdo, array $filters = []): array
    {
        $desde = inventario_sanitize_date($filters['desde'] ?? '');
        $hasta = inventario_sanitize_date($filters['hasta'] ?? '');

        if ($desde !== '' && $hasta !== '' && $desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        $existencias = inventario_existencias_dataset($pdo);
        $stockMap = [];
        if (($existencias['ok'] ?? false) === true) {
            foreach ($existencias['rows'] ?? [] as $row) {
                $pid = (int) ($row['producto_id'] ?? 0);
                if ($pid <= 0) {
                    continue;
                }
                $stockMap[$pid] = (float) ($row['stock'] ?? 0);
            }
        }

        $movementMap = inventario_movements_map($pdo, $desde, $hasta);

        $productIds = array_values(array_unique(array_merge(array_keys($stockMap), array_keys($movementMap))));
        sort($productIds);

        if (!$productIds) {
            return [
                'ok' => true,
                'rows' => [],
                'totals' => inventario_movimientos_compute_totals([]),
                'meta' => [
                    'filters' => [
                        'desde' => $desde,
                        'hasta' => $hasta,
                    ],
                ],
            ];
        }

        $productInfo = inventario_fetch_product_info($pdo, $productIds);

        $rows = [];
        foreach ($productIds as $pid) {
            $info = $productInfo[$pid] ?? [
                'sku' => '',
                'descripcion' => '',
                'unidades_por_caja' => null,
                'cajas_por_pallet' => null,
                'costo_unitario' => null,
            ];
            $mov = $movementMap[$pid] ?? ['ingresos' => 0.0, 'salidas' => 0.0];
            $final = $stockMap[$pid] ?? 0.0;

            $ingresos = (float) ($mov['ingresos'] ?? 0);
            $salidas = (float) ($mov['salidas'] ?? 0);
            $saldoInicial = $final - $ingresos + $salidas;

            $inventarioFisico = null;
            $diferencia = null;
            if ($inventarioFisico !== null) {
                $diferencia = $inventarioFisico - $final;
            }

            $costoUnitario = $info['costo_unitario'];
            $valorUnidades = ($costoUnitario !== null) ? $final * $costoUnitario : null;
            $valorDiferencia = ($costoUnitario !== null && $diferencia !== null) ? $diferencia * $costoUnitario : null;

            $unidadesPorCaja = $info['unidades_por_caja'] !== null ? (float) $info['unidades_por_caja'] : null;
            $cajasPorPallet = $info['cajas_por_pallet'] !== null ? (float) $info['cajas_por_pallet'] : null;

            $saldoInicialBreak = inventario_breakdown_units($saldoInicial, $cajasPorPallet, $unidadesPorCaja);
            $ingresosBreak = inventario_breakdown_units($ingresos, $cajasPorPallet, $unidadesPorCaja);
            $salidasBreak = inventario_breakdown_units($salidas, $cajasPorPallet, $unidadesPorCaja);
            $saldoFinalBreak = inventario_breakdown_units($final, $cajasPorPallet, $unidadesPorCaja);
            $inventarioFisicoBreak = ($inventarioFisico !== null)
                ? inventario_breakdown_units((float) $inventarioFisico, $cajasPorPallet, $unidadesPorCaja)
                : null;

            $rows[] = [
                'producto_id' => $pid,
                'codigo' => (string) $info['sku'],
                'descripcion' => (string) $info['descripcion'],
                'cajas_por_pallet' => $cajasPorPallet,
                'unidades_por_caja' => $unidadesPorCaja,
                'saldo_inicial_pallets' => $saldoInicialBreak['pallets'],
                'saldo_inicial_cajas' => $saldoInicialBreak['cajas'],
                'saldo_inicial_unidades_sueltas' => $saldoInicialBreak['unidades_sueltas'],
                'saldo_inicial' => $saldoInicialBreak['total_unidades'],
                'ingresos_pallets' => $ingresosBreak['pallets'],
                'ingresos_cajas' => $ingresosBreak['cajas'],
                'ingresos_unidades_sueltas' => $ingresosBreak['unidades_sueltas'],
                'ingresos' => $ingresosBreak['total_unidades'],
                'salidas_pallets' => $salidasBreak['pallets'],
                'salidas_cajas' => $salidasBreak['cajas'],
                'salidas_unidades_sueltas' => $salidasBreak['unidades_sueltas'],
                'salidas' => $salidasBreak['total_unidades'],
                'saldo_final_pallets' => $saldoFinalBreak['pallets'],
                'saldo_final_cajas' => $saldoFinalBreak['cajas'],
                'saldo_final_unidades_sueltas' => $saldoFinalBreak['unidades_sueltas'],
                'saldo_final' => $saldoFinalBreak['total_unidades'],
                'inventario_fisico_pallets' => $inventarioFisicoBreak['pallets'] ?? null,
                'inventario_fisico_cajas' => $inventarioFisicoBreak['cajas'] ?? null,
                'inventario_fisico_unidades_sueltas' => $inventarioFisicoBreak['unidades_sueltas'] ?? null,
                'inventario_fisico' => $inventarioFisicoBreak['total_unidades'] ?? $inventarioFisico,
                'diferencia_unidades' => $diferencia,
                'valor_unidades' => $valorUnidades,
                'valor_diferencia' => $valorDiferencia,
            ];
        }

        return [
            'ok' => true,
            'rows' => $rows,
            'totals' => inventario_movimientos_compute_totals($rows),
            'meta' => [
                'filters' => [
                    'desde' => $desde,
                    'hasta' => $hasta,
                ],
            ],
        ];
    }
}
