<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Soporte: app/Support/MenuRouter.php
 *
 * Objetivo:
 *   - Derivar automáticamente el mapa de rutas (path => view, gate) desde config/menu.php.
 *   - Una sola fuente de verdad: tocás config/menu.php y se reflejan sidebar + rutas.
 *
 * Convenciones:
 *   - Todo ítem con 'url' (no-null) se considera una ruta "pública interna" de la app.
 *   - La vista se infiere con la regla:
 *       '/control/dashboard'     => 'views/control/dashboard/index.php'
 *       '/ingresos'              => 'views/ingresos/index.php'
 *       '/parametros/clientes'   => 'views/parametros/clientes/index.php'
 *     Regla general: 'views' . $url . '/index.php' (removiendo la barra inicial).
 *
 * Gate de sesión:
 *   - Por defecto TRUE (toda ruta requiere sesión).
 *   - Si en el futuro querés overrides por ítem, podés agregar 'gate' en config/menu.php
 *     y este archivo lo respetará.
 *
 * Uso recomendado (en public/index.php):
 *   require_once BASE_PATH . '/app/Support/MenuRouter.php';
 *   $routes = route_map_from_menu(); // path => ['view' => ..., 'gate' => bool]
 *   // luego usás $routes[$requestPath] si existe
 *
 * Nota:
 *   - Este archivo no imprime nada y no toca headers. Solo helpers puros.
 */

if (!function_exists('sol_base_path')) {
    /**
     * Retorna la raíz del proyecto de forma robusta.
     */
    function sol_base_path(): string
    {
        // Si tenés BASE_PATH ya definido en config/config.php, úsalo.
        if (defined('BASE_PATH') && is_string(constant('BASE_PATH'))) {
            return (string) constant('BASE_PATH');
        }
        // Fallback: .../app/Support -> raíz del proyecto
        return dirname(__DIR__, 2);
    }
}

if (!function_exists('menu_load')) {
    /**
     * Carga el arreglo del menú desde config/menu.php
     *
     * @return array<int,array<string,mixed>>
     * @throws RuntimeException si el archivo no existe o no retorna array.
     */
    function menu_load(): array
    {
        $file = sol_base_path() . '/config/menu.php';
        if (!is_file($file)) {
            throw new RuntimeException("No se encontró config/menu.php en: {$file}");
        }
        $menu = require $file;
        if (!is_array($menu)) {
            throw new RuntimeException("config/menu.php debe retornar un array válido.");
        }
        return $menu;
    }
}

if (!function_exists('menu_flat')) {
    /**
     * Aplana el menú en una lista de ítems con URL (ignora sections y dropdowns sin url).
     * NOTA: 'visible' solo afecta el sidebar, NO el routing. Todas las rutas con URL son accesibles.
     *
     * @param array<int,array<string,mixed>> $menu
     * @return array<int,array<string,mixed>>
     */
    function menu_flat(array $menu): array
    {
        $acc = [];
        $walk = function (array $items) use (&$acc, &$walk): void {
            foreach ($items as $it) {
                $isSection = isset($it['section']) && $it['section'] === true;
                $url       = $it['url']      ?? null;
                $children  = $it['children'] ?? [];

                // Solo omitimos secciones, no ítems con URLs
                if (!$isSection && is_string($url) && $url !== '') {
                    $acc[] = $it;
                }
                
                // Siempre procesar children, independientemente de 'visible'
                if (is_array($children) && $children) {
                    $walk($children);
                }
            }
        };
        $walk($menu);
        return $acc;
    }
}

if (!function_exists('infer_view_from_url')) {
    /**
     * Infire la ruta de vista a partir del path de la URL (convención views/<url>/index.php).
     *
     * @param string $url Path absoluto tipo '/parametros/clientes'
     * @return string Ruta relativa a BASE_PATH (ej. 'views/parametros/clientes/index.php')
     */
    function infer_view_from_url(string $url): string
    {
        $clean = ltrim($url, '/'); // 'parametros/clientes'
        if ($clean === '') {
            // si alguna vez querés mapear '/' a una vista específica, poné tu preferencia acá:
            // return 'views/control/dashboard/index.php';
            return 'views/control/dashboard/index.php';
        }
        return 'views/' . $clean . '/index.php';
    }
}

if (!function_exists('route_map_from_menu')) {
    /**
     * Construye el mapa de rutas desde el menú.
     *
     * @param array<int,array<string,mixed>>|null $menu Si es null, carga desde config/menu.php
                // return 'views/control/dashboard/index.php';
                return 'views/control/dashboard/index.php';
     */
    function route_map_from_menu(?array $menu = null, bool $validateFiles = true): array
    {
        $menu = $menu ?? menu_load();
        $flat = menu_flat($menu);

        $routes = [];
        $base   = sol_base_path();

        foreach ($flat as $it) {
            $url   = (string)($it['url'] ?? '');
            if ($url === '') {
                continue;
            }

            // gate por defecto TRUE, pero si el ítem define 'gate' lo respetamos.
            $gate = $it['gate'] ?? true;
            $gate = (bool)$gate;

            $viewRel = infer_view_from_url($url);
            $viewAbs = $base . '/' . $viewRel;

            $entry = [
                'view'  => $viewRel,
                'gate'  => $gate,
                'id'    => (string)($it['id'] ?? ''),
                'label' => (string)($it['label'] ?? ''),
                'icon'  => (string)($it['icon'] ?? ''),
            ];

            if ($validateFiles) {
                $entry['exists'] = is_file($viewAbs);
            }

            $routes[$url] = $entry;
        }

        return $routes;
    }
}

if (!function_exists('route_lookup')) {
    /**
     * Busca una ruta en el mapa generado (helper).
     *
     * @param string $path Path pedido (ej. '/parametros/productos')
     * @param array<string,array<string,mixed>>|null $routes
     * @return array<string,mixed>|null
     */
    function route_lookup(string $path, ?array $routes = null): ?array
    {
        $routes = $routes ?? route_map_from_menu();
        return $routes[$path] ?? null;
    }
}

if (!function_exists('route_require_view')) {
    /**
     * Requiere una vista resuelta contra BASE_PATH, con chequeo de existencia.
     *
    * @param string $viewRel Ruta relativa (ej. 'views/control/dashboard/index.php')
     * @throws RuntimeException si el archivo no existe
     */
    function route_require_view(string $viewRel): void
    {
        $full = sol_base_path() . '/' . ltrim($viewRel, '/');
        if (!is_file($full)) {
            throw new RuntimeException("Vista no encontrada: {$full}");
        }
        require $full;
    }
}
