<?php

declare(strict_types=1);

require_once dirname(__DIR__) . '/InventarioReport.php';

if (!function_exists('almacenamiento_indicator_estado')) {
    /**
     * Determina el estado semáforo en base al valor contra la meta configurada.
     */
    function almacenamiento_indicator_estado(?float $valor, ?float $meta): string
    {
        if ($valor === null || $meta === null) {
            return 'SIN_DATOS';
        }

        $meta = (float) $meta;
        if ($meta <= 0.0) {
            return 'SIN_DATOS';
        }

        $valor = (float) $valor;

        if ($valor >= $meta) {
            return 'OK';
        }

        if ($valor <= ($meta * 0.4)) {
            return 'CRITICO';
        }

        return 'ALERTA';
    }
}

if (!function_exists('almacenamiento_fetch_filters')) {
    function almacenamiento_fetch_filters(PDO $pdo): array
    {
        return [
            'depositos' => almacenamiento_fetch_depositos($pdo),
            'clientes'  => almacenamiento_fetch_clientes($pdo),
        ];
    }
}

if (!function_exists('almacenamiento_fetch_data')) {
    function almacenamiento_fetch_data(PDO $pdo, array $rawFilters): array
    {
        $filters = almacenamiento_normalize_filters($rawFilters);

        $counts = almacenamiento_fetch_latest_counts($pdo, $filters);
        $stockMap = almacenamiento_fetch_stock_map($pdo, $filters);

        $productIds = array_values(array_unique(array_merge(array_keys($counts['by_product']), array_keys($stockMap['by_product']))));
        sort($productIds);

        if (!$productIds) {
            return [
                'filters'    => $filters,
                'rows'       => [],
                'summary'    => almacenamiento_compute_summary([], $filters['tolerancia_pct'], null),
                'indicators' => [],
                'meta'       => $counts['meta'],
            ];
        }

        $productMeta = almacenamiento_fetch_product_meta($pdo, $productIds);
        $depositos = almacenamiento_fetch_depositos_map($pdo);

        $rows = almacenamiento_build_rows(
            $productIds,
            $productMeta,
            $counts['by_key'],
            $stockMap['by_key'],
            $filters,
            $depositos
        );

        $occupancy = almacenamiento_compute_occupancy($pdo, $filters);
        $summary = almacenamiento_compute_summary($rows, $filters['tolerancia_pct'], $occupancy);
        $indicators = almacenamiento_build_indicators($summary, $filters);

        return [
            'filters'    => $filters,
            'rows'       => $rows,
            'summary'    => $summary,
            'indicators' => $indicators,
            'meta'       => [
                'counts' => $counts['meta'],
                'stock'  => $stockMap['meta'],
            ],
        ];
    }
}

if (!function_exists('almacenamiento_normalize_filters')) {
    function almacenamiento_normalize_filters(array $raw): array
    {
        $fechaDesde = inventario_sanitize_date($raw['fecha_desde'] ?? '');
        $fechaHasta = inventario_sanitize_date($raw['fecha_hasta'] ?? '');
        if ($fechaDesde !== '' && $fechaHasta !== '' && $fechaDesde > $fechaHasta) {
            [$fechaDesde, $fechaHasta] = [$fechaHasta, $fechaDesde];
        }

        $tolerancia = isset($raw['tolerancia_pct']) ? (float) $raw['tolerancia_pct'] : 2.0;
        if ($tolerancia <= 0) {
            $tolerancia = 2.0;
        }

        $periodoRotacion = (int) ($raw['rotacion_dias'] ?? 30);
        if ($periodoRotacion <= 0) {
            $periodoRotacion = 30;
        }

        return [
            'deposito_id'     => trim((string) ($raw['deposito_id'] ?? '')),
            'cliente_id'      => trim((string) ($raw['cliente_id'] ?? '')),
            'fecha_desde'     => $fechaDesde,
            'fecha_hasta'     => $fechaHasta,
            'tolerancia_pct'  => $tolerancia,
            'rotacion_dias'   => $periodoRotacion,
        ];
    }
}

if (!function_exists('almacenamiento_fetch_depositos')) {
    function almacenamiento_fetch_depositos(PDO $pdo): array
    {
        $table = almacenamiento_detect_deposito_table($pdo);
        if ($table === null) {
            return [];
        }

        $cols = almacenamiento_detect_deposito_columns($pdo, $table);

        $where = ['1=1'];
        if ($cols['deleted_at']) {
            $where[] = '(d.`' . $cols['deleted_at'] . '` IS NULL)';
        }
        if ($cols['activo']) {
            $where[] = '(d.`' . $cols['activo'] . '` IS NULL OR d.`' . $cols['activo'] . '` = 1)';
        }

        $sql = 'SELECT d.`' . $cols['id'] . '` AS id';
        $sql .= ', d.`' . $cols['code'] . '` AS code';
        $sql .= $cols['name'] ? ', d.`' . $cols['name'] . '` AS nombre' : ', NULL AS nombre';
        $sql .= ' FROM `' . $table . '` d WHERE ' . implode(' AND ', $where) . ' ORDER BY d.`' . $cols['code'] . '` ASC';

        return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}

if (!function_exists('almacenamiento_fetch_clientes')) {
    function almacenamiento_fetch_clientes(PDO $pdo): array
    {
        if (!inventario_has_table($pdo, 'para_clientes')) {
            return [];
        }

        $sql = 'SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social';
        return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}

if (!function_exists('almacenamiento_fetch_depositos_map')) {
    function almacenamiento_fetch_depositos_map(PDO $pdo): array
    {
        $map = [];
        foreach (almacenamiento_fetch_depositos($pdo) as $deposito) {
            $id = isset($deposito['id']) ? (string) $deposito['id'] : '';
            if ($id === '') {
                continue;
            }
            $label = trim((string) ($deposito['code'] ?? ''));
            $nombre = trim((string) ($deposito['nombre'] ?? ''));
            if ($nombre !== '' && stripos($label, $nombre) === false) {
                $label = $label !== '' ? $label . ' · ' . $nombre : $nombre;
            }
            if ($label === '') {
                $label = 'Depósito #' . $id;
            }
            $map[(int) $id] = $label;
        }
        return $map;
    }
}

if (!function_exists('almacenamiento_fetch_product_meta')) {
    function almacenamiento_fetch_product_meta(PDO $pdo, array $productIds): array
    {
        $productIds = array_values(array_unique(array_map('intval', $productIds)));
        if (!$productIds) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($productIds), '?'));
        $sql = [
            'SELECT',
            '  pr.id AS producto_id,',
            '  pr.sku,',
            '  pr.denominacion,',
            '  pr.cliente_id,',
            '  COALESCE(cli.razon_social, "") AS cliente,',
            '  COALESCE(op.nombre, "") AS operativa',
            'FROM para_productos pr',
            'LEFT JOIN para_clientes cli ON cli.id = pr.cliente_id',
            'LEFT JOIN sys_operativas op ON op.id = pr.operativa_id',
            'WHERE pr.id IN (' . $placeholders . ')'
        ];

        if (inventario_has_column($pdo, 'para_productos', 'deleted_at')) {
            $sql[] = 'AND pr.deleted_at IS NULL';
        }

        $stmt = $pdo->prepare(implode(' ', $sql));
        foreach ($productIds as $index => $pid) {
            $stmt->bindValue($index + 1, $pid, PDO::PARAM_INT);
        }
        $stmt->execute();

        $meta = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
            $pid = (int) ($row['producto_id'] ?? 0);
            if ($pid <= 0) {
                continue;
            }
            $meta[$pid] = [
                'sku'        => (string) ($row['sku'] ?? ''),
                'producto'   => (string) ($row['denominacion'] ?? ''),
                'cliente_id' => (string) ($row['cliente_id'] ?? ''),
                'cliente'    => (string) ($row['cliente'] ?? ''),
                'operativa'  => (string) ($row['operativa'] ?? ''),
            ];
        }

        return $meta;
    }
}

if (!function_exists('almacenamiento_fetch_latest_counts')) {
    function almacenamiento_fetch_latest_counts(PDO $pdo, array $filters): array
    {
        if (!inventario_has_table($pdo, 'inv_conteos') || !inventario_has_table($pdo, 'inv_conteo_items')) {
            return [
                'by_product' => [],
                'by_key'     => [],
                'meta'       => ['source' => 'absent'],
            ];
        }

        $depositoFilter = $filters['deposito_id'];
        $clienteFilter = $filters['cliente_id'];
        $fechaDesde = $filters['fecha_desde'];
        $fechaHasta = $filters['fecha_hasta'];

        $estadoList = ['APROBADO', 'FINALIZADO'];

        $select = [];
        $select[] = 'SELECT * FROM (';
        $select[] = '    SELECT';
        $select[] = '        ci.producto_id AS producto_id,';
        $select[] = '        c.deposito_id AS deposito_id,';
        $select[] = '        c.codigo AS conteo_codigo,';
        $select[] = '        c.fecha_conteo AS fecha_conteo,';
        $select[] = '        c.estado AS conteo_estado,';
        $select[] = '        c.tipo AS conteo_tipo,';
        $select[] = '        ci.total_unidades AS total_unidades,';
        $select[] = '        ci.cantidad_pallets AS pallets,';
        $select[] = '        ci.cantidad_cajas AS cajas,';
        $select[] = '        ci.cantidad_unidades AS unidades_sueltas,';
        $select[] = '        COALESCE(c.aprobado_at, c.updated_at, c.created_at) AS marca_tiempo,';
        $select[] = '        ROW_NUMBER() OVER (';
        $select[] = '            PARTITION BY ci.producto_id, COALESCE(c.deposito_id, 0)';
        $select[] = '            ORDER BY c.fecha_conteo DESC, COALESCE(c.aprobado_at, c.updated_at, c.created_at) DESC, ci.id DESC';
        $select[] = '        ) AS rn';
        $select[] = '    FROM inv_conteo_items ci';
        $select[] = '    JOIN inv_conteos c ON c.id = ci.conteo_id';
        $select[] = '    JOIN para_productos pr ON pr.id = ci.producto_id';
        $select[] = '    WHERE c.deleted_at IS NULL';
        $select[] = '      AND ci.deleted_at IS NULL';
        $select[] = '      AND c.estado IN (' . implode(',', array_fill(0, count($estadoList), '?')) . ')';
        if ($depositoFilter !== '') {
            $select[] = '      AND c.deposito_id = ?';
        }
        if ($clienteFilter !== '') {
            $select[] = '      AND pr.cliente_id = ?';
        }
        if ($fechaDesde !== '') {
            $select[] = '      AND c.fecha_conteo >= ?';
        }
        if ($fechaHasta !== '') {
            $select[] = '      AND c.fecha_conteo <= ?';
        }
        if (inventario_has_column($pdo, 'para_productos', 'deleted_at')) {
            $select[] = '      AND pr.deleted_at IS NULL';
        }
        $select[] = ') ranked WHERE rn = 1';

        $params = [];
        foreach ($estadoList as $estado) {
            $params[] = $estado;
        }
        if ($depositoFilter !== '') {
            $params[] = $depositoFilter;
        }
        if ($clienteFilter !== '') {
            $params[] = $clienteFilter;
        }
        if ($fechaDesde !== '') {
            $params[] = $fechaDesde;
        }
        if ($fechaHasta !== '') {
            $params[] = $fechaHasta;
        }

        $sql = implode(' ', $select);
        $stmt = $pdo->prepare($sql);
        foreach ($params as $index => $value) {
            $paramType = is_int($value) ? PDO::PARAM_INT : PDO::PARAM_STR;
            $stmt->bindValue($index + 1, $value, $paramType);
        }
        $stmt->execute();

        $byProduct = [];
        $byKey = [];
        $maxFecha = null;
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
            $productoId = (int) ($row['producto_id'] ?? 0);
            if ($productoId <= 0) {
                continue;
            }
            $depositoId = isset($row['deposito_id']) && $row['deposito_id'] !== null ? (int) $row['deposito_id'] : 0;
            $key = $depositoId . ':' . $productoId;
            $fecha = (string) ($row['fecha_conteo'] ?? '');
            if ($fecha !== '') {
                if ($maxFecha === null || $fecha > $maxFecha) {
                    $maxFecha = $fecha;
                }
            }

            $payload = [
                'producto_id'    => $productoId,
                'deposito_id'    => $depositoId,
                'fecha_conteo'   => $fecha,
                'conteo_codigo'  => (string) ($row['conteo_codigo'] ?? ''),
                'conteo_estado'  => (string) ($row['conteo_estado'] ?? ''),
                'conteo_tipo'    => (string) ($row['conteo_tipo'] ?? ''),
                'total_unidades' => (float) ($row['total_unidades'] ?? 0),
                'pallets'        => (float) ($row['pallets'] ?? 0),
                'cajas'          => (float) ($row['cajas'] ?? 0),
                'unidades'       => (float) ($row['unidades_sueltas'] ?? 0),
                'marca_tiempo'   => (string) ($row['marca_tiempo'] ?? ''),
            ];

            $byProduct[$productoId] = $payload;
            $byKey[$key] = $payload;
        }

        return [
            'by_product' => $byProduct,
            'by_key'     => $byKey,
            'meta'       => [
                'ultima_fecha' => $maxFecha,
                'source'       => 'inv_conteos',
                'rows'         => count($byProduct),
            ],
        ];
    }
}

if (!function_exists('almacenamiento_fetch_stock_map')) {
    function almacenamiento_fetch_stock_map(PDO $pdo, array $filters): array
    {
        $depositoFilter = $filters['deposito_id'];
        $clienteFilter = $filters['cliente_id'];

        $result = [
            'by_product' => [],
            'by_key'     => [],
            'meta'       => ['source' => ''],
        ];

        if (inventario_has_table($pdo, 'wh_stock')) {
            $result['meta']['source'] = 'wh_stock';
            $depositColumn = inventario_detect_column($pdo, 'wh_stock', ['deposito_id', 'deposit_id']);
            $posColumn = inventario_detect_column($pdo, 'wh_stock', ['posicion_id', 'position_id']);
            $qtyUv = inventario_has_column($pdo, 'wh_stock', 'qty_uv');
            $qtyUc = inventario_has_column($pdo, 'wh_stock', 'qty_uc');

            $stockExprParts = [];
            if ($qtyUc) {
                $stockExprParts[] = 'COALESCE(s.qty_uc, 0)';
            }
            if ($qtyUv) {
                $stockExprParts[] = 'COALESCE(s.qty_uv, 0) * COALESCE(ppu.unidades_por_uv, 0)';
            }
            if (!$stockExprParts) {
                $stockExprParts[] = '0';
            }
            $stockExpr = implode(' + ', $stockExprParts);

            $select = [
                'SELECT',
                '  s.producto_id AS producto_id,',
                '  COALESCE(' . ($depositColumn ? 's.`' . $depositColumn . '`' : '0') . ', 0) AS deposito_id,',
                '  SUM(' . $stockExpr . ') AS unidades',
                'FROM wh_stock s',
                'JOIN para_productos pr ON pr.id = s.producto_id',
            ];

            if (inventario_has_table($pdo, 'para_producto_pack') && inventario_has_column($pdo, 'para_producto_pack', 'unidades_por_uv')) {
                $select[] = 'LEFT JOIN (SELECT producto_id, MAX(unidades_por_uv) AS unidades_por_uv FROM para_producto_pack GROUP BY producto_id) ppu ON ppu.producto_id = pr.id';
            } else {
                $select[] = 'LEFT JOIN (SELECT 0 AS producto_id, 0 AS unidades_por_uv) ppu ON ppu.producto_id = pr.id';
            }

            $where = ['1=1'];
            if ($depositoFilter !== '' && $depositColumn) {
                $where[] = 's.`' . $depositColumn . '` = :deposito_id';
            }
            if ($clienteFilter !== '') {
                $where[] = 'pr.cliente_id = :cliente_id';
            }
            if (inventario_has_column($pdo, 'wh_stock', 'deleted_at')) {
                $where[] = 's.deleted_at IS NULL';
            }
            if (inventario_has_column($pdo, 'para_productos', 'deleted_at')) {
                $where[] = 'pr.deleted_at IS NULL';
            }

            $select[] = 'WHERE ' . implode(' AND ', $where);
            $select[] = 'GROUP BY s.producto_id, deposito_id';

            $stmt = $pdo->prepare(implode(' ', $select));
            if ($depositoFilter !== '' && $depositColumn) {
                $stmt->bindValue(':deposito_id', $depositoFilter, PDO::PARAM_INT);
            }
            if ($clienteFilter !== '') {
                $stmt->bindValue(':cliente_id', $clienteFilter, PDO::PARAM_INT);
            }
            $stmt->execute();

            foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
                $productoId = (int) ($row['producto_id'] ?? 0);
                if ($productoId <= 0) {
                    continue;
                }
                $depositoId = (int) ($row['deposito_id'] ?? 0);
                $key = $depositoId . ':' . $productoId;
                $unidades = (float) ($row['unidades'] ?? 0);

                $result['by_product'][$productoId] = (
                    $result['by_product'][$productoId] ?? 0
                ) + $unidades;
                $result['by_key'][$key] = (
                    $result['by_key'][$key] ?? 0
                ) + $unidades;
            }

            return $result;
        }

        if (inventario_has_table($pdo, 'wh_pallets') && inventario_has_table($pdo, 'wh_pallet_items')) {
            $result['meta']['source'] = 'wh_pallet_items';

            $hasUcTotal = inventario_has_column($pdo, 'wh_pallet_items', 'uc_total_cache');
            $hasUcPorCaja = inventario_has_column($pdo, 'wh_pallet_items', 'uc_por_caja');
            $hasUvCajas = inventario_has_column($pdo, 'wh_pallet_items', 'uv_cajas');
            $hasUcSueltas = inventario_has_column($pdo, 'wh_pallet_items', 'uc_sueltas');
            $hasUcUnidades = inventario_has_column($pdo, 'wh_pallet_items', 'uc_unidades');
            $hasEstado = inventario_has_column($pdo, 'wh_pallet_items', 'estado');

            $unitExprParts = [];
            if ($hasUcTotal) {
                $unitExprParts[] = 'COALESCE(it.uc_total_cache, 0)';
            }
            if ($hasUcPorCaja && $hasUvCajas) {
                $unitExprParts[] = 'COALESCE(it.uc_por_caja, 0) * COALESCE(it.uv_cajas, 0)';
            }
            if ($hasUcSueltas) {
                $unitExprParts[] = 'COALESCE(it.uc_sueltas, 0)';
            }
            if ($hasUcUnidades) {
                $unitExprParts[] = 'COALESCE(it.uc_unidades, 0)';
            }
            if (!$unitExprParts) {
                $unitExprParts[] = '0';
            }
            $unitExpr = 'COALESCE(' . implode(', ', $unitExprParts) . ', 0)';

            $positionTable = almacenamiento_detect_positions_table($pdo);
            $occupancyTable = inventario_has_table($pdo, 'wh_position_occupancy') ? 'wh_position_occupancy' : null;

            $joins = [];
            $joins[] = 'JOIN wh_pallets pa ON pa.id = it.pallet_id';
            $joins[] = 'JOIN para_productos pr ON pr.id = it.producto_id';

            $depositExpr = 'COALESCE(pa.deposito_id, 0)';
            if ($occupancyTable && $positionTable) {
                $depositColumn = inventario_detect_column($pdo, $positionTable, ['deposito_id', 'deposit_id']);
                if ($depositColumn) {
                    $joins[] = 'LEFT JOIN ' . $occupancyTable . ' occ ON occ.pallet_id = pa.id AND occ.hasta IS NULL';
                    $joins[] = 'LEFT JOIN ' . $positionTable . ' pos ON pos.id = occ.position_id';
                    $depositExpr = 'COALESCE(pos.`' . $depositColumn . '`, pa.deposito_id, 0)';
                }
            } elseif (inventario_has_column($pdo, 'wh_pallets', 'deposito_id')) {
                $depositExpr = 'COALESCE(pa.deposito_id, 0)';
            }

            $where = ['1=1'];
            if (inventario_has_column($pdo, 'wh_pallet_items', 'deleted_at')) {
                $where[] = 'it.deleted_at IS NULL';
            }
            if (inventario_has_column($pdo, 'wh_pallets', 'deleted_at')) {
                $where[] = 'pa.deleted_at IS NULL';
            }
            if (inventario_has_column($pdo, 'para_productos', 'deleted_at')) {
                $where[] = 'pr.deleted_at IS NULL';
            }
            if ($clienteFilter !== '') {
                $where[] = 'pr.cliente_id = :cliente_id';
            }

            $sql = [];
            $sql[] = 'SELECT producto_id, deposito_id, SUM(unidades) AS unidades';
            $sql[] = 'FROM (';
            $sql[] = '    SELECT';
            $sql[] = '        it.producto_id AS producto_id,';
            $sql[] = '        ' . $depositExpr . ' AS deposito_id,';
            $sql[] = '        ' . $unitExpr . ' AS unidades';
            $sql[] = '    FROM wh_pallet_items it';
            $sql[] = implode(' ', $joins);
            $sql[] = '    WHERE ' . implode(' AND ', $where);
            if ($depositoFilter !== '') {
                $sql[] = '      AND ' . $depositExpr . ' = :deposito_id';
            }
            $sql[] = ') agg'
                . ' GROUP BY producto_id, deposito_id';

            $stmt = $pdo->prepare(implode(' ', $sql));
            if ($clienteFilter !== '') {
                $stmt->bindValue(':cliente_id', $clienteFilter, PDO::PARAM_INT);
            }
            if ($depositoFilter !== '') {
                $stmt->bindValue(':deposito_id', $depositoFilter, PDO::PARAM_INT);
            }
            $stmt->execute();

            foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [] as $row) {
                $productoId = (int) ($row['producto_id'] ?? 0);
                if ($productoId <= 0) {
                    continue;
                }
                $depositoId = (int) ($row['deposito_id'] ?? 0);
                $key = $depositoId . ':' . $productoId;
                $unidades = (float) ($row['unidades'] ?? 0);

                $result['by_product'][$productoId] = (
                    $result['by_product'][$productoId] ?? 0
                ) + $unidades;
                $result['by_key'][$key] = (
                    $result['by_key'][$key] ?? 0
                ) + $unidades;
            }
        }

        return $result;
    }
}

if (!function_exists('almacenamiento_build_rows')) {
    function almacenamiento_build_rows(
        array $productIds,
        array $productMeta,
        array $countMapByKey,
        array $stockMapByKey,
        array $filters,
        array $depositos
    ): array {
        $rows = [];
        $tolerancia = $filters['tolerancia_pct'];
        $today = new DateTimeImmutable('now', new DateTimeZone(env('TIMEZONE', 'UTC') ?: 'UTC'));

        foreach ($productIds as $pid) {
            $meta = $productMeta[$pid] ?? [
                'sku'        => '',
                'producto'   => '',
                'cliente_id' => '',
                'cliente'    => '',
                'operativa'  => '',
            ];

            $depositIds = [];
            if ($filters['deposito_id'] !== '') {
                $depositIds[(int) $filters['deposito_id']] = true;
            } else {
                foreach ($countMapByKey as $key => $_payload) {
                    [$depRaw, $prodRaw] = array_pad(explode(':', $key, 2), 2, null);
                    if ((int) $prodRaw === $pid) {
                        $depositIds[(int) $depRaw] = true;
                    }
                }
                foreach ($stockMapByKey as $key => $_value) {
                    [$depRaw, $prodRaw] = array_pad(explode(':', $key, 2), 2, null);
                    if ((int) $prodRaw === $pid) {
                        $depositIds[(int) $depRaw] = true;
                    }
                }
                if (!$depositIds) {
                    $depositIds[0] = true;
                }
            }

            foreach (array_keys($depositIds) as $depositoId) {
                $key = $depositoId . ':' . $pid;
                $count = $countMapByKey[$key] ?? null;
                if ($count === null && $depositoId !== 0) {
                    $count = $countMapByKey['0:' . $pid] ?? null;
                }
                $stock = $stockMapByKey[$key] ?? null;
                if ($stock === null && $depositoId !== 0) {
                    $stock = $stockMapByKey['0:' . $pid] ?? null;
                }

                if (!$count && !$stock) {
                    continue;
                }

                $conteoUnidades = $count['total_unidades'] ?? null;
                $stockUnidades = $stock !== null ? (float) $stock : null;
                $delta = null;
                if ($stockUnidades !== null && $conteoUnidades !== null) {
                    $delta = $stockUnidades - $conteoUnidades;
                }

                $deltaPct = null;
                $exactitud = null;
                if ($stockUnidades !== null && $stockUnidades > 0 && $delta !== null) {
                    $deltaPct = ($delta / $stockUnidades) * 100;
                    $exactitud = max(0.0, min(100.0, 100.0 - abs($deltaPct)));
                } elseif ($delta === 0.0) {
                    $deltaPct = 0.0;
                    $exactitud = 100.0;
                }

                $fechaConteo = $count['fecha_conteo'] ?? null;
                $diasDesdeConteo = null;
                if ($fechaConteo) {
                    try {
                        $fecha = new DateTimeImmutable($fechaConteo, $today->getTimezone());
                        $diasDesdeConteo = (int) $fecha->diff($today)->format('%a');
                    } catch (Throwable $e) {
                        $diasDesdeConteo = null;
                    }
                }

                $semaforo = 'SIN_DATOS';
                if ($exactitud !== null) {
                    $absDelta = abs($deltaPct ?? 0.0);
                    if ($absDelta <= $tolerancia) {
                        $semaforo = 'OK';
                    } elseif ($absDelta <= ($tolerancia * 2)) {
                        $semaforo = 'ALERTA';
                    } else {
                        $semaforo = 'CRITICO';
                    }
                } elseif ($conteoUnidades === null) {
                    $semaforo = 'SIN_CONTEO';
                }

                $observacion = '';
                if ($semaforo === 'SIN_CONTEO') {
                    $observacion = 'Sin inventario físico registrado';
                } elseif ($semaforo === 'ALERTA') {
                    $observacion = 'Revisar diferencia, supera tolerancia configurada';
                } elseif ($semaforo === 'CRITICO') {
                    $observacion = 'Diferencia crítica entre stock y conteo';
                } elseif ($semaforo === 'OK') {
                    $observacion = 'Dentro de tolerancia';
                }

                $rows[] = [
                    'producto_id'        => $pid,
                    'sku'                => $meta['sku'],
                    'producto'           => $meta['producto'],
                    'cliente'            => $meta['cliente'],
                    'operativa'          => $meta['operativa'],
                    'deposito_id'        => $depositoId,
                    'deposito'           => $depositos[$depositoId] ?? ($depositoId === 0 ? 'General' : 'Depósito #' . $depositoId),
                    'conteo_fecha'       => $fechaConteo,
                    'conteo_codigo'      => $count['conteo_codigo'] ?? null,
                    'conteo_estado'      => $count['conteo_estado'] ?? null,
                    'conteo_tipo'        => $count['conteo_tipo'] ?? null,
                    'conteo_unidades'    => $conteoUnidades,
                    'stock_unidades'     => $stockUnidades,
                    'delta_unidades'     => $delta,
                    'delta_pct'          => $deltaPct,
                    'exactitud_pct'      => $exactitud,
                    'dias_desde_conteo'  => $diasDesdeConteo,
                    'semaforo'           => $semaforo,
                    'observaciones'      => $observacion,
                ];
            }
        }

        return $rows;
    }
}

if (!function_exists('almacenamiento_compute_summary')) {
    function almacenamiento_compute_summary(array $rows, float $tolerancia, ?array $occupancy): array
    {
        $totals = [
            'skus'                   => count($rows),
            'con_conteo'             => 0,
            'sin_conteo'             => 0,
            'delta_total'            => 0.0,
            'delta_absoluto'         => 0.0,
            'stock_total'            => 0.0,
            'conteo_total'           => 0.0,
            'exactitud_promedio'     => null,
            'exactitud_global'       => null,
            'skus_fuera_tolerancia'  => 0,
            'ultima_fecha_conteo'    => null,
            'dias_promedio'          => null,
            'ocupacion'              => $occupancy ?? null,
            'rotacion_estimada'      => null,
            'productividad_estimada' => null,
        ];

        if (!$rows) {
            return $totals;
        }

        $exactitudSum = 0.0;
        $exactitudCount = 0;
        $diasSum = 0;
        $diasCount = 0;

        foreach ($rows as $row) {
            $stock = $row['stock_unidades'];
            $conteo = $row['conteo_unidades'];
            $delta = $row['delta_unidades'];

            if ($conteo !== null) {
                $totals['con_conteo']++;
                $totals['conteo_total'] += (float) $conteo;
            } else {
                $totals['sin_conteo']++;
            }
            if ($stock !== null) {
                $totals['stock_total'] += (float) $stock;
            }
            if ($delta !== null) {
                $totals['delta_total'] += (float) $delta;
                $totals['delta_absoluto'] += abs((float) $delta);
            }

            if ($row['exactitud_pct'] !== null) {
                $exactitudSum += (float) $row['exactitud_pct'];
                $exactitudCount++;
            }
            if ($row['dias_desde_conteo'] !== null) {
                $diasSum += (int) $row['dias_desde_conteo'];
                $diasCount++;
            }
            if ($row['semaforo'] === 'ALERTA' || $row['semaforo'] === 'CRITICO') {
                $totals['skus_fuera_tolerancia']++;
            }
            $fecha = $row['conteo_fecha'];
            if ($fecha) {
                if ($totals['ultima_fecha_conteo'] === null || $fecha > $totals['ultima_fecha_conteo']) {
                    $totals['ultima_fecha_conteo'] = $fecha;
                }
            }
        }

        if ($exactitudCount > 0) {
            $totals['exactitud_promedio'] = $exactitudSum / $exactitudCount;
        }

        if ($totals['stock_total'] > 0) {
            $totals['exactitud_global'] = max(0.0, min(100.0, 100.0 - (($totals['delta_absoluto'] / $totals['stock_total']) * 100)));
        } elseif ($totals['delta_absoluto'] === 0.0) {
            $totals['exactitud_global'] = 100.0;
        }

        if ($diasCount > 0) {
            $totals['dias_promedio'] = $diasSum / $diasCount;
        }

        $totals['tolerancia'] = $tolerancia;

        return $totals;
    }
}

if (!function_exists('almacenamiento_build_indicators')) {
    function almacenamiento_build_indicators(array $summary, array $filters): array
    {
        $indicators = [];

        $exactitud = $summary['exactitud_global'] ?? null;
        $tolerancia = isset($summary['tolerancia']) ? (float) $summary['tolerancia'] : null;
        $metaExactitud = $tolerancia !== null ? max(0.0, 100.0 - $tolerancia) : null;
        $fueraTolerancia = (int) ($summary['skus_fuera_tolerancia'] ?? 0);
        $indicators[] = [
            'clave'       => 'exactitud',
            'label'       => 'Exactitud de inventario',
            'valor'       => $exactitud,
            'meta'        => $metaExactitud,
            'estado'      => almacenamiento_indicator_estado(
                $exactitud !== null ? (float) $exactitud : null,
                $metaExactitud
            ),
            'descripcion' => $fueraTolerancia . ' SKU(s) fuera de tolerancia',
        ];

        $rotacion = almacenamiento_estimar_rotacion($summary, $filters);
        $indicators[] = [
            'clave'       => 'rotacion',
            'label'       => 'Rotación estimada',
            'valor'       => $rotacion['valor'],
            'meta'        => $rotacion['meta'],
            'estado'      => almacenamiento_indicator_estado($rotacion['valor'], $rotacion['meta']),
            'descripcion' => $rotacion['descripcion'],
        ];

        $ocupacion = $summary['ocupacion'];
        $ocupValor = $ocupacion['porcentaje'] ?? null;
        $metaOcupacion = 85.0;
        $indicators[] = [
            'clave'       => 'ocupacion',
            'label'       => 'Ocupación de posiciones',
            'valor'       => $ocupValor,
            'meta'        => $metaOcupacion,
            'estado'      => almacenamiento_indicator_estado(
                $ocupValor !== null ? (float) $ocupValor : null,
                $metaOcupacion
            ),
            'descripcion' => $ocupacion['resumen'] ?? 'Sin datos de ocupación',
        ];

        $productividad = almacenamiento_estimar_productividad($summary, $filters);
        $indicators[] = [
            'clave'       => 'productividad',
            'label'       => 'Productividad de conteos',
            'valor'       => $productividad['valor'],
            'meta'        => $productividad['meta'],
            'estado'      => almacenamiento_indicator_estado($productividad['valor'], $productividad['meta']),
            'descripcion' => $productividad['descripcion'],
        ];

        return $indicators;
    }
}

if (!function_exists('almacenamiento_estimar_rotacion')) {
    function almacenamiento_estimar_rotacion(array $summary, array $filters): array
    {
        $rotacionDias = max(1, (int) $filters['rotacion_dias']);
        $stockPromedio = max(1.0, ($summary['stock_total'] + max($summary['conteo_total'], 0.0)) / 2.0);
        $salidasEstimadas = max(0.0, ($summary['delta_absoluto'] ?? 0.0));
        $valor = ($salidasEstimadas / $stockPromedio) * (365.0 / $rotacionDias);
        $valor = round($valor, 2);

        $meta = 6.0;

        return [
            'valor'       => $valor,
            'meta'        => $meta,
            'descripcion' => 'Estimación anualizada en base a diferencias detectadas',
        ];
    }
}

if (!function_exists('almacenamiento_estimar_productividad')) {
    function almacenamiento_estimar_productividad(array $summary, array $filters): array
    {
        $diasPromedio = max(1.0, (float) ($summary['dias_promedio'] ?? $filters['rotacion_dias']));
        $valor = ($summary['conteo_total'] ?? 0.0) / $diasPromedio;
        $valor = round($valor, 1);

        $meta = 500.0;

        return [
            'valor'       => $valor,
            'meta'        => $meta,
            'descripcion' => 'Unidades físicas contadas por día estimado',
        ];
    }
}

if (!function_exists('almacenamiento_compute_occupancy')) {
    function almacenamiento_compute_occupancy(PDO $pdo, array $filters): ?array
    {
        $positionTable = almacenamiento_detect_positions_table($pdo);
        if ($positionTable === null) {
            return null;
        }

        $depositoFilter = $filters['deposito_id'];
        $depositColumn = inventario_detect_column($pdo, $positionTable, ['deposito_id', 'deposit_id']);
        $capacidadColumn = inventario_detect_column($pdo, $positionTable, ['capacidad_pallets', 'capacidad']);
        $deletedColumn = inventario_detect_column($pdo, $positionTable, ['deleted_at']);
        $activoColumn = inventario_detect_column($pdo, $positionTable, ['activo']);

        $wherePos = ['1=1'];
        if ($deletedColumn) {
            $wherePos[] = 'p.`' . $deletedColumn . '` IS NULL';
        }
        if ($activoColumn) {
            $wherePos[] = '(p.`' . $activoColumn . '` IS NULL OR p.`' . $activoColumn . '` = 1)';
        }
        if ($depositoFilter !== '' && $depositColumn) {
            $wherePos[] = 'p.`' . $depositColumn . '` = :deposito_id';
        }

        $capacidadCol = null;
        if ($capacidadColumn && inventario_has_column($pdo, $positionTable, $capacidadColumn)) {
            $capacidadCol = $capacidadColumn;
        } elseif (inventario_has_column($pdo, $positionTable, 'capacidad_pallets')) {
            $capacidadCol = 'capacidad_pallets';
        }
        $capExpr = $capacidadCol ? 'COALESCE(p.`' . $capacidadCol . '`,1)' : '1';

        $sqlPos = 'SELECT COUNT(*) AS posiciones, SUM(' . $capExpr . ') AS capacidad FROM `' . $positionTable . '` p WHERE ' . implode(' AND ', $wherePos);
        $stmt = $pdo->prepare($sqlPos);
        if ($depositoFilter !== '' && $depositColumn) {
            $stmt->bindValue(':deposito_id', $depositoFilter, PDO::PARAM_INT);
        }
        $stmt->execute();
        $pos = $stmt->fetch(PDO::FETCH_ASSOC) ?: ['posiciones' => 0, 'capacidad' => 0];

        if ((int) $pos['posiciones'] === 0) {
            return null;
        }

        $occupied = 0;
        if (inventario_has_table($pdo, 'wh_stock')) {
            $depositColumnStock = inventario_detect_column($pdo, 'wh_stock', ['deposito_id', 'deposit_id']);
            $posColumnStock = inventario_detect_column($pdo, 'wh_stock', ['posicion_id', 'position_id']);
            $qtyConditions = [];
            if (inventario_has_column($pdo, 'wh_stock', 'qty_uc')) {
                $qtyConditions[] = 'COALESCE(s.qty_uc,0) > 0';
            }
            if (inventario_has_column($pdo, 'wh_stock', 'qty_uv')) {
                $qtyConditions[] = 'COALESCE(s.qty_uv,0) > 0';
            }
            if (!$qtyConditions) {
                $qtyConditions[] = '1=1';
            }
            $whereStock = ['(' . implode(' OR ', $qtyConditions) . ')'];
            if ($depositoFilter !== '' && $depositColumnStock) {
                $whereStock[] = 's.`' . $depositColumnStock . '` = :dep';
            }
            if ($posColumnStock) {
                $sqlOcc = 'SELECT COUNT(DISTINCT s.`' . $posColumnStock . '`) FROM wh_stock s WHERE ' . implode(' AND ', $whereStock);
            } else {
                $sqlOcc = 'SELECT COUNT(*) FROM wh_stock s WHERE ' . implode(' AND ', $whereStock);
            }
            $stmtOcc = $pdo->prepare($sqlOcc);
            if ($depositoFilter !== '' && $depositColumnStock) {
                $stmtOcc->bindValue(':dep', $depositoFilter, PDO::PARAM_INT);
            }
            $stmtOcc->execute();
            $occupied = (int) $stmtOcc->fetchColumn();
        }

        $posiciones = (int) ($pos['posiciones'] ?? 0);
        if ($posiciones <= 0) {
            return null;
        }

        $porcentaje = $posiciones > 0 ? ($occupied / $posiciones) * 100.0 : 0.0;

        return [
            'posiciones_totales' => $posiciones,
            'posiciones_ocupadas' => $occupied,
            'porcentaje' => round($porcentaje, 1),
            'resumen' => $occupied . ' / ' . $posiciones . ' posiciones ocupadas',
        ];
    }
}

if (!function_exists('almacenamiento_detect_deposito_table')) {
    function almacenamiento_detect_deposito_table(PDO $pdo): ?string
    {
        foreach (['wh_deposito', 'wh_depositos'] as $candidate) {
            if (inventario_has_table($pdo, $candidate)) {
                return $candidate;
            }
        }
        return null;
    }
}

if (!function_exists('almacenamiento_detect_deposito_columns')) {
    function almacenamiento_detect_deposito_columns(PDO $pdo, string $table): array
    {
        return [
            'id'         => inventario_detect_column($pdo, $table, ['id']) ?? 'id',
            'code'       => inventario_detect_column($pdo, $table, ['code', 'codigo']) ?? 'code',
            'name'       => inventario_detect_column($pdo, $table, ['nombre', 'name', 'descripcion']) ?? null,
            'deleted_at' => inventario_detect_column($pdo, $table, ['deleted_at']) ?? null,
            'activo'     => inventario_detect_column($pdo, $table, ['activo']) ?? null,
        ];
    }
}

if (!function_exists('almacenamiento_detect_positions_table')) {
    function almacenamiento_detect_positions_table(PDO $pdo): ?string
    {
        foreach (['wh_posicion', 'wh_posiciones', 'wh_positions'] as $candidate) {
            if (inventario_has_table($pdo, $candidate)) {
                return $candidate;
            }
        }
        return null;
    }
}
