<?php

declare(strict_types=1);

if (!function_exists('clientes_otif_fetch_combos')) {
    /**
     * Obtiene catálogos para filtros del reporte OTIF.
     */
    function clientes_otif_fetch_combos(PDO $pdo): array
    {
        $clientes = $pdo->query('SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social LIMIT 400')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $estados = $pdo->query('SELECT id, code, nombre FROM so_pedido_estado ORDER BY orden ASC, nombre ASC')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $tolerancias = array_map(static function (int $value): array {
            return [
                'value' => $value,
                'label' => $value === 0 ? '0 horas' : ($value . ' horas'),
            ];
        }, clientes_otif_allowed_tolerancias());

        return [
            'clientes' => array_map(static function (array $row): array {
                $nombre = trim((string) ($row['razon_social'] ?? ''));
                return [
                    'id' => (int) ($row['id'] ?? 0),
                    'nombre' => $nombre !== '' ? $nombre : 'Sin cliente',
                ];
            }, $clientes),
            'estados' => array_map(static function (array $row): array {
                return [
                    'id' => (int) ($row['id'] ?? 0),
                    'code' => $row['code'] ?? null,
                    'nombre' => trim((string) ($row['nombre'] ?? '')) ?: 'Sin estado',
                ];
            }, $estados),
            'tolerancias' => $tolerancias,
            'default_tolerancia' => 0,
        ];
    }
}

if (!function_exists('clientes_otif_fetch_data')) {
    /**
     * Devuelve filas, resumen y agregados para el reporte OTIF.
     */
    function clientes_otif_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = clientes_otif_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        [$whereSql, $params] = clientes_otif_build_where($filters);

        $sql = '
            SELECT
                p.id,
                p.codigo,
                p.fecha_pedido,
                p.created_at,
                p.cliente_id,
                cli.razon_social AS cliente_nombre,
                est.id AS estado_id,
                est.code AS estado_code,
                est.nombre AS estado_nombre,
                COALESCE(items.expected_uv, 0) AS expected_uv,
                COALESCE(items.expected_uc, 0) AS expected_uc,
                COALESCE(items.shipped_uv, 0) AS shipped_uv,
                COALESCE(items.shipped_uc, 0) AS shipped_uc,
                COALESCE(dest.destinos_total, 0) AS destinos_total,
                COALESCE(dest.destinos_con_entrega, 0) AS destinos_con_entrega,
                dest.first_arrival,
                dest.last_departure
            FROM so_pedido p
            LEFT JOIN para_clientes cli ON cli.id = p.cliente_id
            LEFT JOIN so_pedido_estado est ON est.id = p.estado_id
            LEFT JOIN (
                SELECT
                    d.pedido_id,
                    COUNT(*) AS destinos_total,
                    SUM(CASE WHEN par.hora_llegada IS NOT NULL THEN 1 ELSE 0 END) AS destinos_con_entrega,
                    MIN(par.hora_llegada) AS first_arrival,
                    MAX(par.hora_fin_descarga) AS last_departure
                FROM so_pedido_dest d
                LEFT JOIN so_embarque_parada par ON par.pedido_dest_id = d.id
                GROUP BY d.pedido_id
            ) dest ON dest.pedido_id = p.id
            LEFT JOIN (
                SELECT
                    d.pedido_id,
                    SUM(COALESCE(i.expected_uv, 0)) AS expected_uv,
                    SUM(COALESCE(i.expected_uc, 0)) AS expected_uc,
                    SUM(COALESCE(i.shipped_uv, 0)) AS shipped_uv,
                    SUM(COALESCE(i.shipped_uc, 0)) AS shipped_uc
                FROM so_pedido_dest d
                LEFT JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id
                GROUP BY d.pedido_id
            ) items ON items.pedido_id = p.id
            ' . $whereSql . '
            ORDER BY p.fecha_pedido DESC, p.id DESC
            LIMIT :limit_plus
        ';

        $stmt = $pdo->prepare($sql);
        clientes_otif_bind_params($stmt, $params);
        $stmt->bindValue(':limit_plus', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();

        $rawRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $formatted = [];
        foreach ($rawRows as $raw) {
            $formatted[] = clientes_otif_format_row($raw, $filters);
        }

        $filtered = array_values(array_filter($formatted, static function (array $row) use ($filters): bool {
            if (!empty($filters['solo_entregados']) && empty($row['has_entrega'])) {
                return false;
            }
            if (!empty($filters['solo_otif']) && empty($row['otif'])) {
                return false;
            }
            return true;
        }));

        $truncated = false;
        if (count($filtered) > $limit) {
            $filtered = array_slice($filtered, 0, $limit);
            $truncated = true;
        }

        $summary = clientes_otif_build_summary($filtered, $filters);
        $aggregates = clientes_otif_build_aggregates($filtered);

        return [
            'filters' => $filters,
            'rows' => $filtered,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('clientes_otif_build_where')) {
    /**
     * Construye cláusula WHERE compartida por el reporte.
     */
    function clientes_otif_build_where(array $filters): array
    {
        $where = [];
        $params = [];

        $where[] = 'DATE(p.fecha_pedido) >= :fecha_desde';
        $where[] = 'DATE(p.fecha_pedido) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['cliente_id'] !== '') {
            $where[] = 'p.cliente_id = :cliente_id';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        if ($filters['estado_id'] !== '') {
            $where[] = 'p.estado_id = :estado_id';
            $params[':estado_id'] = (int) $filters['estado_id'];
        }

        $sql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        return [$sql, $params];
    }
}

if (!function_exists('clientes_otif_bind_params')) {
    function clientes_otif_bind_params(PDOStatement $stmt, array $params): void
    {
        foreach ($params as $key => $value) {
            if (is_int($value)) {
                $stmt->bindValue($key, $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue($key, $value);
            }
        }
    }
}

if (!function_exists('clientes_otif_format_row')) {
    /**
     * Formatea fila cruda y calcula banderas OTIF.
     */
    function clientes_otif_format_row(array $row, array $filters): array
    {
        $pedidoId = (int) ($row['id'] ?? 0);
        $pedidoCodigo = trim((string) ($row['codigo'] ?? ''));
        $fechaPedido = $row['fecha_pedido'] ?? null;

        $clienteId = isset($row['cliente_id']) ? (int) $row['cliente_id'] : null;
        $clienteNombre = trim((string) ($row['cliente_nombre'] ?? ''));
        if ($clienteNombre === '') {
            $clienteNombre = 'Sin cliente';
        }

        $estadoId = isset($row['estado_id']) ? (int) $row['estado_id'] : null;
        $estadoCode = $row['estado_code'] ?? null;
        $estadoNombre = trim((string) ($row['estado_nombre'] ?? ''));
        if ($estadoNombre === '') {
            $estadoNombre = 'Sin estado';
        }

        $expectedUv = max(0, (int) ($row['expected_uv'] ?? 0));
        $expectedUc = max(0, (int) ($row['expected_uc'] ?? 0));
        $shippedUv = max(0, (int) ($row['shipped_uv'] ?? 0));
        $shippedUc = max(0, (int) ($row['shipped_uc'] ?? 0));

        $destinosTotal = max(0, (int) ($row['destinos_total'] ?? 0));
        $destinosEntregados = max(0, (int) ($row['destinos_con_entrega'] ?? 0));
        if ($destinosEntregados > $destinosTotal) {
            $destinosEntregados = $destinosTotal;
        }
        $destinosPct = $destinosTotal > 0
            ? round(($destinosEntregados / $destinosTotal) * 100, 1)
            : null;

        $firstArrivalRaw = $row['first_arrival'] ?? null;
        $arrivalDt = $firstArrivalRaw ? clientes_otif_parse_datetime((string) $firstArrivalRaw, false) : null;
        $arrivalStr = $arrivalDt ? $arrivalDt->format('Y-m-d H:i:s') : null;

        $deadlineStr = clientes_otif_calculate_deadline($fechaPedido, (int) $filters['tolerancia_horas']);
        $deadlineDt = $deadlineStr ? clientes_otif_parse_datetime($deadlineStr, false) : null;

        $deltaHours = null;
        if ($arrivalDt && $deadlineDt) {
            $deltaSeconds = $arrivalDt->getTimestamp() - $deadlineDt->getTimestamp();
            $deltaHours = $deltaSeconds / 3600;
        }

        $onTime = false;
        if ($arrivalDt && $deadlineDt) {
            $onTime = $arrivalDt->getTimestamp() <= $deadlineDt->getTimestamp();
        }

        $unitsOkUc = $expectedUc <= 0 ? true : ($shippedUc >= $expectedUc);
        $unitsOkUv = $expectedUv <= 0 ? true : ($shippedUv >= $expectedUv);
        $destinosOk = $destinosTotal <= 0 ? true : ($destinosEntregados >= $destinosTotal);
        $inFull = $unitsOkUc && $unitsOkUv && $destinosOk;

        $hasEntrega = ($arrivalDt !== null) || ($destinosEntregados > 0);
        $otif = $hasEntrega && $onTime && $inFull;

        $faltanteUc = max(0, $expectedUc - $shippedUc);
        $faltanteUv = max(0, $expectedUv - $shippedUv);
        $destinosPendientes = max(0, $destinosTotal - $destinosEntregados);

        $clasificacion = 'SIN_ENTREGA';
        if ($hasEntrega) {
            if ($otif) {
                $clasificacion = 'OTIF';
            } elseif ($onTime && !$inFull) {
                $clasificacion = 'ON_TIME_INCOMPLETE';
            } elseif (!$onTime && $inFull) {
                $clasificacion = 'LATE_FULL';
            } elseif (!$onTime && !$inFull) {
                $clasificacion = 'LATE_INCOMPLETE';
            }
        }
        $clasificacionLabel = clientes_otif_classification_label($clasificacion);

        $deltaLabel = clientes_otif_format_delta_label($deltaHours);
        $deltaSign = null;
        if ($deltaHours !== null) {
            $rounded = round($deltaHours, 2);
            if ($rounded < 0) {
                $deltaSign = 'early';
            } elseif (abs($rounded) <= 0.05) {
                $deltaSign = 'ontime';
            } else {
                $deltaSign = 'late';
            }
        }

        return [
            'pedido_id' => $pedidoId,
            'pedido_codigo' => $pedidoCodigo,
            'fecha_pedido' => $fechaPedido,
            'created_at' => $row['created_at'] ?? null,
            'cliente_id' => $clienteId,
            'cliente_nombre' => $clienteNombre,
            'estado_id' => $estadoId,
            'estado_code' => $estadoCode,
            'estado_nombre' => $estadoNombre,
            'expected_uv' => $expectedUv,
            'expected_uc' => $expectedUc,
            'shipped_uv' => $shippedUv,
            'shipped_uc' => $shippedUc,
            'faltante_uv' => $faltanteUv,
            'faltante_uc' => $faltanteUc,
            'destinos_total' => $destinosTotal,
            'destinos_entregados' => $destinosEntregados,
            'destinos_pendientes' => $destinosPendientes,
            'destinos_pct' => $destinosPct,
            'destinos_cumplidos' => $destinosOk,
            'primer_arribo' => $arrivalStr,
            'primer_arribo_fmt' => $arrivalDt ? $arrivalDt->format('Y-m-d H:i') : null,
            'ultimo_evento' => $row['last_departure'] ?? null,
            'deadline_at' => $deadlineStr,
            'deadline_fmt' => $deadlineDt ? $deadlineDt->format('Y-m-d H:i') : null,
            'delta_horas' => $deltaHours !== null ? round($deltaHours, 2) : null,
            'delta_label' => $deltaLabel,
            'delta_sign' => $deltaSign,
            'on_time' => $onTime,
            'in_full' => $inFull,
            'otif' => $otif,
            'has_entrega' => $hasEntrega,
            'clasificacion' => $clasificacion,
            'clasificacion_label' => $clasificacionLabel,
            'tolerancia_horas' => (int) $filters['tolerancia_horas'],
        ];
    }
}

if (!function_exists('clientes_otif_build_summary')) {
    /**
     * Construye totales y métricas principales.
     */
    function clientes_otif_build_summary(array $rows, array $filters): array
    {
        $total = count($rows);
        $otif = 0;
        $onTime = 0;
        $inFull = 0;
        $withDelivery = 0;
        $destinosTotal = 0;
        $destinosEntregados = 0;
        $delays = [];

        foreach ($rows as $row) {
            if (!empty($row['otif'])) {
                $otif++;
            }
            if (!empty($row['on_time'])) {
                $onTime++;
            }
            if (!empty($row['in_full'])) {
                $inFull++;
            }
            if (!empty($row['has_entrega'])) {
                $withDelivery++;
            }
            $destinosTotal += (int) ($row['destinos_total'] ?? 0);
            $destinosEntregados += (int) ($row['destinos_entregados'] ?? 0);
            if ($row['delta_horas'] !== null) {
                $delays[] = (float) $row['delta_horas'];
            }
        }

        $sinEntrega = $total - $withDelivery;

        $avgDelta = null;
        $medianDelta = null;
        if ($delays) {
            $avgDelta = array_sum($delays) / count($delays);
            sort($delays);
            $mid = (int) floor(count($delays) / 2);
            if (count($delays) % 2 === 0) {
                $medianDelta = ($delays[$mid - 1] + $delays[$mid]) / 2;
            } else {
                $medianDelta = $delays[$mid];
            }
        }

        return [
            'total_pedidos' => $total,
            'otif_total' => $otif,
            'otif_pct' => clientes_otif_pct($otif, $total),
            'on_time_total' => $onTime,
            'on_time_pct' => clientes_otif_pct($onTime, $total),
            'in_full_total' => $inFull,
            'in_full_pct' => clientes_otif_pct($inFull, $total),
            'pedidos_con_entrega' => $withDelivery,
            'pedidos_sin_entrega' => $sinEntrega,
            'destinos_total' => $destinosTotal,
            'destinos_entregados' => $destinosEntregados,
            'destinos_pct' => $destinosTotal > 0 ? round(($destinosEntregados / $destinosTotal) * 100, 1) : null,
            'avg_delta_horas' => $avgDelta !== null ? round($avgDelta, 2) : null,
            'avg_delta_label' => clientes_otif_format_delta_label($avgDelta),
            'median_delta_horas' => $medianDelta !== null ? round($medianDelta, 2) : null,
            'median_delta_label' => clientes_otif_format_delta_label($medianDelta),
            'tolerancia_horas' => (int) $filters['tolerancia_horas'],
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('clientes_otif_build_aggregates')) {
    /**
     * Genera agregados por cliente, estado y clasificación.
     */
    function clientes_otif_build_aggregates(array $rows): array
    {
        if (!$rows) {
            return [
                'clientes' => [],
                'estados' => [],
                'clasificaciones' => [],
            ];
        }

        $byCliente = [];
        $byEstado = [];
        $byClasif = [];
        $totalRows = count($rows);

        foreach ($rows as $row) {
            $clienteKey = (string) ($row['cliente_id'] ?? 0);
            if (!isset($byCliente[$clienteKey])) {
                $byCliente[$clienteKey] = [
                    'key' => $clienteKey,
                    'label' => $row['cliente_nombre'] ?? 'Sin cliente',
                    'pedidos' => 0,
                    'otif' => 0,
                    'on_time' => 0,
                    'in_full' => 0,
                ];
            }
            $byCliente[$clienteKey]['pedidos']++;
            if (!empty($row['otif'])) {
                $byCliente[$clienteKey]['otif']++;
            }
            if (!empty($row['on_time'])) {
                $byCliente[$clienteKey]['on_time']++;
            }
            if (!empty($row['in_full'])) {
                $byCliente[$clienteKey]['in_full']++;
            }

            $estadoKey = (string) ($row['estado_id'] ?? 0);
            if (!isset($byEstado[$estadoKey])) {
                $byEstado[$estadoKey] = [
                    'key' => $estadoKey,
                    'label' => $row['estado_nombre'] ?? 'Sin estado',
                    'code' => $row['estado_code'] ?? null,
                    'pedidos' => 0,
                    'otif' => 0,
                    'on_time' => 0,
                    'in_full' => 0,
                ];
            }
            $byEstado[$estadoKey]['pedidos']++;
            if (!empty($row['otif'])) {
                $byEstado[$estadoKey]['otif']++;
            }
            if (!empty($row['on_time'])) {
                $byEstado[$estadoKey]['on_time']++;
            }
            if (!empty($row['in_full'])) {
                $byEstado[$estadoKey]['in_full']++;
            }

            $clasKey = $row['clasificacion'] ?? 'SIN_CLASIFICAR';
            if (!isset($byClasif[$clasKey])) {
                $byClasif[$clasKey] = [
                    'key' => $clasKey,
                    'label' => $row['clasificacion_label'] ?? $clasKey,
                    'pedidos' => 0,
                ];
            }
            $byClasif[$clasKey]['pedidos']++;
        }

        $clientes = array_values($byCliente);
        foreach ($clientes as &$item) {
            $item['otif_pct'] = clientes_otif_pct($item['otif'], $item['pedidos']);
            $item['on_time_pct'] = clientes_otif_pct($item['on_time'], $item['pedidos']);
            $item['in_full_pct'] = clientes_otif_pct($item['in_full'], $item['pedidos']);
        }
        unset($item);
        usort($clientes, static function (array $a, array $b): int {
            $cmp = $b['otif_pct'] <=> $a['otif_pct'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return $b['pedidos'] <=> $a['pedidos'];
        });
        $clientes = array_slice($clientes, 0, 10);

        $estados = array_values($byEstado);
        foreach ($estados as &$item) {
            $item['otif_pct'] = clientes_otif_pct($item['otif'], $item['pedidos']);
            $item['on_time_pct'] = clientes_otif_pct($item['on_time'], $item['pedidos']);
            $item['in_full_pct'] = clientes_otif_pct($item['in_full'], $item['pedidos']);
        }
        unset($item);
        usort($estados, static function (array $a, array $b): int {
            $cmp = $b['pedidos'] <=> $a['pedidos'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });

        $clasificaciones = array_values($byClasif);
        foreach ($clasificaciones as &$item) {
            $item['pct'] = clientes_otif_pct($item['pedidos'], $totalRows);
        }
        unset($item);
        usort($clasificaciones, static function (array $a, array $b): int {
            $cmp = $b['pedidos'] <=> $a['pedidos'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });

        return [
            'clientes' => $clientes,
            'estados' => $estados,
            'clasificaciones' => $clasificaciones,
        ];
    }
}

if (!function_exists('clientes_otif_normalize_filters')) {
    /**
     * Limpia filtros provenientes del request.
     */
    function clientes_otif_normalize_filters(array $input): array
    {
        $defaultHasta = date('Y-m-d');
        $defaultDesde = date('Y-m-d', strtotime('-29 days'));

        $desde = clientes_otif_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = clientes_otif_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);
        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'cliente_id' => clientes_otif_sanitize_numeric($input['cliente_id'] ?? null),
            'estado_id' => clientes_otif_sanitize_numeric($input['estado_id'] ?? null),
            'tolerancia_horas' => clientes_otif_sanitize_tolerancia($input['tolerancia_horas'] ?? null),
            'solo_otif' => clientes_otif_sanitize_bool($input['solo_otif'] ?? null),
            'solo_entregados' => clientes_otif_sanitize_bool($input['solo_entregados'] ?? null),
        ];
    }
}

if (!function_exists('clientes_otif_allowed_tolerancias')) {
    /**
     * Lista de tolerancias válidas (horas).
     */
    function clientes_otif_allowed_tolerancias(): array
    {
        return [0, 6, 12, 24, 48, 72];
    }
}

if (!function_exists('clientes_otif_sanitize_date')) {
    function clientes_otif_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('clientes_otif_sanitize_numeric')) {
    function clientes_otif_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('clientes_otif_sanitize_tolerancia')) {
    function clientes_otif_sanitize_tolerancia($value): int
    {
        $options = clientes_otif_allowed_tolerancias();
        $intVal = is_numeric($value) ? (int) $value : 0;
        if (!in_array($intVal, $options, true)) {
            return 0;
        }
        return $intVal;
    }
}

if (!function_exists('clientes_otif_sanitize_bool')) {
    function clientes_otif_sanitize_bool($value): bool
    {
        if (is_bool($value)) {
            return $value;
        }
        if (is_numeric($value)) {
            return (int) $value === 1;
        }
        $value = strtolower(trim((string) $value));
        return in_array($value, ['1', 'true', 'yes', 'on', 'si'], true);
    }
}

if (!function_exists('clientes_otif_calculate_deadline')) {
    function clientes_otif_calculate_deadline($fechaPedido, int $toleranciaHoras): ?string
    {
        $fecha = clientes_otif_parse_datetime((string) $fechaPedido, true);
        if (!$fecha) {
            return null;
        }
        if ($toleranciaHoras !== 0) {
            $fecha = $fecha->modify(($toleranciaHoras > 0 ? '+' : '') . $toleranciaHoras . ' hours');
        }
        return $fecha->format('Y-m-d H:i:s');
    }
}

if (!function_exists('clientes_otif_parse_datetime')) {
    function clientes_otif_parse_datetime(?string $value, bool $endOfDay): ?\DateTimeImmutable
    {
        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }

        $formats = ['Y-m-d H:i:s', 'Y-m-d\TH:i:s', 'Y-m-d H:i', 'Y-m-d'];
        foreach ($formats as $format) {
            $dt = \DateTimeImmutable::createFromFormat($format, $value);
            if ($dt instanceof \DateTimeImmutable) {
                if ($format === 'Y-m-d' && $endOfDay) {
                    $dt = $dt->setTime(23, 59, 59);
                }
                return $dt;
            }
        }

        try {
            $dt = new \DateTimeImmutable($value);
            if ($endOfDay && $dt->format('H:i:s') === '00:00:00') {
                $dt = $dt->setTime(23, 59, 59);
            }
            return $dt;
        } catch (\Throwable $e) {
            return null;
        }
    }
}

if (!function_exists('clientes_otif_pct')) {
    function clientes_otif_pct(int $part, int $total): float
    {
        if ($total <= 0) {
            return 0.0;
        }
        return round(($part / $total) * 100, 1);
    }
}

if (!function_exists('clientes_otif_format_delta_label')) {
    function clientes_otif_format_delta_label($delta): ?string
    {
        if ($delta === null) {
            return null;
        }
        $rounded = round((float) $delta, 2);
        $sign = $rounded > 0 ? '+' : '';
        return $sign . number_format($rounded, 2, '.', '') . ' h';
    }
}

if (!function_exists('clientes_otif_classification_label')) {
    function clientes_otif_classification_label(string $key): string
    {
        $map = [
            'OTIF' => 'OTIF (a tiempo y completo)',
            'ON_TIME_INCOMPLETE' => 'A tiempo pero incompleto',
            'LATE_FULL' => 'Tarde pero completo',
            'LATE_INCOMPLETE' => 'Tarde e incompleto',
            'SIN_ENTREGA' => 'Sin entrega registrada',
        ];
        return $map[$key] ?? $key;
    }
}
