<?php

declare(strict_types=1);

if (!function_exists('cobertura_sku_fetch_data')) {
    /**
     * Genera los datos del reporte de cobertura por SKU.
     *
     * @param array<string,mixed> $input
     * @return array<string,mixed>
     */
    function cobertura_sku_fetch_data(PDO $pdo, array $input): array
    {
        $filters = cobertura_sku_normalize_filters($input);

        $producto = cobertura_sku_resolve_producto($pdo, $filters);
        if (!$producto) {
            throw new RuntimeException('No se encontró el producto solicitado.');
        }

        $filters['producto_id'] = $producto['id'];
        $filters['sku'] = $producto['sku'];

        $stock = cobertura_sku_current_stock($pdo, $producto['id']);
        $consumo = cobertura_sku_calculate_consumo(
            $pdo,
            $producto['id'],
            $filters['fecha_desde'],
            $filters['fecha_hasta']
        );

        $stockDisponible = $stock['disponibles_uc'];
        $promedioDiario = $consumo['period_days'] > 0
            ? $consumo['total_salidas_uc'] / $consumo['period_days']
            : 0.0;
        $diasCobertura = $promedioDiario > 0 ? $stockDisponible / $promedioDiario : null;

        $horizons = [];
        foreach ($filters['horizontes'] as $dias) {
            $dias = (int) $dias;
            if ($dias <= 0) {
                continue;
            }
            $estimado = $promedioDiario * $dias;
            $alcance = $stockDisponible + 1e-6 >= $estimado;
            $faltante = max(0.0, $estimado - $stockDisponible);
            $sobrante = max(0.0, $stockDisponible - $estimado);
            $diasCubiertos = $promedioDiario > 0 ? min($dias, $stockDisponible / $promedioDiario) : null;

            $horizons[] = [
                'dias' => $dias,
                'consumo_estimado_uc' => $estimado,
                'stock_actual_uc' => $stockDisponible,
                'alcance' => $alcance,
                'faltante_uc' => $faltante,
                'sobrante_uc' => $sobrante,
                'dias_cubiertos' => $diasCubiertos,
            ];
        }

        $diasSin = max(0, $consumo['period_days'] - $consumo['dias_con_consumo']);

        return [
            'filters' => $filters,
            'producto' => $producto,
            'stock' => $stock,
            'consumo' => [
                'total_salidas_uc' => $consumo['total_salidas_uc'],
                'promedio_diario_uc' => $promedioDiario,
                'total_documentos' => $consumo['total_documentos'],
                'promedio_diario_documentos' => $consumo['promedio_diario_documentos'],
                'periodo_dias' => $consumo['period_days'],
                'dias_con_consumo' => $consumo['dias_con_consumo'],
                'dias_sin_consumo' => $diasSin,
                'dias_cobertura' => $diasCobertura,
                'ultima_salida' => $consumo['ultima_salida'],
                'serie_diaria' => $consumo['serie_diaria'],
            ],
            'cobertura' => [
                'horizontes' => $horizons,
            ],
        ];
    }
}

if (!function_exists('cobertura_sku_search_productos')) {
    /**
     * Devuelve hasta 20 productos que coinciden con el texto indicado (para autocompletar).
     *
     * @return array<int,array{id:int,sku:string,nombre:string}>
     */
    function cobertura_sku_search_productos(PDO $pdo, string $term): array
    {
        $term = strtoupper(trim($term));
        if ($term === '') {
            return [];
        }

        $like = cobertura_sku_like($term);
        $sql = "
            SELECT
                p.id,
                UPPER(p.sku) AS sku,
                p.denominacion
            FROM para_productos p
            WHERE p.deleted_at IS NULL
              AND (UPPER(p.sku) LIKE :like OR UPPER(p.denominacion) LIKE :like)
            ORDER BY p.sku ASC
            LIMIT 20
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':like', $like);
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(static function (array $row): array {
            return [
                'id' => (int) ($row['id'] ?? 0),
                'sku' => strtoupper((string) ($row['sku'] ?? '')),
                'nombre' => (string) ($row['denominacion'] ?? ''),
            ];
        }, $rows);
    }
}

if (!function_exists('cobertura_sku_normalize_filters')) {
    /**
     * @return array<string,mixed>
     */
    function cobertura_sku_normalize_filters(array $input): array
    {
        $productoId = isset($input['producto_id']) ? (int) $input['producto_id'] : 0;
        $sku = strtoupper(trim((string) ($input['sku'] ?? '')));

        $today = new DateTimeImmutable('today');
        $defaultDesde = $today->sub(new DateInterval('P90D'));

        $desdeInput = trim((string) ($input['fecha_desde'] ?? ''));
        $hastaInput = trim((string) ($input['fecha_hasta'] ?? ''));

        $fechaDesde = cobertura_sku_parse_date($desdeInput) ?? $defaultDesde;
        $fechaHasta = cobertura_sku_parse_date($hastaInput) ?? $today;

        if ($fechaDesde > $fechaHasta) {
            [$fechaDesde, $fechaHasta] = [$fechaHasta, $fechaDesde];
        }

        $limitStart = $fechaHasta->sub(new DateInterval('P365D'));
        if ($fechaDesde < $limitStart) {
            $fechaDesde = $limitStart;
        }

        return [
            'producto_id' => $productoId,
            'sku' => $sku,
            'fecha_desde' => $fechaDesde->format('Y-m-d'),
            'fecha_hasta' => $fechaHasta->format('Y-m-d'),
            'horizontes' => [30, 60, 90, 120],
        ];
    }
}

if (!function_exists('cobertura_sku_parse_date')) {
    function cobertura_sku_parse_date(?string $value): ?DateTimeImmutable
    {
        if (!$value) {
            return null;
        }
        try {
            return (new DateTimeImmutable($value))->setTime(0, 0);
        } catch (Throwable $e) {
            return null;
        }
    }
}

if (!function_exists('cobertura_sku_resolve_producto')) {
    /**
     * @param array<string,mixed> $filters
     * @return array{id:int,sku:string,nombre:string,cliente:string,operativa:string,stock_min:float,stock_max:?float}
     */
    function cobertura_sku_resolve_producto(PDO $pdo, array $filters): array
    {
        $productoId = (int) ($filters['producto_id'] ?? 0);
        $sku = strtoupper(trim((string) ($filters['sku'] ?? '')));

        if ($productoId <= 0 && $sku === '') {
            throw new RuntimeException('Ingrese un SKU para continuar.');
        }

        $select = "
            SELECT
                p.id,
                UPPER(p.sku) AS sku,
                p.denominacion,
                COALESCE(c.razon_social, '') AS cliente,
                COALESCE(op.nombre, '') AS operativa,
                COALESCE(p.stock_min_default, 0) AS stock_min,
                " . (cobertura_sku_has_column($pdo, 'para_productos', 'stock_max_default') ? 'COALESCE(p.stock_max_default, 0)' : 'NULL') . " AS stock_max
            FROM para_productos p
            LEFT JOIN para_clientes c ON c.id = p.cliente_id
            LEFT JOIN sys_operativas op ON op.id = p.operativa_id
            WHERE p.deleted_at IS NULL
        ";

        if ($productoId > 0) {
            $stmt = $pdo->prepare($select . ' AND p.id = :id LIMIT 1');
            $stmt->bindValue(':id', $productoId, PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
        } else {
            $stmt = $pdo->prepare($select . ' AND UPPER(p.sku) = :sku LIMIT 1');
            $stmt->bindValue(':sku', $sku);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;

            if (!$row) {
                $stmt = $pdo->prepare($select . ' AND (UPPER(p.sku) LIKE :like OR UPPER(p.denominacion) LIKE :like) ORDER BY p.sku ASC LIMIT 1');
                $stmt->bindValue(':like', cobertura_sku_like($sku));
                $stmt->execute();
                $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
            }
        }

        if (!$row) {
            throw new RuntimeException('No se encontró un producto que coincida con el SKU indicado.');
        }

        return [
            'id' => (int) ($row['id'] ?? 0),
            'sku' => strtoupper((string) ($row['sku'] ?? '')),
            'nombre' => (string) ($row['denominacion'] ?? ''),
            'cliente' => (string) ($row['cliente'] ?? ''),
            'operativa' => (string) ($row['operativa'] ?? ''),
            'stock_min' => (float) ($row['stock_min'] ?? 0),
            'stock_max' => isset($row['stock_max']) ? (float) $row['stock_max'] : null,
        ];
    }
}

if (!function_exists('cobertura_sku_current_stock')) {
    /**
     * @return array{stock_uc:float,reservados_uc:float,disponibles_uc:float,pallets:int}
     */
    function cobertura_sku_current_stock(PDO $pdo, int $productoId): array
    {
        if ($productoId <= 0) {
            return [
                'stock_uc' => 0.0,
                'reservados_uc' => 0.0,
                'disponibles_uc' => 0.0,
                'pallets' => 0,
            ];
        }

        if (cobertura_sku_has_table($pdo, 'wh_pallet_items') && cobertura_sku_has_table($pdo, 'wh_pallets')) {
            $unitExpr = cobertura_sku_item_unit_expr($pdo);
            $reservedExpr = cobertura_sku_item_reserved_expr($pdo, $unitExpr);

            $sql = "
                SELECT
                    COUNT(DISTINCT pa.id) AS pallets,
                    SUM({$unitExpr}) AS total_uc,
                    SUM({$reservedExpr}) AS reservados_uc
                FROM wh_pallet_items it
                JOIN wh_pallets pa ON pa.id = it.pallet_id
                WHERE it.producto_id = :producto_id
            ";
            $stmt = $pdo->prepare($sql);
            $stmt->bindValue(':producto_id', $productoId, PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

            $total = (float) ($row['total_uc'] ?? 0);
            $reservados = (float) ($row['reservados_uc'] ?? 0);
            $disponibles = max(0.0, $total - $reservados);

            return [
                'stock_uc' => $total,
                'reservados_uc' => $reservados,
                'disponibles_uc' => $disponibles,
                'pallets' => (int) ($row['pallets'] ?? 0),
            ];
        }

        if (cobertura_sku_has_table($pdo, 'wh_stock')) {
            $stockExpr = 'COALESCE(s.qty_uc,0)';
            if (cobertura_sku_has_column($pdo, 'wh_stock', 'qty_uv')) {
                if (
                    cobertura_sku_has_table($pdo, 'para_producto_pack') &&
                    cobertura_sku_has_column($pdo, 'para_producto_pack', 'unidades_por_uv')
                ) {
                    $stockExpr = 'COALESCE(s.qty_uc,0) + COALESCE(s.qty_uv,0) * COALESCE(pp.unidades_por_uv,0)';
                    $packJoin = 'LEFT JOIN (
                            SELECT producto_id, MAX(unidades_por_uv) AS unidades_por_uv
                            FROM para_producto_pack
                            GROUP BY producto_id
                        ) pp ON pp.producto_id = s.producto_id';
                } else {
                    $stockExpr = 'COALESCE(s.qty_uc,0) + COALESCE(s.qty_uv,0)';
                    $packJoin = '';
                }
            } else {
                $packJoin = '';
            }

            $palletSelect = cobertura_sku_has_column($pdo, 'wh_stock', 'pallet_id')
                ? 'COUNT(DISTINCT s.pallet_id) AS pallets'
                : '0 AS pallets';

            $sql = "
                SELECT {$palletSelect}, SUM({$stockExpr}) AS total_uc
                FROM wh_stock s
                {$packJoin}
                WHERE s.producto_id = :producto_id
            ";
            $stmt = $pdo->prepare($sql);
            $stmt->bindValue(':producto_id', $productoId, PDO::PARAM_INT);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

            $total = (float) ($row['total_uc'] ?? 0);
            return [
                'stock_uc' => $total,
                'reservados_uc' => 0.0,
                'disponibles_uc' => $total,
                'pallets' => (int) ($row['pallets'] ?? 0),
            ];
        }

        return [
            'stock_uc' => 0.0,
            'reservados_uc' => 0.0,
            'disponibles_uc' => 0.0,
            'pallets' => 0,
        ];
    }
}

if (!function_exists('cobertura_sku_calculate_consumo')) {
    /**
     * @return array{
     *   total_salidas_uc:float,
     *   period_days:int,
     *   dias_con_consumo:int,
     *   serie_diaria:array<int,array{fecha:string,salidas_uc:float,documentos?:int}>,
     *   ultima_salida:?string,
     *   total_documentos:int,
     *   promedio_diario_documentos:float
     * }
     */
    function cobertura_sku_calculate_consumo(PDO $pdo, int $productoId, string $fechaDesde, string $fechaHasta): array
    {
        $desde = cobertura_sku_parse_date($fechaDesde) ?? new DateTimeImmutable($fechaDesde);
        $hasta = cobertura_sku_parse_date($fechaHasta) ?? new DateTimeImmutable($fechaHasta);
        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }
        $periodDays = max(1, $hasta->diff($desde)->days + 1);

        $serie = [];
        if (cobertura_sku_has_table($pdo, 'so_pedido_dest_item')) {
            $serie = cobertura_sku_fetch_daily_salidas_orders($pdo, $productoId, $fechaDesde, $fechaHasta);
        }

        if (!$serie) {
            if (
                cobertura_sku_has_table($pdo, 'wh_moves') &&
                cobertura_sku_has_table($pdo, 'wh_move_items') &&
                cobertura_sku_has_table($pdo, 'wh_pallet_items')
            ) {
                $serie = cobertura_sku_fetch_daily_salidas_new_schema($pdo, $productoId, $fechaDesde, $fechaHasta);
            } elseif (cobertura_sku_has_table($pdo, 'wh_move')) {
                $serie = cobertura_sku_fetch_daily_salidas_legacy($pdo, $productoId, $fechaDesde, $fechaHasta);
            }
        }

            if (!function_exists('cobertura_sku_fetch_daily_salidas_orders')) {
                /**
                 * @return array<int,array{fecha:string,salidas_uc:float,documentos:int}>
                 */
                function cobertura_sku_fetch_daily_salidas_orders(PDO $pdo, int $productoId, string $fechaDesde, string $fechaHasta): array
                {
                    if ($productoId <= 0) {
                        return [];
                    }

                    $rows = [];
                    if (
                        cobertura_sku_has_table($pdo, 'so_ship_link') &&
                        cobertura_sku_has_column($pdo, 'so_ship_link', 'pedido_dest_item_id') &&
                        cobertura_sku_has_column($pdo, 'so_ship_link', 'created_at')
                    ) {
                        $rows = cobertura_sku_fetch_daily_salidas_ship_link($pdo, $productoId, $fechaDesde, $fechaHasta);
                    }

                    if (!$rows) {
                        if (
                            cobertura_sku_has_table($pdo, 'so_pedido_dest_item') &&
                            cobertura_sku_has_table($pdo, 'so_pedido_dest') &&
                            cobertura_sku_has_table($pdo, 'so_pedido')
                        ) {
                            $rows = cobertura_sku_fetch_daily_salidas_from_pedido_items($pdo, $productoId, $fechaDesde, $fechaHasta);
                        }
                    }

                    return $rows;
                }
            }

            if (!function_exists('cobertura_sku_fetch_daily_salidas_ship_link')) {
                /**
                 * @return array<int,array{fecha:string,salidas_uc:float,documentos:int}>
                 */
                function cobertura_sku_fetch_daily_salidas_ship_link(PDO $pdo, int $productoId, string $fechaDesde, string $fechaHasta): array
                {
                    [$packJoin, $packMultiplier] = cobertura_sku_pack_join_clause($pdo, 'pdi.producto_id', 'pk_sl');
                    $unitExpr = 'COALESCE(sl.uc_unidades,0) + COALESCE(sl.uv_cajas,0) * ' . $packMultiplier;

                    $sql = "
                        SELECT
                            DATE(sl.created_at) AS fecha,
                            SUM({$unitExpr}) AS salidas_uc,
                            COUNT(DISTINCT p.id) AS documentos
                        FROM so_ship_link sl
                        JOIN so_pedido_dest_item pdi ON pdi.id = sl.pedido_dest_item_id
                        JOIN so_pedido_dest pd ON pd.id = pdi.pedido_dest_id
                        JOIN so_pedido p ON p.id = pd.pedido_id
                        {$packJoin}
                        WHERE pdi.producto_id = :producto_id
                          AND sl.created_at IS NOT NULL
                          AND sl.created_at BETWEEN :desde AND :hasta
                        GROUP BY DATE(sl.created_at)
                        HAVING salidas_uc > 0
                        ORDER BY fecha ASC
                    ";

                    $stmt = $pdo->prepare($sql);
                    $stmt->bindValue(':producto_id', $productoId, PDO::PARAM_INT);
                    $stmt->bindValue(':desde', $fechaDesde . ' 00:00:00');
                    $stmt->bindValue(':hasta', $fechaHasta . ' 23:59:59');
                    $stmt->execute();

                    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
                    return array_map(static function (array $row): array {
                        return [
                            'fecha' => (string) ($row['fecha'] ?? ''),
                            'salidas_uc' => (float) ($row['salidas_uc'] ?? 0),
                            'documentos' => (int) ($row['documentos'] ?? 0),
                        ];
                    }, $rows);
                }
            }

            if (!function_exists('cobertura_sku_fetch_daily_salidas_from_pedido_items')) {
                /**
                 * @return array<int,array{fecha:string,salidas_uc:float,documentos:int}>
                 */
                function cobertura_sku_fetch_daily_salidas_from_pedido_items(PDO $pdo, int $productoId, string $fechaDesde, string $fechaHasta): array
                {
                    [$packJoin, $packMultiplier] = cobertura_sku_pack_join_clause($pdo, 'pdi.producto_id', 'pk_pi');
                    [$ucExpr, $uvExpr] = cobertura_sku_item_uc_uv_expr($pdo, 'pdi');
                    $unitExpr = $ucExpr . ' + ' . $uvExpr . ' * ' . $packMultiplier;
                    $dateExpr = 'COALESCE(p.fecha_pedido, p.updated_at, p.created_at)';

                    $sql = "
                        SELECT
                            DATE({$dateExpr}) AS fecha,
                            SUM({$unitExpr}) AS salidas_uc,
                            COUNT(DISTINCT p.id) AS documentos
                        FROM so_pedido_dest_item pdi
                        JOIN so_pedido_dest pd ON pd.id = pdi.pedido_dest_id
                        JOIN so_pedido p ON p.id = pd.pedido_id
                        {$packJoin}
                        WHERE pdi.producto_id = :producto_id
                          AND {$dateExpr} IS NOT NULL
                          AND {$dateExpr} BETWEEN :desde AND :hasta
                        GROUP BY DATE({$dateExpr})
                        HAVING salidas_uc > 0
                        ORDER BY fecha ASC
                    ";

                    $stmt = $pdo->prepare($sql);
                    $stmt->bindValue(':producto_id', $productoId, PDO::PARAM_INT);
                    $stmt->bindValue(':desde', $fechaDesde . ' 00:00:00');
                    $stmt->bindValue(':hasta', $fechaHasta . ' 23:59:59');
                    $stmt->execute();

                    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
                    return array_map(static function (array $row): array {
                        return [
                            'fecha' => (string) ($row['fecha'] ?? ''),
                            'salidas_uc' => (float) ($row['salidas_uc'] ?? 0),
                            'documentos' => (int) ($row['documentos'] ?? 0),
                        ];
                    }, $rows);
                }
            }

            if (!function_exists('cobertura_sku_pack_join_clause')) {
                /**
                 * @return array{0:string,1:string}
                 */
                function cobertura_sku_pack_join_clause(PDO $pdo, string $productoColumn, string $alias = 'pk'): array
                {
                    if (
                        cobertura_sku_has_table($pdo, 'para_producto_pack') &&
                        cobertura_sku_has_column($pdo, 'para_producto_pack', 'unidades_por_uv')
                    ) {
                        $join = "LEFT JOIN (
                            SELECT producto_id, MAX(unidades_por_uv) AS unidades_por_uv
                            FROM para_producto_pack
                            GROUP BY producto_id
                        ) {$alias} ON {$alias}.producto_id = {$productoColumn}";
                        $expr = "COALESCE({$alias}.unidades_por_uv, 1)";
                        return [$join, $expr];
                    }

                    return ['', '1'];
                }
            }

            if (!function_exists('cobertura_sku_item_uc_uv_expr')) {
                /**
                 * @return array{0:string,1:string}
                 */
                function cobertura_sku_item_uc_uv_expr(PDO $pdo, string $alias = 'pdi'): array
                {
                    $pairs = [
                        ['uc' => 'shipped_uc', 'uv' => 'shipped_uv'],
                        ['uc' => 'despachado_uc', 'uv' => 'despachado_uv'],
                        ['uc' => 'prepared_uc', 'uv' => 'prepared_uv'],
                        ['uc' => 'preparado_uc', 'uv' => 'preparado_uv'],
                        ['uc' => 'expected_uc', 'uv' => 'expected_uv'],
                    ];

                    foreach ($pairs as $pair) {
                        $ucCol = $pair['uc'];
                        if (!cobertura_sku_has_column($pdo, 'so_pedido_dest_item', $ucCol)) {
                            continue;
                        }

                        $uvCol = $pair['uv'];
                        $ucExpr = "COALESCE({$alias}.{$ucCol}, 0)";
                        $uvExpr = cobertura_sku_has_column($pdo, 'so_pedido_dest_item', $uvCol)
                            ? "COALESCE({$alias}.{$uvCol}, 0)"
                            : '0';

                        return [$ucExpr, $uvExpr];
                    }

                    return ['0', '0'];
                }
            }

        $fullSerie = $serie;
        $total = 0.0;
        $totalDocs = 0;
        foreach ($fullSerie as $row) {
            $total += max(0.0, (float) ($row['salidas_uc'] ?? 0));
            $totalDocs += (int) ($row['documentos'] ?? 0);
        }

        $diasCon = count($fullSerie);
        $ultima = $diasCon > 0 ? ($fullSerie[$diasCon - 1]['fecha'] ?? null) : null;

        if (count($serie) > 180) {
            $serie = array_slice($serie, -180);
        }

        return [
            'total_salidas_uc' => $total,
            'period_days' => $periodDays,
            'dias_con_consumo' => $diasCon,
            'serie_diaria' => $serie,
            'ultima_salida' => $ultima,
            'total_documentos' => $totalDocs,
            'promedio_diario_documentos' => $periodDays > 0 ? $totalDocs / $periodDays : 0.0,
        ];
    }
}

if (!function_exists('cobertura_sku_fetch_daily_salidas_new_schema')) {
    /**
     * @return array<int,array{fecha:string,salidas_uc:float}>
     */
    function cobertura_sku_fetch_daily_salidas_new_schema(PDO $pdo, int $productoId, string $fechaDesde, string $fechaHasta): array
    {
        $timestampExpr = cobertura_sku_move_time_expr($pdo);
        $unitExpr = cobertura_sku_move_unit_expr($pdo);
        $outboundCondition = cobertura_sku_outbound_condition($pdo);

        if ($outboundCondition !== null) {
            $salidaExpr = "CASE WHEN {$outboundCondition} THEN {$unitExpr} ELSE 0 END";
        } else {
            $deltaExpr = cobertura_sku_move_delta_fallback_expr($pdo);
            if ($deltaExpr === null) {
                return [];
            }
            $salidaExpr = $deltaExpr;
        }

        $sql = "
            SELECT
                DATE({$timestampExpr}) AS fecha,
                SUM({$salidaExpr}) AS salidas_uc
            FROM wh_moves m
            JOIN wh_move_items mi ON mi.move_id = m.id
            JOIN wh_pallet_items pi ON pi.id = mi.pallet_item_id
            WHERE pi.producto_id = :producto_id
              AND {$timestampExpr} BETWEEN :desde AND :hasta
            GROUP BY DATE({$timestampExpr})
            HAVING salidas_uc > 0
            ORDER BY fecha ASC
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':producto_id', $productoId, PDO::PARAM_INT);
        $stmt->bindValue(':desde', $fechaDesde . ' 00:00:00');
        $stmt->bindValue(':hasta', $fechaHasta . ' 23:59:59');
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(static function (array $row): array {
            return [
                'fecha' => (string) ($row['fecha'] ?? ''),
                'salidas_uc' => (float) ($row['salidas_uc'] ?? 0),
            ];
        }, $rows);
    }
}

if (!function_exists('cobertura_sku_fetch_daily_salidas_legacy')) {
    /**
     * @return array<int,array{fecha:string,salidas_uc:float}>
     */
    function cobertura_sku_fetch_daily_salidas_legacy(PDO $pdo, int $productoId, string $fechaDesde, string $fechaHasta): array
    {
        $hasCreated = cobertura_sku_has_column($pdo, 'wh_move', 'created_at');
        $dateExpr = $hasCreated ? 'DATE(m.created_at)' : 'DATE(NOW())';
        $primaryExpr = cobertura_sku_has_column($pdo, 'wh_move', 'delta_uc') ? 'COALESCE(m.delta_uc,0)' : '0';
        $fallbackExpr = cobertura_sku_has_column($pdo, 'wh_move', 'delta_uv') ? 'COALESCE(m.delta_uv,0)' : '0';

        $salidaExpr = "CASE
            WHEN {$primaryExpr} < 0 THEN ABS({$primaryExpr})
            WHEN {$primaryExpr} = 0 AND {$fallbackExpr} < 0 THEN ABS({$fallbackExpr})
            ELSE 0
        END";

        $sql = "
            SELECT {$dateExpr} AS fecha, SUM({$salidaExpr}) AS salidas_uc
            FROM wh_move m
            WHERE m.producto_id = :producto_id
        ";
        if ($hasCreated) {
            $sql .= ' AND m.created_at BETWEEN :desde AND :hasta';
        }
        $sql .= ' GROUP BY ' . $dateExpr . ' HAVING salidas_uc > 0 ORDER BY fecha ASC';

        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':producto_id', $productoId, PDO::PARAM_INT);
        if ($hasCreated) {
            $stmt->bindValue(':desde', $fechaDesde . ' 00:00:00');
            $stmt->bindValue(':hasta', $fechaHasta . ' 23:59:59');
        }
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(static function (array $row): array {
            return [
                'fecha' => (string) ($row['fecha'] ?? ''),
                'salidas_uc' => (float) ($row['salidas_uc'] ?? 0),
            ];
        }, $rows);
    }
}

if (!function_exists('cobertura_sku_move_time_expr')) {
    function cobertura_sku_move_time_expr(PDO $pdo): string
    {
        $columns = [];
        foreach (['finalizado_at', 'iniciado_at', 'asignado_at', 'created_at'] as $column) {
            if (cobertura_sku_has_column($pdo, 'wh_moves', $column)) {
                $columns[] = "m.`{$column}`";
            }
        }
        return $columns ? 'COALESCE(' . implode(', ', $columns) . ')' : 'm.created_at';
    }
}

if (!function_exists('cobertura_sku_move_unit_expr')) {
    function cobertura_sku_move_unit_expr(PDO $pdo): string
    {
        $hasMiUc = cobertura_sku_has_column($pdo, 'wh_move_items', 'uc_unidades');
        $hasMiUv = cobertura_sku_has_column($pdo, 'wh_move_items', 'uv_cajas');
        $hasPiPor = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_por_caja');
        $hasPiTotal = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_total_cache');
        $hasPiSueltas = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_sueltas');

        $options = [];
        if ($hasMiUc) {
            $options[] = 'mi.uc_unidades';
        }
        if ($hasPiPor && $hasMiUv) {
            $options[] = 'COALESCE(mi.uv_cajas,0) * COALESCE(pi.uc_por_caja,0)';
        } elseif ($hasMiUv) {
            $options[] = 'COALESCE(mi.uv_cajas,0)';
        }
        if ($hasPiTotal) {
            $options[] = 'COALESCE(pi.uc_total_cache,0)';
        } elseif ($hasPiSueltas) {
            $options[] = 'COALESCE(pi.uc_sueltas,0)';
        }
        if (!$options) {
            $options[] = '0';
        }

        return 'COALESCE(' . implode(', ', $options) . ', 0)';
    }
}

if (!function_exists('cobertura_sku_outbound_condition')) {
    function cobertura_sku_outbound_condition(PDO $pdo): ?string
    {
        $parts = [];
        if (cobertura_sku_has_column($pdo, 'wh_moves', 'move_type')) {
            $parts[] = "m.move_type IN ('OUTBOUND','OUT','ADJUSTMENT_OUT')";
        }
        $hasTipo = cobertura_sku_has_column($pdo, 'wh_moves', 'tipo');
        if ($hasTipo) {
            $parts[] = "m.tipo IN ('BAJA','SALIDA','AJUSTE_OUT','AJUSTE_BAJA','AJUSTE')";
        }
        $hasDesde = cobertura_sku_has_column($pdo, 'wh_moves', 'desde_position_id');
        $hasHasta = cobertura_sku_has_column($pdo, 'wh_moves', 'hasta_position_id');
        if ($hasDesde && $hasHasta) {
            $parts[] = '(m.desde_position_id IS NOT NULL AND m.hasta_position_id IS NULL)';
            if ($hasTipo) {
                $parts[] = "(m.tipo = 'AJUSTE' AND m.desde_position_id IS NOT NULL AND m.hasta_position_id IS NULL)";
            }
        }

        if (!$parts) {
            return null;
        }

        return '(' . implode(' OR ', $parts) . ')';
    }
}

if (!function_exists('cobertura_sku_move_delta_fallback_expr')) {
    function cobertura_sku_move_delta_fallback_expr(PDO $pdo): ?string
    {
        if (cobertura_sku_has_column($pdo, 'wh_move_items', 'delta_uc')) {
            return 'CASE WHEN mi.delta_uc < 0 THEN ABS(mi.delta_uc) ELSE 0 END';
        }
        if (cobertura_sku_has_column($pdo, 'wh_moves', 'delta_uc')) {
            return 'CASE WHEN m.delta_uc < 0 THEN ABS(m.delta_uc) ELSE 0 END';
        }
        return null;
    }
}

if (!function_exists('cobertura_sku_item_unit_expr')) {
    function cobertura_sku_item_unit_expr(PDO $pdo): string
    {
        $hasUcTotal = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_total_cache');
        $hasUcPorCaja = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_por_caja');
        $hasUvCajas = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uv_cajas');
        $hasUcSueltas = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_sueltas');
        $hasUcUnidades = cobertura_sku_has_column($pdo, 'wh_pallet_items', 'uc_unidades');

        if ($hasUcTotal) {
            $fallback = [];
            if ($hasUcPorCaja && $hasUvCajas) {
                $fallback[] = 'COALESCE(it.uc_por_caja,0) * COALESCE(it.uv_cajas,0)';
            }
            if ($hasUcSueltas) {
                $fallback[] = 'COALESCE(it.uc_sueltas,0)';
            }
            if (!$fallback && $hasUcUnidades) {
                $fallback[] = 'COALESCE(it.uc_unidades,0)';
            }
            if (!$fallback) {
                $fallback[] = '0';
            }
            return 'COALESCE(it.uc_total_cache, ' . implode(' + ', $fallback) . ')';
        }

        if ($hasUcUnidades) {
            return 'COALESCE(it.uc_unidades,0)';
        }

        $parts = [];
        if ($hasUcPorCaja && $hasUvCajas) {
            $parts[] = 'COALESCE(it.uc_por_caja,0) * COALESCE(it.uv_cajas,0)';
        } elseif ($hasUvCajas) {
            $parts[] = 'COALESCE(it.uv_cajas,0)';
        }
        if ($hasUcSueltas) {
            $parts[] = 'COALESCE(it.uc_sueltas,0)';
        }
        if (!$parts) {
            return '0';
        }
        return '(' . implode(' + ', $parts) . ')';
    }
}

if (!function_exists('cobertura_sku_item_reserved_expr')) {
    function cobertura_sku_item_reserved_expr(PDO $pdo, string $unitExpr): string
    {
        if (cobertura_sku_has_column($pdo, 'wh_pallet_items', 'estado')) {
            return "CASE WHEN it.estado = 'RESERVADO' THEN {$unitExpr} ELSE 0 END";
        }
        if (cobertura_sku_has_column($pdo, 'wh_pallets', 'reservado')) {
            return "CASE WHEN pa.reservado = 1 THEN {$unitExpr} ELSE 0 END";
        }
        return '0';
    }
}

if (!function_exists('cobertura_sku_like')) {
    function cobertura_sku_like(string $value): string
    {
        $clean = preg_replace('/[^A-Z0-9\-\s_]/u', '', strtoupper($value));
        $clean = str_replace(' ', '%', $clean ?? '');
        return '%' . $clean . '%';
    }
}

if (!function_exists('cobertura_sku_has_table')) {
    function cobertura_sku_has_table(PDO $pdo, string $table): bool
    {
        static $cache = [];
        $key = spl_object_id($pdo) . ':tbl:' . strtolower($table);
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$table]);
        return $cache[$key] = ((int) $stmt->fetchColumn() > 0);
    }
}

if (!function_exists('cobertura_sku_has_column')) {
    function cobertura_sku_has_column(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = spl_object_id($pdo) . ':col:' . strtolower($table) . ':' . strtolower($column);
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        if (!cobertura_sku_has_table($pdo, $table)) {
            return $cache[$key] = false;
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        return $cache[$key] = ((int) $stmt->fetchColumn() > 0);
    }
}
