<?php

declare(strict_types=1);

require_once dirname(__DIR__) . '/InventarioReport.php';

if (!function_exists('distribucion_fetch_filters')) {
    /**
     * Obtiene catálogos para filtros del reporte de distribución.
     */
    function distribucion_fetch_filters(PDO $pdo): array
    {
        $depositos = [];
        if (inventario_has_table($pdo, 'wh_deposito')) {
            $idCol = inventario_detect_column($pdo, 'wh_deposito', ['id']) ?? 'id';
            $codeCol = inventario_detect_column($pdo, 'wh_deposito', ['code', 'codigo']);
            $nameCol = inventario_detect_column($pdo, 'wh_deposito', ['nombre', 'name', 'descripcion']);
            $activoCol = inventario_detect_column($pdo, 'wh_deposito', ['activo']);
            $deletedCol = inventario_detect_column($pdo, 'wh_deposito', ['deleted_at']);

            $select = ['d.`' . $idCol . '` AS id'];
            if ($codeCol) {
                $select[] = 'COALESCE(d.`' . $codeCol . '`, "") AS codigo';
            }
            if ($nameCol) {
                $select[] = 'COALESCE(d.`' . $nameCol . '`, "") AS nombre';
            }

            $where = [];
            if ($activoCol) {
                $where[] = '(d.`' . $activoCol . '` = 1 OR d.`' . $activoCol . '` IS NULL)';
            }
            if ($deletedCol) {
                $where[] = 'd.`' . $deletedCol . '` IS NULL';
            }

            $sql = 'SELECT ' . implode(', ', $select)
                . ' FROM `wh_deposito` d';
            if ($where) {
                $sql .= ' WHERE ' . implode(' AND ', $where);
            }
            $orderBy = $nameCol ?: ($codeCol ?: $idCol);
            $sql .= ' ORDER BY d.`' . $orderBy . '` ASC LIMIT 200';

            foreach ($pdo->query($sql) as $row) {
                $id = isset($row['id']) ? (int) $row['id'] : 0;
                if ($id <= 0) {
                    continue;
                }
                $labelParts = [];
                if (!empty($row['codigo'])) {
                    $labelParts[] = trim((string) $row['codigo']);
                }
                if (!empty($row['nombre'])) {
                    $labelParts[] = trim((string) $row['nombre']);
                }
                $label = $labelParts ? implode(' - ', $labelParts) : ('Depósito #' . $id);
                $depositos[] = ['id' => $id, 'label' => $label];
            }
        }

        $clientes = [];
        if (inventario_has_table($pdo, 'para_clientes')) {
            $sql = 'SELECT id, razon_social FROM para_clientes WHERE 1=1';
            if (inventario_has_column($pdo, 'para_clientes', 'deleted_at')) {
                $sql .= ' AND deleted_at IS NULL';
            }
            if (inventario_has_column($pdo, 'para_clientes', 'activo')) {
                $sql .= ' AND (activo = 1 OR activo IS NULL)';
            }
            $sql .= ' ORDER BY razon_social ASC LIMIT 400';
            foreach ($pdo->query($sql) as $row) {
                $id = isset($row['id']) ? (int) $row['id'] : 0;
                if ($id <= 0) {
                    continue;
                }
                $nombre = trim((string) ($row['razon_social'] ?? ''));
                if ($nombre === '') {
                    $nombre = 'Cliente #' . $id;
                }
                $clientes[] = ['id' => $id, 'label' => $nombre];
            }
        }

        $moviles = [];
        if (inventario_has_table($pdo, 'para_moviles')) {
            $sql = 'SELECT id, chapa FROM para_moviles';
            if (inventario_has_column($pdo, 'para_moviles', 'activo')) {
                $sql .= ' WHERE (activo = 1 OR activo IS NULL)';
            }
            $sql .= ' ORDER BY COALESCE(chapa, "") ASC LIMIT 200';
            foreach ($pdo->query($sql) as $row) {
                $id = isset($row['id']) ? (int) $row['id'] : 0;
                if ($id <= 0) {
                    continue;
                }
                $chapa = trim((string) ($row['chapa'] ?? ''));
                $label = $chapa !== '' ? $chapa : ('Movil #' . $id);
                $moviles[] = ['id' => $id, 'label' => $label];
            }
        }

        return [
            'depositos' => $depositos,
            'clientes'  => $clientes,
            'moviles'   => $moviles,
        ];
    }
}

if (!function_exists('distribucion_bind_params')) {
    /**
     * Enlaza parámetros solo si el marcador existe en la sentencia preparada.
     *
     * @param array<string, mixed> $params
     */
    function distribucion_bind_params(PDOStatement $stmt, array $params, string $sql): void
    {
        if ($params === []) {
            return;
        }

        preg_match_all('/:([a-zA-Z0-9_]+)/', $sql, $matches);
        $placeholders = array_unique($matches[0] ?? []);
        if (!$placeholders) {
            return;
        }

        $lookup = array_flip($placeholders);
        $provided = [];
        foreach ($params as $key => $value) {
            if (!is_string($key) || !isset($lookup[$key])) {
                continue;
            }

            if (is_int($value)) {
                $stmt->bindValue($key, $value, PDO::PARAM_INT);
            } elseif (is_float($value)) {
                $stmt->bindValue($key, $value);
            } else {
                $stmt->bindValue($key, (string) $value);
            }
            $provided[$key] = true;
        }

        foreach ($placeholders as $placeholder) {
            if (!isset($provided[$placeholder])) {
                $stmt->bindValue($placeholder, null, PDO::PARAM_NULL);
            }
        }
    }
}

if (!function_exists('distribucion_prepare_execute')) {
    /**
     * Prepara y ejecuta una sentencia con manejo de parámetros dinámicos.
     *
     * @param array<string, mixed> $params
     * @param callable|null $afterBind Callback opcional para enlazar parámetros adicionales.
     * @throws RuntimeException
     */
    function distribucion_prepare_execute(PDO $pdo, string $sql, array $params = [], ?callable $afterBind = null): PDOStatement
    {
        $stmt = $pdo->prepare($sql);
        distribucion_bind_params($stmt, $params, $sql);

        if ($afterBind !== null) {
            $afterBind($stmt);
        }

        try {
            $stmt->execute();
        } catch (Throwable $e) {
            $message = $e->getMessage() . ' | SQL: ' . $sql . ' | Params: ' . json_encode($params, JSON_PARTIAL_OUTPUT_ON_ERROR);
            throw new RuntimeException($message, (int) ($e->getCode() ?: 0), $e);
        }

        return $stmt;
    }
}

if (!function_exists('distribucion_fetch_data')) {
    /**
     * Devuelve métricas consolidadas para el reporte de distribución.
     *
     * @param array<string, mixed> $rawFilters
     * @return array<string, mixed>
     */
    function distribucion_fetch_data(PDO $pdo, array $rawFilters): array
    {
        $filters = distribucion_normalize_filters($rawFilters);

        $deliveries = distribucion_metric_entregas_tiempo($pdo, $filters);
        $kms = distribucion_metric_kilometros($pdo, $filters);
        $combustible = distribucion_metric_combustible($pdo, $filters, $kms);
        $costos = distribucion_metric_costo_logistico($pdo, $filters);
        $devoluciones = distribucion_metric_devoluciones($pdo, $filters, $deliveries);

        $metrics = [];
        $metrics[] = distribucion_build_metric([
            'clave'   => 'entregas_tiempo',
            'indicador' => 'Entregas a tiempo (%)',
            'formula' => '(Entregas dentro del plazo / total entregas) x 100',
            'meta'    => 98.0,
            'valor'   => $deliveries['porcentaje'] ?? null,
            'unidad'  => '%',
            'observaciones' => $deliveries['observacion'] ?? '',
            'direccion' => 'max',
            'detalle' => $deliveries,
        ]);

        $metrics[] = distribucion_build_metric([
            'clave'   => 'costo_logistico',
            'indicador' => 'Costo logístico sobre ventas (%)',
            'formula' => '(Costos logísticos / Ventas totales) x 100',
            'meta'    => 8.0,
            'valor'   => $costos['porcentaje'] ?? null,
            'unidad'  => '%',
            'observaciones' => $costos['observacion'] ?? '',
            'direccion' => 'min',
            'detalle' => $costos,
        ]);

        $metrics[] = distribucion_build_metric([
            'clave'   => 'km_entrega',
            'indicador' => 'Kilómetros recorridos por entrega',
            'formula' => 'Distancia total / entregas realizadas',
            'meta'    => 50.0,
            'valor'   => $kms['promedio'] ?? null,
            'unidad'  => 'km',
            'observaciones' => $kms['observacion'] ?? '',
            'direccion' => 'min',
            'detalle' => $kms,
        ]);

        $metrics[] = distribucion_build_metric([
            'clave'   => 'combustible',
            'indicador' => 'Gasto de combustible por periodo',
            'formula' => 'Consumo total estimado (litros)',
            'meta'    => $filters['combustible_meta'] ?? null,
            'valor'   => $combustible['litros'] ?? null,
            'unidad'  => 'L',
            'observaciones' => $combustible['observacion'] ?? '',
            'direccion' => 'min',
            'detalle' => $combustible,
        ]);

        $metrics[] = distribucion_build_metric([
            'clave'   => 'devoluciones',
            'indicador' => 'Devoluciones (%)',
            'formula' => '(Pedidos devueltos / pedidos entregados) x 100',
            'meta'    => 1.5,
            'valor'   => $devoluciones['porcentaje'] ?? null,
            'unidad'  => '%',
            'observaciones' => $devoluciones['observacion'] ?? '',
            'direccion' => 'min',
            'detalle' => $devoluciones,
        ]);

        return [
            'filters'   => $filters,
            'periodo'   => distribucion_format_periodo($filters),
            'metrics'   => $metrics,
            'summary'   => distribucion_build_summary($deliveries, $kms, $combustible, $devoluciones, $costos),
        ];
    }
}

if (!function_exists('distribucion_normalize_filters')) {
    /**
     * Sanitiza y normaliza filtros del request.
     *
     * @param array<string, mixed> $raw
     * @return array<string, mixed>
     */
    function distribucion_normalize_filters(array $raw): array
    {
    $defaultHasta = date('Y-m-d');
    $defaultDesde = date('Y-m-d', strtotime('-29 days'));
    $defaultCombustibleMeta = 890.0;

        $fechaDesde = inventario_sanitize_date($raw['fecha_desde'] ?? null);
        $fechaHasta = inventario_sanitize_date($raw['fecha_hasta'] ?? null);
        if ($fechaDesde === '') {
            $fechaDesde = $defaultDesde;
        }
        if ($fechaHasta === '') {
            $fechaHasta = $defaultHasta;
        }
        if ($fechaDesde > $fechaHasta) {
            [$fechaDesde, $fechaHasta] = [$fechaHasta, $fechaDesde];
        }

        $toleranciaMin = isset($raw['tolerancia_min']) ? (int) $raw['tolerancia_min'] : 60;
        if ($toleranciaMin <= 0) {
            $toleranciaMin = 60;
        }

        $combustibleMeta = $defaultCombustibleMeta;
        if (array_key_exists('combustible_meta', $raw) && $raw['combustible_meta'] !== '') {
            $combustibleMeta = max(0.0, (float) $raw['combustible_meta']);
        }

        return [
            'fecha_desde'      => $fechaDesde,
            'fecha_hasta'      => $fechaHasta,
            'cliente_id'       => distribucion_sanitize_id($raw['cliente_id'] ?? null),
            'deposito_id'      => distribucion_sanitize_id($raw['deposito_id'] ?? null),
            'movil_id'         => distribucion_sanitize_id($raw['movil_id'] ?? null),
            'tolerancia_min'   => $toleranciaMin,
            'combustible_meta' => $combustibleMeta,
        ];
    }
}

if (!function_exists('distribucion_sanitize_id')) {
    /**
     * Devuelve string con id entero o cadena vacía.
     * @param mixed $value
     */
    function distribucion_sanitize_id($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('distribucion_format_periodo')) {
    /**
     * Devuelve string con periodo legible.
     * @param array<string, mixed> $filters
     */
    function distribucion_format_periodo(array $filters): string
    {
        return $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'];
    }
}

if (!function_exists('distribucion_build_metric')) {
    /**
     * @param array<string, mixed> $config
     * @return array<string, mixed>
     */
    function distribucion_build_metric(array $config): array
    {
        $valor = isset($config['valor']) && $config['valor'] !== null ? (float) $config['valor'] : null;
        $meta = isset($config['meta']) && $config['meta'] !== null ? (float) $config['meta'] : null;
        $direccion = isset($config['direccion']) && $config['direccion'] === 'min' ? 'min' : 'max';
        $estado = distribucion_indicator_estado($valor, $meta, $direccion);

        return [
            'clave'        => (string) ($config['clave'] ?? ''),
            'indicador'    => (string) ($config['indicador'] ?? ''),
            'formula'      => (string) ($config['formula'] ?? ''),
            'meta'         => $meta,
            'valor'        => $valor,
            'unidad'       => (string) ($config['unidad'] ?? ''),
            'estado'       => $estado,
            'observaciones'=> (string) ($config['observaciones'] ?? ''),
            'direccion'    => $direccion,
            'detalle'      => $config['detalle'] ?? [],
        ];
    }
}

if (!function_exists('distribucion_indicator_estado')) {
    function distribucion_indicator_estado(?float $valor, ?float $meta, string $direccion): string
    {
        if ($valor === null || $meta === null) {
            return 'SIN_DATOS';
        }

        if ($direccion === 'min') {
            if ($valor <= $meta) {
                return 'OK';
            }
            if ($meta <= 0.0) {
                return 'ALERTA';
            }
            $ratio = $valor / $meta;
            if ($ratio <= 1.6) {
                return 'ALERTA';
            }
            return 'CRITICO';
        }

        if ($valor >= $meta) {
            return 'OK';
        }
        if ($meta <= 0.0) {
            return 'ALERTA';
        }
        $ratio = $valor / $meta;
        if ($ratio >= 0.4) {
            return 'ALERTA';
        }
        return 'CRITICO';
    }
}

if (!function_exists('distribucion_metric_entregas_tiempo')) {
    /**
     * @param array<string, mixed> $filters
     * @return array<string, mixed>
     */
    function distribucion_metric_entregas_tiempo(PDO $pdo, array $filters): array
    {
        if (!inventario_has_table($pdo, 'so_embarque') || !inventario_has_table($pdo, 'so_embarque_parada')) {
            return ['observacion' => 'No se encontraron tablas de embarques'];
        }

        $actualParts = [];
        foreach (['hora_fin_descarga', 'hora_salida', 'hora_inicio_descarga', 'hr_salida', 'hr_termino', 'hora_llegada', 'hr_llegada'] as $column) {
            if (inventario_has_column($pdo, 'so_embarque_parada', $column)) {
                $actualParts[] = 'p.`' . $column . '`';
            }
        }
        if (!$actualParts) {
            return ['observacion' => 'No se detectaron columnas de hora de entrega'];
        }
        $actualExpr = 'COALESCE(' . implode(', ', $actualParts) . ')';

        $referenceParts = [];
        foreach (['compromiso_at', 'hora_compromiso', 'ventana_hasta', 'ventana_fin', 'hora_cita', 'hora_programada', 'fecha_compromiso', 'cita_at'] as $column) {
            if (inventario_has_column($pdo, 'so_embarque_parada', $column)) {
                $referenceParts[] = 'p.`' . $column . '`';
            }
        }
        foreach (['salida_at', 'carga_fecha', 'creado_at'] as $column) {
            if (inventario_has_column($pdo, 'so_embarque', $column)) {
                $referenceParts[] = 'e.`' . $column . '`';
            }
        }
        if (!$referenceParts) {
            $referenceParts[] = 'e.`creado_at`';
        }
        $referenceExpr = 'COALESCE(' . implode(', ', $referenceParts) . ')';

        $dateExpr = 'COALESCE(' . $actualExpr . ', ' . $referenceExpr . ')';

        $where = [];
        $params = [];
        $where[] = 'DATE(' . $dateExpr . ') >= :fecha_desde';
        $where[] = 'DATE(' . $dateExpr . ') <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['cliente_id'] !== '') {
            if (inventario_has_table($pdo, 'para_destinatarios') && inventario_has_column($pdo, 'para_destinatarios', 'cliente_id')) {
                $where[] = 'dest.cliente_id = :cliente_id';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            }
        }
        if ($filters['deposito_id'] !== '') {
            if (inventario_has_column($pdo, 'so_embarque', 'deposito_id')) {
                $where[] = 'e.`deposito_id` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            }
        }
        if ($filters['movil_id'] !== '') {
            if (inventario_has_column($pdo, 'so_embarque', 'movil_id')) {
                $where[] = 'e.`movil_id` = :movil_id';
                $params[':movil_id'] = (int) $filters['movil_id'];
            }
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $baseSql = 'SELECT
            CASE WHEN ' . $actualExpr . ' IS NOT NULL THEN 1 ELSE 0 END AS entregado,
            CASE
                WHEN ' . $actualExpr . ' IS NOT NULL AND ' . $referenceExpr . ' IS NOT NULL
                THEN TIMESTAMPDIFF(MINUTE, ' . $referenceExpr . ', ' . $actualExpr . ')
                ELSE NULL
            END AS demora_min
        FROM so_embarque_parada p
        JOIN so_embarque e ON e.id = p.embarque_id
        LEFT JOIN para_destinatarios dest ON dest.id = p.destinatario_id
        ' . $whereSql;

        $sql = 'SELECT
            COUNT(*) AS total_registros,
            SUM(entregado) AS entregas,
            SUM(CASE WHEN entregado = 1 AND demora_min IS NOT NULL AND demora_min <= :tol_en_plazo THEN 1 ELSE 0 END) AS en_plazo,
            SUM(CASE WHEN entregado = 1 AND demora_min IS NOT NULL AND demora_min > :tol_retraso THEN 1 ELSE 0 END) AS con_retraso,
            AVG(CASE WHEN demora_min IS NOT NULL THEN demora_min END) AS demora_promedio,
            AVG(CASE WHEN demora_min > 0 THEN demora_min END) AS demora_promedio_retraso,
            MAX(demora_min) AS demora_maxima
        FROM (' . $baseSql . ') t';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $placeholder => $value) {
            if (!is_string($placeholder) || strpos($sql, $placeholder) === false) {
                continue;
            }
            if (is_int($value)) {
                $stmt->bindValue($placeholder, $value, PDO::PARAM_INT);
            } elseif (is_float($value)) {
                $stmt->bindValue($placeholder, $value);
            } else {
                $stmt->bindValue($placeholder, (string) $value);
            }
        }
        $tolValue = (int) $filters['tolerancia_min'];
        $stmt->bindValue(':tol_en_plazo', $tolValue, PDO::PARAM_INT);
        $stmt->bindValue(':tol_retraso', $tolValue, PDO::PARAM_INT);
        try {
            $stmt->execute();
        } catch (Throwable $e) {
            $msg = $e->getMessage() . ' | SQL: ' . $sql . ' | Params: ' . json_encode($params + [
                ':tol_en_plazo'  => $tolValue,
                ':tol_retraso'   => $tolValue,
            ], JSON_PARTIAL_OUTPUT_ON_ERROR);
            throw new RuntimeException($msg, (int) ($e->getCode() ?: 0), $e);
        }
        $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        $entregas = (int) ($row['entregas'] ?? 0);
        $enPlazo = (int) ($row['en_plazo'] ?? 0);
        $conRetraso = (int) ($row['con_retraso'] ?? 0);
        $porcentaje = null;
        if ($entregas > 0) {
            $porcentaje = round(($enPlazo / $entregas) * 100, 2);
        }

        $observacion = '';
        if ($entregas === 0) {
            $observacion = 'Sin entregas registradas en el periodo';
        } else {
            $observacion = sprintf(
                '%d de %d entregas dentro del umbral de %d minutos (%d con retraso)',
                $enPlazo,
                $entregas,
                (int) $filters['tolerancia_min'],
                $conRetraso
            );
        }

        return [
            'total_registros' => (int) ($row['total_registros'] ?? 0),
            'entregas'        => $entregas,
            'en_plazo'        => $enPlazo,
            'con_retraso'     => $conRetraso,
            'porcentaje'      => $porcentaje,
            'demora_promedio' => $row['demora_promedio'] !== null ? (float) $row['demora_promedio'] : null,
            'demora_promedio_retraso' => $row['demora_promedio_retraso'] !== null ? (float) $row['demora_promedio_retraso'] : null,
            'demora_maxima'   => $row['demora_maxima'] !== null ? (float) $row['demora_maxima'] : null,
            'observacion'     => $observacion,
        ];
    }
}

if (!function_exists('distribucion_metric_kilometros')) {
    /**
     * @param array<string, mixed> $filters
     * @return array<string, mixed>
     */
    function distribucion_metric_kilometros(PDO $pdo, array $filters): array
    {
        if (!inventario_has_table($pdo, 'so_embarque_seguimiento_dest')) {
            return ['observacion' => 'Seguimiento de kilómetros no disponible'];
        }

        $hasKmInicial = inventario_has_column($pdo, 'so_embarque_seguimiento_dest', 'km_inicial');
        $hasKmLlegada = inventario_has_column($pdo, 'so_embarque_seguimiento_dest', 'km_llegada');
        if (!$hasKmInicial || !$hasKmLlegada) {
            return ['observacion' => 'No se encontraron columnas de kilometraje'];
        }

        $dateColumns = [];
        foreach (['hr_salida', 'hr_llegada', 'inicio_carga', 'fin_carga', 'created_at'] as $col) {
            if (inventario_has_column($pdo, 'so_embarque_seguimiento_dest', $col)) {
                $dateColumns[] = 'seg.`' . $col . '`';
            }
        }
        if (inventario_has_column($pdo, 'so_embarque', 'salida_at')) {
            $dateColumns[] = 'e.`salida_at`';
        }
        if (inventario_has_column($pdo, 'so_embarque', 'creado_at')) {
            $dateColumns[] = 'e.`creado_at`';
        }
        if (!$dateColumns) {
            $dateColumns[] = 'CURRENT_DATE';
        }
        $dateExpr = 'COALESCE(' . implode(', ', $dateColumns) . ')';

        $where = [];
        $params = [];
        $where[] = 'DATE(' . $dateExpr . ') >= :fecha_desde';
        $where[] = 'DATE(' . $dateExpr . ') <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['cliente_id'] !== '') {
            if (inventario_has_table($pdo, 'para_destinatarios') && inventario_has_column($pdo, 'para_destinatarios', 'cliente_id')) {
                $where[] = 'dest.cliente_id = :cliente_id';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            }
        }
        if ($filters['deposito_id'] !== '') {
            if (inventario_has_column($pdo, 'so_embarque', 'deposito_id')) {
                $where[] = 'e.`deposito_id` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            }
        }
        if ($filters['movil_id'] !== '') {
            if (inventario_has_column($pdo, 'so_embarque', 'movil_id')) {
                $where[] = 'e.`movil_id` = :movil_id';
                $params[':movil_id'] = (int) $filters['movil_id'];
            }
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = 'SELECT
            COUNT(*) AS registros,
            SUM(CASE
                    WHEN seg.`km_llegada` IS NOT NULL AND seg.`km_inicial` IS NOT NULL
                    THEN GREATEST(seg.`km_llegada` - seg.`km_inicial`, 0)
                    ELSE 0
                END) AS km_totales
        FROM so_embarque_seguimiento_dest seg
        JOIN so_embarque e ON e.id = seg.embarque_id
        LEFT JOIN para_destinatarios dest ON dest.id = seg.destinatario_id
        ' . $whereSql;

        $stmt = distribucion_prepare_execute($pdo, $sql, $params);
        $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        $registros = (int) ($row['registros'] ?? 0);
        $kmTotales = (float) ($row['km_totales'] ?? 0.0);
        $promedio = $registros > 0 ? ($kmTotales / $registros) : null;

        $observacion = $registros > 0
            ? sprintf('%.1f km totales en %d entregas (~%.1f km promedio)', $kmTotales, $registros, $promedio ?? 0.0)
            : 'Sin registros de seguimiento en el periodo';

        return [
            'registros'    => $registros,
            'km_total'     => $kmTotales,
            'promedio'     => $promedio !== null ? round($promedio, 2) : null,
            'observacion'  => $observacion,
        ];
    }
}

if (!function_exists('distribucion_metric_combustible')) {
    /**
     * @param array<string, mixed> $filters
     * @param array<string, mixed> $kms
     * @return array<string, mixed>
     */
    function distribucion_metric_combustible(PDO $pdo, array $filters, array $kms): array
    {
        if (!inventario_has_table($pdo, 'so_embarque')) {
            return ['observacion' => 'No hay datos de embarques'];
        }

        $litros = null;
        $observacion = '';
        $rows = 0;

        $combustibleCol = null;
        foreach (['combustible_litros', 'litros_consumidos', 'fuel_liters', 'litros'] as $candidate) {
            if (inventario_has_column($pdo, 'so_embarque', $candidate)) {
                $combustibleCol = 'e.`' . $candidate . '`';
                break;
            }
        }

        $where = [];
        $params = [];
        $dateParts = [];
        if (inventario_has_column($pdo, 'so_embarque', 'salida_at')) {
            $dateParts[] = 'e.`salida_at`';
        }
        if (inventario_has_column($pdo, 'so_embarque', 'creado_at')) {
            $dateParts[] = 'e.`creado_at`';
        }
        $dateExpr = $dateParts ? 'COALESCE(' . implode(', ', $dateParts) . ')' : 'CURRENT_DATE';
        $where[] = 'DATE(' . $dateExpr . ') >= :fecha_desde';
        $where[] = 'DATE(' . $dateExpr . ') <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['cliente_id'] !== '' && inventario_has_table($pdo, 'para_destinatarios') && inventario_has_table($pdo, 'so_embarque_parada')) {
            $where[] = 'EXISTS (
                SELECT 1 FROM so_embarque_parada p
                JOIN para_destinatarios dest ON dest.id = p.destinatario_id
                WHERE p.embarque_id = e.id AND dest.cliente_id = :cliente_id
            )';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }
        if ($filters['deposito_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'deposito_id')) {
            $where[] = 'e.`deposito_id` = :deposito_id';
            $params[':deposito_id'] = (int) $filters['deposito_id'];
        }
        if ($filters['movil_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'movil_id')) {
            $where[] = 'e.`movil_id` = :movil_id';
            $params[':movil_id'] = (int) $filters['movil_id'];
        }

        if ($combustibleCol !== null) {
            $sql = 'SELECT SUM(COALESCE(' . $combustibleCol . ', 0)) AS litros, COUNT(*) AS registros
                    FROM so_embarque e
                    ' . ($where ? ('WHERE ' . implode(' AND ', $where)) : '');
            $stmt = distribucion_prepare_execute($pdo, $sql, $params);
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
            $litros = (float) ($row['litros'] ?? 0.0);
            $rows = (int) ($row['registros'] ?? 0);
            $observacion = $rows > 0
                ? sprintf('Consumo reportado en %d embarques', $rows)
                : 'Sin datos de consumo registrados';
        } else {
            // Aproximación utilizando kilómetros y rendimiento del móvil
            $rendimientoCol = null;
            if (inventario_has_table($pdo, 'para_moviles')) {
                foreach (['rendimiento_km_litro', 'rendimiento_km_l', 'rendimiento', 'consumo_km_l'] as $candidate) {
                    if (inventario_has_column($pdo, 'para_moviles', $candidate)) {
                        $rendimientoCol = '`' . $candidate . '`';
                        break;
                    }
                }
            }
            if ($rendimientoCol !== null && (($kms['km_total'] ?? 0) > 0)) {
                $sql = 'SELECT AVG(CASE WHEN ' . $rendimientoCol . ' > 0 THEN ' . $rendimientoCol . ' END) AS promedio
                        FROM para_moviles';
                if ($filters['movil_id'] !== '') {
                    $sql .= ' WHERE id = :movil_id';
                }
                $stmt = distribucion_prepare_execute(
                    $pdo,
                    $sql,
                    [],
                    static function (PDOStatement $stmt) use ($filters): void {
                        if ($filters['movil_id'] !== '') {
                            $stmt->bindValue(':movil_id', (int) $filters['movil_id'], PDO::PARAM_INT);
                        }
                    }
                );
                $rendimiento = $stmt->fetchColumn();
                $rendimiento = $rendimiento !== false ? (float) $rendimiento : null;
                if ($rendimiento !== null && $rendimiento > 0) {
                    $litros = ($kms['km_total'] ?? 0.0) / $rendimiento;
                    $observacion = sprintf('Estimado usando %.1f km totales y rendimiento promedio %.2f km/L', $kms['km_total'] ?? 0.0, $rendimiento);
                } else {
                    $observacion = 'No hay datos de rendimiento para estimar consumo';
                }
            } else {
                $observacion = 'No se pudo estimar consumo de combustible';
            }
        }

        if ($litros !== null) {
            $litros = round($litros, 2);
        }

        return [
            'litros'      => $litros,
            'registros'   => $rows,
            'observacion' => $observacion,
        ];
    }
}

if (!function_exists('distribucion_metric_costo_logistico')) {
    /**
     * @param array<string, mixed> $filters
     * @return array<string, mixed>
     */
    function distribucion_metric_costo_logistico(PDO $pdo, array $filters): array
    {
        if (!inventario_has_table($pdo, 'so_embarque')) {
            return ['observacion' => 'No hay datos de embarques'];
        }

        $costos = null;
        $facturas = 0;
        if (inventario_has_table($pdo, 'so_embarque_rendiciones') && inventario_has_column($pdo, 'so_embarque_rendiciones', 'monto')) {
            $where = [];
            $params = [];
            $dateParts = [];
            foreach (['fecha_rendicion', 'fecha_factura', 'creado_at'] as $col) {
                if (inventario_has_column($pdo, 'so_embarque_rendiciones', $col)) {
                    $dateParts[] = 'r.`' . $col . '`';
                }
            }
            if (inventario_has_column($pdo, 'so_embarque', 'salida_at')) {
                $dateParts[] = 'e.`salida_at`';
            }
            if (inventario_has_column($pdo, 'so_embarque', 'creado_at')) {
                $dateParts[] = 'e.`creado_at`';
            }
            $dateExpr = $dateParts ? 'COALESCE(' . implode(', ', $dateParts) . ')' : 'CURRENT_DATE';
            $where[] = 'DATE(' . $dateExpr . ') >= :fecha_desde';
            $where[] = 'DATE(' . $dateExpr . ') <= :fecha_hasta';
            $params[':fecha_desde'] = $filters['fecha_desde'];
            $params[':fecha_hasta'] = $filters['fecha_hasta'];

            if ($filters['deposito_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'deposito_id')) {
                $where[] = 'e.`deposito_id` = :deposito_id';
                $params[':deposito_id'] = (int) $filters['deposito_id'];
            }
            if ($filters['movil_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'movil_id')) {
                $where[] = 'e.`movil_id` = :movil_id';
                $params[':movil_id'] = (int) $filters['movil_id'];
            }
            if ($filters['cliente_id'] !== '' && inventario_has_table($pdo, 'so_embarque_parada') && inventario_has_table($pdo, 'para_destinatarios')) {
                $where[] = 'EXISTS (
                    SELECT 1 FROM so_embarque_parada p
                    JOIN para_destinatarios dest ON dest.id = p.destinatario_id
                    WHERE p.embarque_id = e.id AND dest.cliente_id = :cliente_id
                )';
                $params[':cliente_id'] = (int) $filters['cliente_id'];
            }

            $sql = 'SELECT SUM(COALESCE(r.`monto`, 0)) AS total, COUNT(*) AS facturas
                    FROM so_embarque_rendiciones r
                    JOIN so_embarque e ON e.id = r.embarque_id'
                . ($where ? (' WHERE ' . implode(' AND ', $where)) : '');
            $stmt = distribucion_prepare_execute($pdo, $sql, $params);
            $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
            $costos = (float) ($row['total'] ?? 0.0);
            $facturas = (int) ($row['facturas'] ?? 0);
        }

        $valorMercaderia = distribucion_calcular_valor_mercaderia($pdo, $filters);
        $porcentaje = null;
        if ($costos !== null && $valorMercaderia > 0) {
            $porcentaje = round(($costos / $valorMercaderia) * 100, 2);
        }

        $observacion = '';
        if ($costos === null) {
            $observacion = 'Sin datos de rendiciones para estimar costos';
        } else {
            $observacion = sprintf('Costos logísticos: %s en %d facturas. Valor mercancía estimado: %s',
                distribucion_format_currency($costos),
                $facturas,
                distribucion_format_currency($valorMercaderia)
            );
        }
        if ($porcentaje === null) {
            $observacion .= ' (sin suficiente información para calcular porcentaje)';
        }

        return [
            'costos'      => $costos,
            'valor'       => $valorMercaderia,
            'porcentaje'  => $porcentaje,
            'observacion' => $observacion,
        ];
    }
}

if (!function_exists('distribucion_calcular_valor_mercaderia')) {
    /**
     * Calcula valor de mercadería despachada usando costos unitarios de productos.
     * @param array<string, mixed> $filters
     */
    function distribucion_calcular_valor_mercaderia(PDO $pdo, array $filters): float
    {
        if (!inventario_has_table($pdo, 'so_embarque_pre') || !inventario_has_table($pdo, 'so_preembarque') || !inventario_has_table($pdo, 'so_pedido_dest_item')) {
            return 0.0;
        }

        $shipUcCols = [];
        foreach (['shipped_uc', 'entregado_uc', 'uc_enviado', 'uc_despachado'] as $col) {
            if (inventario_has_column($pdo, 'so_pedido_dest_item', $col)) {
                $shipUcCols[] = 'i.`' . $col . '`';
            }
        }
        $shipUvCols = [];
        foreach (['shipped_uv', 'entregado_uv', 'uv_enviado', 'uv_despachado'] as $col) {
            if (inventario_has_column($pdo, 'so_pedido_dest_item', $col)) {
                $shipUvCols[] = 'i.`' . $col . '`';
            }
        }

        $expectedUcCol = inventario_has_column($pdo, 'so_pedido_dest_item', 'expected_uc') ? 'i.`expected_uc`' : '0';
        $expectedUvCol = inventario_has_column($pdo, 'so_pedido_dest_item', 'expected_uv') ? 'i.`expected_uv`' : '0';

        $ucExpr = $shipUcCols ? 'COALESCE(' . implode(', ', $shipUcCols) . ', ' . $expectedUcCol . ')' : $expectedUcCol;
        $uvExpr = $shipUvCols ? 'COALESCE(' . implode(', ', $shipUvCols) . ', ' . $expectedUvCol . ')' : $expectedUvCol;

        $where = [];
        $params = [];
        $dateParts = [];
        if (inventario_has_column($pdo, 'so_embarque', 'salida_at')) {
            $dateParts[] = 'e.`salida_at`';
        }
        if (inventario_has_column($pdo, 'so_embarque', 'creado_at')) {
            $dateParts[] = 'e.`creado_at`';
        }
        if (inventario_has_column($pdo, 'so_preembarque', 'creado_at')) {
            $dateParts[] = 'pre.`creado_at`';
        }
        $dateExpr = $dateParts ? 'COALESCE(' . implode(', ', $dateParts) . ')' : 'CURRENT_DATE';
        $where[] = 'DATE(' . $dateExpr . ') >= :fecha_desde';
        $where[] = 'DATE(' . $dateExpr . ') <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['deposito_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'deposito_id')) {
            $where[] = 'e.`deposito_id` = :deposito_id';
            $params[':deposito_id'] = (int) $filters['deposito_id'];
        }
        if ($filters['movil_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'movil_id')) {
            $where[] = 'e.`movil_id` = :movil_id';
            $params[':movil_id'] = (int) $filters['movil_id'];
        }
        if ($filters['cliente_id'] !== '' && inventario_has_table($pdo, 'para_destinatarios')) {
            $where[] = 'dest.cliente_id = :cliente_id';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = 'SELECT
            i.producto_id,
            SUM(' . $ucExpr . ') AS total_uc,
            SUM(' . $uvExpr . ') AS total_uv
        FROM so_embarque_pre ep
        JOIN so_embarque e ON e.id = ep.embarque_id
        JOIN so_preembarque pre ON pre.id = ep.preembarque_id
        JOIN so_pedido ped ON ped.id = pre.pedido_id
        JOIN so_pedido_dest pd ON pd.pedido_id = ped.id
        JOIN so_pedido_dest_item i ON i.pedido_dest_id = pd.id
        LEFT JOIN para_destinatarios dest ON dest.id = pd.destinatario_id
        ' . $whereSql . '
        GROUP BY i.producto_id';

        $stmt = $pdo->prepare($sql);
        distribucion_bind_params($stmt, $params, $sql);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        if (!$rows) {
            return 0.0;
        }

        $productIds = [];
        foreach ($rows as $row) {
            $pid = isset($row['producto_id']) ? (int) $row['producto_id'] : 0;
            if ($pid > 0) {
                $productIds[] = $pid;
            }
        }
        $productInfo = inventario_fetch_product_info($pdo, $productIds);

        $totalValor = 0.0;
        foreach ($rows as $row) {
            $pid = isset($row['producto_id']) ? (int) $row['producto_id'] : 0;
            if ($pid <= 0) {
                continue;
            }
            $info = $productInfo[$pid] ?? null;
            $costoUnitario = $info['costo_unitario'] ?? null;
            if ($costoUnitario === null || $costoUnitario <= 0) {
                $costoUnitario = 15000.0; // Fallback
            }
            $uc = (float) ($row['total_uc'] ?? 0.0);
            $uv = (float) ($row['total_uv'] ?? 0.0);
            $unitsPerCase = $info['unidades_por_caja'] ?? null;
            $casesPerPallet = $info['cajas_por_pallet'] ?? null;

            $uvUnits = distribucion_convert_uv_to_uc($uv, $unitsPerCase, $casesPerPallet);
            $unidadesTotales = $uc + $uvUnits;
            if ($unidadesTotales <= 0) {
                continue;
            }
            $totalValor += $unidadesTotales * $costoUnitario;
        }

        return max(0.0, round($totalValor, 2));
    }
}

if (!function_exists('distribucion_convert_uv_to_uc')) {
    function distribucion_convert_uv_to_uc(float $uv, ?float $unitsPerCase, ?float $casesPerPallet): float
    {
        if ($uv <= 0) {
            return 0.0;
        }
        if ($casesPerPallet !== null && $casesPerPallet > 0 && $unitsPerCase !== null && $unitsPerCase > 0) {
            return $uv * $casesPerPallet * $unitsPerCase;
        }
        if ($unitsPerCase !== null && $unitsPerCase > 0) {
            return $uv * $unitsPerCase;
        }
        if ($casesPerPallet !== null && $casesPerPallet > 0) {
            return $uv * $casesPerPallet;
        }
        return $uv;
    }
}

if (!function_exists('distribucion_metric_devoluciones')) {
    /**
     * @param array<string, mixed> $filters
     * @param array<string, mixed> $deliveries
     * @return array<string, mixed>
     */
    function distribucion_metric_devoluciones(PDO $pdo, array $filters, array $deliveries): array
    {
        if (!inventario_has_table($pdo, 'so_devolucion') || !inventario_has_table($pdo, 'so_retorno')) {
            return ['observacion' => 'No hay datos de devoluciones'];
        }

        $dateExpr = 'COALESCE(r.`llegada_at`, r.`created_at`, d.`created_at`)';
        $where = [];
        $params = [];
        $where[] = 'DATE(' . $dateExpr . ') >= :fecha_desde';
        $where[] = 'DATE(' . $dateExpr . ') <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['deposito_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'deposito_id')) {
            $where[] = 'e.`deposito_id` = :deposito_id';
            $params[':deposito_id'] = (int) $filters['deposito_id'];
        }
        if ($filters['movil_id'] !== '' && inventario_has_column($pdo, 'so_embarque', 'movil_id')) {
            $where[] = 'e.`movil_id` = :movil_id';
            $params[':movil_id'] = (int) $filters['movil_id'];
        }
        if ($filters['cliente_id'] !== '' && inventario_has_table($pdo, 'para_destinatarios') && inventario_has_column($pdo, 'para_destinatarios', 'cliente_id')) {
            $where[] = 'dest.`cliente_id` = :cliente_id';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = 'SELECT COUNT(*) AS total
                FROM so_devolucion d
                JOIN so_retorno r ON r.id = d.retorno_id
                JOIN so_embarque e ON e.id = r.embarque_id
                LEFT JOIN para_destinatarios dest ON dest.id = d.destinatario_id
                ' . $whereSql;

        $stmt = $pdo->prepare($sql);
        distribucion_bind_params($stmt, $params, $sql);
        $stmt->execute();
        $totalDevoluciones = (int) ($stmt->fetchColumn() ?: 0);

        $entregas = (int) ($deliveries['entregas'] ?? 0);
        $porcentaje = null;
        if ($entregas > 0) {
            $porcentaje = round(($totalDevoluciones / $entregas) * 100, 2);
        }

        $observacion = $entregas > 0
            ? sprintf('%d devoluciones sobre %d entregas', $totalDevoluciones, $entregas)
            : 'Sin entregas registradas para evaluar devoluciones';

        return [
            'devoluciones' => $totalDevoluciones,
            'entregas'     => $entregas,
            'porcentaje'   => $porcentaje,
            'observacion'  => $observacion,
        ];
    }
}

if (!function_exists('distribucion_build_summary')) {
    /**
     * @param array<string, mixed> $deliveries
     * @param array<string, mixed> $kms
     * @param array<string, mixed> $combustible
     * @param array<string, mixed> $devoluciones
     * @param array<string, mixed> $costos
     * @return array<string, mixed>
     */
    function distribucion_build_summary(array $deliveries, array $kms, array $combustible, array $devoluciones, array $costos): array
    {
        return [
            'entregas'          => $deliveries['entregas'] ?? 0,
            'en_plazo'          => $deliveries['en_plazo'] ?? 0,
            'con_retraso'       => $deliveries['con_retraso'] ?? 0,
            'puntualidad_pct'   => $deliveries['porcentaje'] ?? null,
            'km_totales'        => $kms['km_total'] ?? null,
            'km_promedio'       => $kms['promedio'] ?? null,
            'combustible_litros'=> $combustible['litros'] ?? null,
            'devoluciones'      => $devoluciones['devoluciones'] ?? 0,
            'devoluciones_pct'  => $devoluciones['porcentaje'] ?? null,
            'costos'            => $costos['costos'] ?? null,
            'valor_mercaderia'  => $costos['valor'] ?? null,
        ];
    }
}

if (!function_exists('distribucion_format_currency')) {
    function distribucion_format_currency(?float $value): string
    {
        if ($value === null) {
            return '-';
        }
        return number_format($value, 0, ',', '.');
    }
}
