<?php

declare(strict_types=1);

$prepHelpers = dirname(__DIR__) . '/SoPreparacionErrors.php';
if (is_file($prepHelpers)) {
    require_once $prepHelpers;
}

if (!function_exists('errores_picking_fetch_combos')) {
    /**
     * Devuelve listas para filtros del reporte.
     */
    function errores_picking_fetch_combos(PDO $pdo): array
    {
        $motivos = function_exists('so_prep_err_fetch_motivos') ? so_prep_err_fetch_motivos($pdo) : [];
        $clientes = [];

        try {
            $stmt = $pdo->query('SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social LIMIT 400');
            $rawClientes = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
        } catch (Throwable $e) {
            $rawClientes = [];
        }

        if ($rawClientes) {
            $clientes = array_map(static function (array $row): array {
                $label = trim((string) ($row['razon_social'] ?? ''));
                return [
                    'id' => (int) ($row['id'] ?? 0),
                    'nombre' => $label !== '' ? $label : 'Sin cliente',
                ];
            }, $rawClientes);
        }

        $defaultMotivoId = errores_picking_detect_default_motivo($motivos);

        return [
            'motivos' => $motivos,
            'clientes' => $clientes,
            'default_motivo_id' => $defaultMotivoId,
        ];
    }
}

if (!function_exists('errores_picking_fetch_data')) {
    /**
     * Obtiene filas del reporte en base a filtros aplicados.
     */
    function errores_picking_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = errores_picking_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        $table = function_exists('so_prep_err_table_name') ? so_prep_err_table_name() : 'so_pre_error_log';
        if (!function_exists('so_prep_err_table_exists') || !so_prep_err_table_exists($pdo, $table)) {
            return [
                'filters' => $filters,
                'rows' => [],
                'summary' => errores_picking_build_summary([], $filters, null),
                'aggregates' => errores_picking_build_aggregates([]),
                'limit' => $limit,
                'truncated' => false,
            ];
        }

        $motivosTable = 'para_motivos_errores';
        $pedidoTable = 'so_pedido';
        $preTable = 'so_preembarque';
        $usersTable = 'sys_users';
        $clientesTable = 'para_clientes';

        $select = [
            'e.id',
            'e.so_id',
            'e.pre_id',
            'e.motivo_id',
            'e.responsable_user_id',
            'e.responsable_nombre',
            'e.observacion',
            'e.logged_at',
            'e.created_at',
            "DATE_FORMAT(e.logged_at, '%Y-%m-%d') AS logged_fecha",
            "DATE_FORMAT(e.logged_at, '%H:%i') AS logged_hora",
        ];

        $joins = [];

        $motivoLabelCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $motivosTable, ['descripcion', 'nombre', 'detalle', 'motivo', 'titulo', 'label']) : null;
        if (function_exists('so_prep_err_table_exists') && so_prep_err_table_exists($pdo, $motivosTable)) {
            $select[] = $motivoLabelCol ? "m.`{$motivoLabelCol}` AS motivo_nombre" : "CONCAT('Motivo #', m.id) AS motivo_nombre";
            $joins[] = "LEFT JOIN {$motivosTable} m ON m.id = e.motivo_id";
        } else {
            $select[] = "CONCAT('Motivo #', e.motivo_id) AS motivo_nombre";
        }

        $pedidoCodigoCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $pedidoTable, ['codigo', 'code']) : null;
        $pedidoClienteCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $pedidoTable, ['cliente_id', 'client_id', 'para_cliente_id']) : null;
        $pedidoFechaCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $pedidoTable, ['fecha', 'fecha_pedido', 'created_at']) : null;
        $pedidoRefCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $pedidoTable, ['cliente_ref', 'cliente_codigo', 'client_ref']) : null;

        if (function_exists('so_prep_err_table_exists') && so_prep_err_table_exists($pdo, $pedidoTable)) {
            $select[] = $pedidoCodigoCol ? "p.`{$pedidoCodigoCol}` AS pedido_codigo" : "CONCAT('PED-', e.so_id) AS pedido_codigo";
            $select[] = $pedidoClienteCol ? "p.`{$pedidoClienteCol}` AS pedido_cliente_id" : 'NULL AS pedido_cliente_id';
            $select[] = $pedidoRefCol ? "p.`{$pedidoRefCol}` AS pedido_cliente_ref" : 'NULL AS pedido_cliente_ref';
            $select[] = $pedidoFechaCol ? "p.`{$pedidoFechaCol}` AS pedido_fecha" : 'NULL AS pedido_fecha';
            $joins[] = "LEFT JOIN {$pedidoTable} p ON p.id = e.so_id";
        } else {
            $select[] = 'NULL AS pedido_codigo';
            $select[] = 'NULL AS pedido_cliente_id';
            $select[] = 'NULL AS pedido_cliente_ref';
            $select[] = 'NULL AS pedido_fecha';
        }

        $clienteNombreCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $clientesTable, ['razon_social', 'nombre', 'descripcion', 'titulo']) : null;
        if ($pedidoClienteCol && $clienteNombreCol && function_exists('so_prep_err_table_exists') && so_prep_err_table_exists($pdo, $clientesTable)) {
            $select[] = "cli.`{$clienteNombreCol}` AS cliente_nombre";
            $joins[] = "LEFT JOIN {$clientesTable} cli ON cli.id = p.`{$pedidoClienteCol}`";
        } else {
            $select[] = 'NULL AS cliente_nombre';
        }

        $preCodigoCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $preTable, ['codigo', 'code']) : null;
        if ($preCodigoCol && function_exists('so_prep_err_table_exists') && so_prep_err_table_exists($pdo, $preTable)) {
            $select[] = "pre.`{$preCodigoCol}` AS pre_codigo";
            $joins[] = "LEFT JOIN {$preTable} pre ON pre.id = e.pre_id";
        } else {
            $select[] = 'NULL AS pre_codigo';
        }

        $userNameCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $usersTable, ['full_name', 'nombre', 'name']) : null;
        $userAltCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, $usersTable, ['username', 'user', 'email']) : null;
        $responsableParts = ["NULLIF(TRIM(e.responsable_nombre), '')"];
        if (function_exists('so_prep_err_table_exists') && so_prep_err_table_exists($pdo, $usersTable)) {
            $joins[] = "LEFT JOIN {$usersTable} u ON u.id = e.responsable_user_id";
            $select[] = 'u.id AS user_id';
            if ($userNameCol) {
                $responsableParts[] = "NULLIF(TRIM(u.`{$userNameCol}`), '')";
            }
            if ($userAltCol && $userAltCol !== $userNameCol) {
                $responsableParts[] = "NULLIF(TRIM(u.`{$userAltCol}`), '')";
            }
            $responsableParts[] = "CONCAT('Usuario #', u.id)";
            $responsableParts[] = "CONCAT('Usuario #', e.responsable_user_id)";
        } else {
            $select[] = 'NULL AS user_id';
            $responsableParts[] = "CONCAT('Usuario #', e.responsable_user_id)";
        }

        $responsableExpr = 'COALESCE(' . implode(', ', $responsableParts) . ')';
        $select[] = $responsableExpr . ' AS responsable_label';

        $where = [];
        $params = [];

        $where[] = 'DATE(e.logged_at) >= :fecha_desde';
        $where[] = 'DATE(e.logged_at) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['motivo_id'] !== '') {
            $where[] = 'e.motivo_id = :motivo_id';
            $params[':motivo_id'] = (int) $filters['motivo_id'];
        }

        if ($filters['cliente_id'] !== '' && $pedidoClienteCol) {
            $where[] = "p.`{$pedidoClienteCol}` = :cliente_id";
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        if ($filters['codigo'] !== '' && $pedidoCodigoCol) {
            $where[] = "p.`{$pedidoCodigoCol}` LIKE :codigo";
            $params[':codigo'] = '%' . $filters['codigo'] . '%';
        }

        if ($filters['responsable'] !== '') {
            $where[] = 'LOWER(' . $responsableExpr . ') LIKE :responsable';
            $params[':responsable'] = '%' . mb_strtolower($filters['responsable']) . '%';
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = 'SELECT ' . implode(', ', $select) . " FROM {$table} e";
        if ($joins) {
            $sql .= ' ' . implode(' ', $joins);
        }
        $sql .= ' ' . $whereSql . ' ORDER BY e.logged_at DESC, e.id DESC LIMIT :limit_plus';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_plus', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();

        $rawRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $truncated = false;
        if (count($rawRows) > $limit) {
            $rawRows = array_slice($rawRows, 0, $limit);
            $truncated = true;
        }

        $rows = array_map('errores_picking_format_row', $rawRows);

        $motivoLabel = null;
        if ($filters['motivo_id'] !== '') {
            $motivoLabel = errores_picking_lookup_motivo_label($pdo, (int) $filters['motivo_id']);
        }

        $summary = errores_picking_build_summary($rows, $filters, $motivoLabel);
        $aggregates = errores_picking_build_aggregates($rows);

        return [
            'filters' => $filters,
            'rows' => $rows,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('errores_picking_build_summary')) {
    /**
     * Calcula indicadores globales para la vista.
     */
    function errores_picking_build_summary(array $rows, array $filters, ?string $motivoLabel): array
    {
        $total = count($rows);
        $pedidos = [];
        $clientes = [];
        $responsables = [];

        foreach ($rows as $row) {
            $pedidoId = (int) ($row['so_id'] ?? 0);
            if ($pedidoId > 0) {
                $pedidos[$pedidoId] = true;
            }
            $clienteId = (int) ($row['cliente_id'] ?? 0);
            if ($clienteId > 0) {
                $clientes[$clienteId] = true;
            }
            $responsableKey = trim((string) ($row['responsable_label'] ?? '')); 
            if ($responsableKey !== '') {
                $responsables[$responsableKey] = true;
            }
        }

        $desde = $filters['fecha_desde'];
        $hasta = $filters['fecha_hasta'];
        $days = 1;
        $tsDesde = strtotime($desde . ' 00:00:00');
        $tsHasta = strtotime($hasta . ' 00:00:00');
        if ($tsDesde && $tsHasta && $tsHasta >= $tsDesde) {
            $days = max(1, (int) floor(($tsHasta - $tsDesde) / 86400) + 1);
        }

        $promedio = $total > 0 ? round($total / $days, 2) : 0.0;

        $topMotivo = errores_picking_detect_top_motivo($rows);

        return [
            'total_errores' => $total,
            'total_pedidos' => count($pedidos),
            'total_clientes' => count($clientes),
            'total_responsables' => count($responsables),
            'promedio_diario' => $promedio,
            'range_label' => $desde . ' al ' . $hasta,
            'motivo_filter_label' => $motivoLabel,
            'motivo_top_label' => $topMotivo['label'] ?? null,
            'motivo_top_total' => $topMotivo['rows'] ?? 0,
        ];
    }
}

if (!function_exists('errores_picking_build_aggregates')) {
    /**
     * Agrega datos por motivo, responsable y cliente.
     */
    function errores_picking_build_aggregates(array $rows): array
    {
        if (!$rows) {
            return [
                'motivos' => [],
                'responsables' => [],
                'clientes' => [],
            ];
        }

        $motivos = [];
        $responsables = [];
        $clientes = [];

        foreach ($rows as $row) {
            $pedidoId = (int) ($row['so_id'] ?? 0);
            $clienteId = (int) ($row['cliente_id'] ?? 0);

            $motivoKey = (string) ($row['motivo_id'] ?? '');
            if (!isset($motivos[$motivoKey])) {
                $motivos[$motivoKey] = [
                    'key' => $motivoKey,
                    'label' => $row['motivo_nombre'] ?? 'Sin motivo',
                    'rows' => 0,
                    'pedidos' => [],
                    'clientes' => [],
                ];
            }
            $motivos[$motivoKey]['rows']++;
            if ($pedidoId > 0) {
                $motivos[$motivoKey]['pedidos'][$pedidoId] = true;
            }
            if ($clienteId > 0) {
                $motivos[$motivoKey]['clientes'][$clienteId] = true;
            }

            $responsableKey = trim((string) ($row['responsable_label'] ?? 'Sin responsable'));
            if ($responsableKey === '') {
                $responsableKey = 'Sin responsable';
            }
            if (!isset($responsables[$responsableKey])) {
                $responsables[$responsableKey] = [
                    'key' => $responsableKey,
                    'label' => $responsableKey,
                    'rows' => 0,
                    'pedidos' => [],
                ];
            }
            $responsables[$responsableKey]['rows']++;
            if ($pedidoId > 0) {
                $responsables[$responsableKey]['pedidos'][$pedidoId] = true;
            }

            $clienteKey = $clienteId > 0 ? (string) $clienteId : '0';
            $clienteLabel = trim((string) ($row['cliente_nombre'] ?? ''));
            if ($clienteLabel === '') {
                $clienteLabel = 'Sin cliente';
            }
            if (!isset($clientes[$clienteKey])) {
                $clientes[$clienteKey] = [
                    'key' => $clienteKey,
                    'label' => $clienteLabel,
                    'rows' => 0,
                    'pedidos' => [],
                ];
            }
            $clientes[$clienteKey]['rows']++;
            if ($pedidoId > 0) {
                $clientes[$clienteKey]['pedidos'][$pedidoId] = true;
            }
        }

        $motivoList = array_map(static function (array $item): array {
            $item['pedidos'] = count($item['pedidos']);
            $item['clientes'] = count($item['clientes']);
            return $item;
        }, array_values($motivos));
        usort($motivoList, static function (array $a, array $b): int {
            $cmp = ($b['rows'] ?? 0) <=> ($a['rows'] ?? 0);
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) ($a['label'] ?? ''), (string) ($b['label'] ?? ''));
        });

        $responsableList = array_map(static function (array $item): array {
            $item['pedidos'] = count($item['pedidos']);
            return $item;
        }, array_values($responsables));
        usort($responsableList, static function (array $a, array $b): int {
            $cmp = ($b['rows'] ?? 0) <=> ($a['rows'] ?? 0);
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) ($a['label'] ?? ''), (string) ($b['label'] ?? ''));
        });

        $clienteList = array_map(static function (array $item): array {
            $item['pedidos'] = count($item['pedidos']);
            return $item;
        }, array_values($clientes));
        usort($clienteList, static function (array $a, array $b): int {
            $cmp = ($b['rows'] ?? 0) <=> ($a['rows'] ?? 0);
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) ($a['label'] ?? ''), (string) ($b['label'] ?? ''));
        });

        return [
            'motivos' => array_slice($motivoList, 0, 10),
            'responsables' => array_slice($responsableList, 0, 10),
            'clientes' => array_slice($clienteList, 0, 10),
        ];
    }
}

if (!function_exists('errores_picking_format_row')) {
    function errores_picking_format_row(array $row): array
    {
        $loggedAt = (string) ($row['logged_at'] ?? '');
        $loggedFecha = (string) ($row['logged_fecha'] ?? '');
        $loggedHora = (string) ($row['logged_hora'] ?? '');

        if ($loggedFecha === '' && $loggedAt !== '') {
            $loggedFecha = substr($loggedAt, 0, 10);
        }
        if ($loggedHora === '' && $loggedAt !== '') {
            $loggedHora = substr($loggedAt, 11, 5);
        }

        $clienteNombre = trim((string) ($row['cliente_nombre'] ?? ''));
        if ($clienteNombre === '' && !empty($row['pedido_cliente_ref'])) {
            $clienteNombre = (string) $row['pedido_cliente_ref'];
        }

        $pedidoCodigo = trim((string) ($row['pedido_codigo'] ?? ''));
        if ($pedidoCodigo === '') {
            $pedidoCodigo = 'Pedido #' . (int) ($row['so_id'] ?? 0);
        }

        $responsable = trim((string) ($row['responsable_label'] ?? ''));
        if ($responsable === '') {
            $responsable = trim((string) ($row['responsable_nombre'] ?? ''));
        }

        return [
            'id' => (int) ($row['id'] ?? 0),
            'so_id' => (int) ($row['so_id'] ?? 0),
            'pre_id' => isset($row['pre_id']) ? (int) $row['pre_id'] : null,
            'motivo_id' => (int) ($row['motivo_id'] ?? 0),
            'motivo_nombre' => trim((string) ($row['motivo_nombre'] ?? 'Sin motivo')) ?: 'Sin motivo',
            'responsable_user_id' => isset($row['responsable_user_id']) ? (int) $row['responsable_user_id'] : null,
            'responsable_label' => $responsable !== '' ? $responsable : 'Sin responsable',
            'observacion' => trim((string) ($row['observacion'] ?? '')),
            'logged_at' => $loggedAt,
            'logged_fecha' => $loggedFecha,
            'logged_hora' => $loggedHora,
            'created_at' => (string) ($row['created_at'] ?? ''),
            'pedido_codigo' => $pedidoCodigo,
            'pedido_fecha' => $row['pedido_fecha'] ?? null,
            'cliente_id' => isset($row['pedido_cliente_id']) ? (int) $row['pedido_cliente_id'] : null,
            'cliente_nombre' => $clienteNombre !== '' ? $clienteNombre : 'Sin cliente',
            'pre_codigo' => trim((string) ($row['pre_codigo'] ?? '')),
        ];
    }
}

if (!function_exists('errores_picking_normalize_filters')) {
    function errores_picking_normalize_filters(array $input): array
    {
        $defaultDesde = date('Y-m-d', strtotime('-14 days'));
        $defaultHasta = date('Y-m-d');

        $desde = errores_picking_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = errores_picking_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);
        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'motivo_id' => errores_picking_sanitize_numeric($input['motivo_id'] ?? null),
            'cliente_id' => errores_picking_sanitize_numeric($input['cliente_id'] ?? null),
            'codigo' => errores_picking_sanitize_string($input['codigo'] ?? '', 80),
            'responsable' => errores_picking_sanitize_string($input['responsable'] ?? '', 80),
        ];
    }
}

if (!function_exists('errores_picking_sanitize_date')) {
    function errores_picking_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('errores_picking_sanitize_numeric')) {
    function errores_picking_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('errores_picking_sanitize_string')) {
    function errores_picking_sanitize_string(?string $value, int $maxLength): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        if (strlen($value) > $maxLength) {
            $value = substr($value, 0, $maxLength);
        }
        return $value;
    }
}

if (!function_exists('errores_picking_detect_default_motivo')) {
    function errores_picking_detect_default_motivo(array $motivos): ?int
    {
        if (!$motivos) {
            return null;
        }
        foreach ($motivos as $motivo) {
            $label = errores_picking_normalize_label((string) ($motivo['nombre'] ?? ''));
            if ($label === errores_picking_normalize_label('Preparación picking')) {
                return (int) $motivo['id'];
            }
        }
        return null;
    }
}

if (!function_exists('errores_picking_normalize_label')) {
    function errores_picking_normalize_label(string $value): string
    {
        $value = mb_strtolower(trim($value));
        $map = ['á' => 'a', 'é' => 'e', 'í' => 'i', 'ó' => 'o', 'ú' => 'u', 'ñ' => 'n'];
        return strtr($value, $map);
    }
}

if (!function_exists('errores_picking_lookup_motivo_label')) {
    function errores_picking_lookup_motivo_label(PDO $pdo, int $motivoId): ?string
    {
        if ($motivoId <= 0) {
            return null;
        }
        if (!function_exists('so_prep_err_table_exists') || !so_prep_err_table_exists($pdo, 'para_motivos_errores')) {
            return null;
        }
        $labelCol = function_exists('so_prep_err_detect_column') ? so_prep_err_detect_column($pdo, 'para_motivos_errores', ['descripcion', 'nombre', 'detalle', 'motivo', 'titulo', 'label']) : null;
        $select = $labelCol ? "`{$labelCol}`" : "CONCAT('Motivo #', id)";
        $stmt = $pdo->prepare("SELECT {$select} FROM para_motivos_errores WHERE id = ? LIMIT 1");
        $stmt->execute([$motivoId]);
        $value = (string) ($stmt->fetchColumn() ?: '');
        return $value !== '' ? $value : null;
    }
}

if (!function_exists('errores_picking_detect_top_motivo')) {
    function errores_picking_detect_top_motivo(array $rows): array
    {
        if (!$rows) {
            return ['label' => null, 'rows' => 0];
        }
        $totals = [];
        foreach ($rows as $row) {
            $key = (string) ($row['motivo_id'] ?? '');
            if (!isset($totals[$key])) {
                $totals[$key] = [
                    'label' => $row['motivo_nombre'] ?? 'Sin motivo',
                    'rows' => 0,
                ];
            }
            $totals[$key]['rows']++;
        }
        usort($totals, static function (array $a, array $b): int {
            $cmp = ($b['rows'] ?? 0) <=> ($a['rows'] ?? 0);
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) ($a['label'] ?? ''), (string) ($b['label'] ?? ''));
        });
        return $totals[0];
    }
}
