<?php

declare(strict_types=1);


if (!function_exists('facturacion_fetch_combos')) {
    function facturacion_fetch_combos(PDO $pdo): array
    {
        $depositos = [];
        if (facturacion_table_exists($pdo, 'wh_deposito')) {
            $select = ['id'];
            if (facturacion_column_exists($pdo, 'wh_deposito', 'code')) {
                $select[] = 'code';
            }
            if (facturacion_column_exists($pdo, 'wh_deposito', 'nombre')) {
                $select[] = 'nombre';
            }
            if (facturacion_column_exists($pdo, 'wh_deposito', 'activo')) {
                $select[] = 'activo';
            }
            if (facturacion_column_exists($pdo, 'wh_deposito', 'deleted_at')) {
                $select[] = 'deleted_at';
            }

            $sql = 'SELECT ' . implode(', ', $select) . ' FROM wh_deposito';
            $where = [];
            if (facturacion_column_exists($pdo, 'wh_deposito', 'activo')) {
                $where[] = '(activo = 1 OR activo IS NULL)';
            }
            if (facturacion_column_exists($pdo, 'wh_deposito', 'deleted_at')) {
                $where[] = 'deleted_at IS NULL';
            }
            if ($where) {
                $sql .= ' WHERE ' . implode(' AND ', $where);
            }
            $orderByDeposito = facturacion_column_exists($pdo, 'wh_deposito', 'nombre') ? 'nombre' : (facturacion_column_exists($pdo, 'wh_deposito', 'code') ? 'code' : 'id');
            $sql .= ' ORDER BY ' . $orderByDeposito . ' ASC LIMIT 200';
            foreach ($pdo->query($sql) as $row) {
                $id = isset($row['id']) ? (int) $row['id'] : 0;
                if ($id <= 0) {
                    continue;
                }
                $code = trim((string) ($row['code'] ?? ''));
                $nombre = trim((string) ($row['nombre'] ?? ''));
                $label = $code !== '' ? ($code . ($nombre !== '' ? ' · ' . $nombre : '')) : ($nombre !== '' ? $nombre : ('Depósito #' . $id));
                $depositos[] = ['id' => $id, 'label' => $label];
            }
        }

        $choferes = [];
        if (facturacion_table_exists($pdo, 'para_choferes')) {
            $select = ['id'];
            if (facturacion_column_exists($pdo, 'para_choferes', 'nombre')) {
                $select[] = 'nombre';
            }
            if (facturacion_column_exists($pdo, 'para_choferes', 'activo')) {
                $select[] = 'activo';
            }

            $sql = 'SELECT ' . implode(', ', $select) . ' FROM para_choferes';
            $whereClauses = [];
            if (facturacion_column_exists($pdo, 'para_choferes', 'activo')) {
                $whereClauses[] = 'activo = 1';
            }
            if ($whereClauses) {
                $sql .= ' WHERE ' . implode(' AND ', $whereClauses);
            }
            $orderByChofer = facturacion_column_exists($pdo, 'para_choferes', 'nombre') ? 'nombre' : 'id';
            $sql .= ' ORDER BY ' . $orderByChofer . ' ASC LIMIT 200';
            foreach ($pdo->query($sql) as $row) {
                $id = isset($row['id']) ? (int) $row['id'] : 0;
                if ($id <= 0) {
                    continue;
                }
                $nombre = trim((string) ($row['nombre'] ?? ''));
                $choferes[] = ['id' => $id, 'label' => $nombre !== '' ? $nombre : ('Chofer #' . $id)];
            }
        }

        $clientes = [];
        if (facturacion_table_exists($pdo, 'para_clientes')) {
            $select = ['id'];
            if (facturacion_column_exists($pdo, 'para_clientes', 'razon_social')) {
                $select[] = 'razon_social';
            }
            if (facturacion_column_exists($pdo, 'para_clientes', 'activo')) {
                $select[] = 'activo';
            }

            $sql = 'SELECT ' . implode(', ', $select) . ' FROM para_clientes';
            $whereClauses = [];
            if (facturacion_column_exists($pdo, 'para_clientes', 'activo')) {
                $whereClauses[] = 'activo = 1';
            }
            if ($whereClauses) {
                $sql .= ' WHERE ' . implode(' AND ', $whereClauses);
            }
            $orderByCliente = facturacion_column_exists($pdo, 'para_clientes', 'razon_social') ? 'razon_social' : 'id';
            $sql .= ' ORDER BY ' . $orderByCliente . ' ASC LIMIT 400';
            foreach ($pdo->query($sql) as $row) {
                $id = isset($row['id']) ? (int) $row['id'] : 0;
                if ($id <= 0) {
                    continue;
                }
                $nombre = trim((string) ($row['razon_social'] ?? ''));
                $clientes[] = ['id' => $id, 'label' => $nombre !== '' ? $nombre : ('Cliente #' . $id)];
            }
        }

        $condiciones = [];
        if (facturacion_table_exists($pdo, 'so_embarque_rendiciones') && facturacion_column_exists($pdo, 'so_embarque_rendiciones', 'condicion')) {
            $stmt = $pdo->query('SELECT DISTINCT condicion FROM so_embarque_rendiciones WHERE condicion IS NOT NULL AND TRIM(condicion) <> "" ORDER BY condicion ASC LIMIT 100');
            while (($row = $stmt->fetch(PDO::FETCH_ASSOC)) !== false) {
                $cond = trim((string) ($row['condicion'] ?? ''));
                if ($cond === '') {
                    continue;
                }
                $condiciones[] = $cond;
            }
        }

        return [
            'depositos' => $depositos,
            'choferes' => $choferes,
            'clientes' => $clientes,
            'condiciones' => $condiciones,
        ];
    }
}

if (!function_exists('facturacion_fetch_data')) {
    function facturacion_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = facturacion_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        $docs = facturacion_table_exists($pdo, 'so_parada_doc') ? facturacion_collect_docs($pdo, $filters, $limit) : [];
        $rends = facturacion_table_exists($pdo, 'so_embarque_rendiciones') ? facturacion_collect_rendiciones($pdo, $filters, $limit) : [];

        $entries = facturacion_merge_entries($docs, $rends);
        $filtered = facturacion_filter_entries($entries, $filters);

        usort($filtered, static function (array $a, array $b): int {
            $refA = (string) ($a['ref_fecha'] ?? '');
            $refB = (string) ($b['ref_fecha'] ?? '');
            if ($refA === $refB) {
                return strcmp((string) ($b['factura'] ?? ''), (string) ($a['factura'] ?? ''));
            }
            return strcmp($refB, $refA);
        });

        $truncated = false;
        if (count($filtered) > $limit) {
            $filtered = array_slice($filtered, 0, $limit);
            $truncated = true;
        }

        $summary = facturacion_build_summary($filtered, $filters);
        $aggregates = facturacion_build_aggregates($filtered);

        return [
            'filters' => $filters,
            'rows' => $filtered,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('facturacion_collect_docs')) {
    function facturacion_collect_docs(PDO $pdo, array $filters, int $limit): array
    {
        $limitDocs = max(50, min($limit * 8, 4000));

        $conditions = [
            'doc.doc_numero IS NOT NULL',
            "TRIM(doc.doc_numero) <> ''",
                '(doc.doc_tipo IS NULL OR UPPER(doc.doc_tipo) LIKE \'FACT%\')',
            'DATE(COALESCE(doc.created_at, par.hora_llegada, e.creado_at)) BETWEEN :fecha_doc_desde AND :fecha_doc_hasta',
        ];

        $params = [
            ':fecha_doc_desde' => $filters['fecha_desde'],
            ':fecha_doc_hasta' => $filters['fecha_hasta'],
        ];

        if ($filters['deposito_id'] !== '') {
            $conditions[] = 'e.deposito_id = :doc_deposito';
            $params[':doc_deposito'] = (int) $filters['deposito_id'];
        }

        if ($filters['chofer_id'] !== '') {
            $conditions[] = 'e.chofer_id = :doc_chofer';
            $params[':doc_chofer'] = (int) $filters['chofer_id'];
        }

        $whereSql = 'WHERE ' . implode(' AND ', $conditions);

        $sql = "
            SELECT
                e.id AS embarque_id,
                e.codigo AS embarque_codigo,
                e.creado_at AS embarque_creado_at,
                e.salida_at AS embarque_salida_at,
                e.estado_id AS embarque_estado_id,
                est.nombre AS embarque_estado_nombre,
                e.deposito_id,
                dep.code AS deposito_code,
                dep.nombre AS deposito_nombre,
                e.chofer_id,
                cho.nombre AS chofer_nombre,
                e.movil_id,
                mov.chapa AS movil_chapa,
                dest.id AS destinatario_id,
                dest.razon_social AS destinatario_nombre,
                cli.id AS cliente_id,
                cli.razon_social AS cliente_nombre,
                doc.doc_numero AS factura_doc_numero,
                doc.doc_tipo AS factura_doc_tipo,
                DATE(COALESCE(doc.created_at, par.hora_llegada, e.creado_at)) AS ref_fecha,
                doc.created_at AS doc_created_at
            FROM so_parada_doc doc
            JOIN so_embarque_parada par ON par.id = doc.parada_id
            JOIN so_embarque e ON e.id = par.embarque_id
            LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
            LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
            LEFT JOIN para_choferes cho ON cho.id = e.chofer_id
            LEFT JOIN para_moviles mov ON mov.id = e.movil_id
            LEFT JOIN para_destinatarios dest ON dest.id = par.destinatario_id
            LEFT JOIN para_clientes cli ON cli.id = dest.cliente_id
            $whereSql
            ORDER BY ref_fecha DESC, doc.doc_numero DESC
            LIMIT :limit_docs
        ";

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_docs', $limitDocs, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}

if (!function_exists('facturacion_collect_rendiciones')) {
    function facturacion_collect_rendiciones(PDO $pdo, array $filters, int $limit): array
    {
        $limitRows = max(50, min($limit * 4, 4000));

        $conditions = [
            'r.factura IS NOT NULL',
            "TRIM(r.factura) <> ''",
            'DATE(COALESCE(r.fecha_factura, r.fecha_rendicion, e.creado_at)) BETWEEN :rend_fecha_desde AND :rend_fecha_hasta',
        ];

        $params = [
            ':rend_fecha_desde' => $filters['fecha_desde'],
            ':rend_fecha_hasta' => $filters['fecha_hasta'],
        ];

        if ($filters['deposito_id'] !== '') {
            $conditions[] = 'e.deposito_id = :rend_deposito';
            $params[':rend_deposito'] = (int) $filters['deposito_id'];
        }

        if ($filters['chofer_id'] !== '') {
            $conditions[] = 'e.chofer_id = :rend_chofer';
            $params[':rend_chofer'] = (int) $filters['chofer_id'];
        }

        if ($filters['condicion'] !== '') {
            $conditions[] = 'UPPER(r.condicion) = :rend_condicion';
            $params[':rend_condicion'] = strtoupper((string) $filters['condicion']);
        }

        $whereSql = 'WHERE ' . implode(' AND ', $conditions);

        $sql = "
            SELECT
                r.embarque_id,
                e.codigo AS embarque_codigo,
                e.creado_at AS embarque_creado_at,
                e.salida_at AS embarque_salida_at,
                e.estado_id AS embarque_estado_id,
                est.nombre AS embarque_estado_nombre,
                e.deposito_id,
                dep.code AS deposito_code,
                dep.nombre AS deposito_nombre,
                e.chofer_id,
                cho.nombre AS chofer_nombre,
                e.movil_id,
                mov.chapa AS movil_chapa,
                r.factura AS rend_factura,
                r.condicion AS rend_condicion,
                r.fecha_factura AS rend_fecha_factura,
                r.monto AS rend_monto,
                r.fecha_rendicion AS rend_fecha_rendicion,
                DATE(COALESCE(r.fecha_factura, r.fecha_rendicion, e.creado_at)) AS ref_fecha
            FROM so_embarque_rendiciones r
            JOIN so_embarque e ON e.id = r.embarque_id
            LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
            LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
            LEFT JOIN para_choferes cho ON cho.id = e.chofer_id
            LEFT JOIN para_moviles mov ON mov.id = e.movil_id
            $whereSql
            ORDER BY ref_fecha DESC, r.factura DESC
            LIMIT :limit_rend
        ";

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_rend', $limitRows, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }
}

if (!function_exists('facturacion_merge_entries')) {
    function facturacion_merge_entries(array $docs, array $rends): array
    {
        $entries = [];

        foreach ($docs as $row) {
            $facturaKey = facturacion_normalize_factura_key($row['factura_doc_numero'] ?? null);
            if ($facturaKey === '') {
                continue;
            }
            $embId = isset($row['embarque_id']) ? (int) $row['embarque_id'] : 0;
            if ($embId <= 0) {
                continue;
            }
            $mapKey = $embId . '|' . $facturaKey;
            if (!isset($entries[$mapKey])) {
                $entries[$mapKey] = facturacion_seed_entry($row, $facturaKey, 'doc');
            }
            facturacion_apply_doc_row($entries[$mapKey], $row);
        }

        foreach ($rends as $row) {
            $facturaKey = facturacion_normalize_factura_key($row['rend_factura'] ?? null);
            if ($facturaKey === '') {
                continue;
            }
            $embId = isset($row['embarque_id']) ? (int) $row['embarque_id'] : 0;
            if ($embId <= 0) {
                continue;
            }
            $mapKey = $embId . '|' . $facturaKey;
            if (!isset($entries[$mapKey])) {
                $entries[$mapKey] = facturacion_seed_entry($row, $facturaKey, 'rend');
            }
            facturacion_apply_rend_row($entries[$mapKey], $row);
        }

        foreach ($entries as &$entry) {
            facturacion_finalize_entry($entry);
        }
        unset($entry);

        return array_values($entries);
    }
}

if (!function_exists('facturacion_seed_entry')) {
    function facturacion_seed_entry(array $row, string $facturaKey, string $source): array
    {
        $facturaNumero = $source === 'doc'
            ? trim((string) ($row['factura_doc_numero'] ?? ''))
            : trim((string) ($row['rend_factura'] ?? ''));
        if ($facturaNumero === '') {
            $facturaNumero = $facturaKey;
        }

        $depositoId = isset($row['deposito_id']) ? (int) $row['deposito_id'] : 0;
        $depositoCode = trim((string) ($row['deposito_code'] ?? ''));
        $depositoNombre = trim((string) ($row['deposito_nombre'] ?? ''));
        $depositoLabel = $depositoNombre !== ''
            ? ($depositoCode !== '' ? ($depositoCode . ' · ' . $depositoNombre) : $depositoNombre)
            : ($depositoCode !== '' ? $depositoCode : 'Sin depósito');

        $choferNombre = trim((string) ($row['chofer_nombre'] ?? ''));
        $movilLabel = trim((string) ($row['movil_chapa'] ?? ''));

        $refFecha = trim((string) ($row['ref_fecha'] ?? ''));
        if ($refFecha !== '' && strlen($refFecha) > 10) {
            $refFecha = substr($refFecha, 0, 10);
        }

        return [
            'factura' => $facturaNumero,
            'factura_key' => $facturaKey,
            'embarque_id' => isset($row['embarque_id']) ? (int) $row['embarque_id'] : 0,
            'embarque_codigo' => trim((string) ($row['embarque_codigo'] ?? '')),
            'embarque_estado' => trim((string) ($row['embarque_estado_nombre'] ?? '')),
            'salida_at' => $row['embarque_salida_at'] ?? null,
            'deposito_id' => $depositoId > 0 ? $depositoId : null,
            'deposito_code' => $depositoCode !== '' ? $depositoCode : null,
            'deposito_nombre' => $depositoNombre !== '' ? $depositoNombre : null,
            'deposito_label' => $depositoLabel,
            'chofer_id' => isset($row['chofer_id']) ? (int) $row['chofer_id'] : null,
            'chofer_nombre' => $choferNombre !== '' ? $choferNombre : 'Sin chofer',
            'movil_id' => isset($row['movil_id']) ? (int) $row['movil_id'] : null,
            'movil_label' => $movilLabel !== '' ? $movilLabel : 'Sin móvil',
            'clientes' => [],
            'destinatarios' => [],
            'clientes_label' => 'Sin cliente',
            'destinatarios_label' => 'Sin destinatario',
            'fecha_factura' => null,
            'fecha_rendicion' => null,
            'fecha_documento' => null,
            'ref_fecha' => $refFecha,
            'monto' => null,
            'condicion' => null,
            'estado' => 'PENDIENTE',
            'dias_a_rendir' => null,
            'dias_pendientes' => null,
            'importe_pendiente' => null,
            'importe_rendido' => 0.0,
            'doc_fuentes' => [],
            '_cliente_ids' => [],
            '_dest_ids' => [],
            '_doc_count' => 0,
            '_creado_at' => $row['embarque_creado_at'] ?? null,
        ];
    }
}

if (!function_exists('facturacion_apply_doc_row')) {
    function facturacion_apply_doc_row(array &$entry, array $row): void
    {
        $facturaNumero = trim((string) ($row['factura_doc_numero'] ?? ''));
        if ($facturaNumero !== '' && ($entry['factura'] === '' || $entry['factura'] === $entry['factura_key'])) {
            $entry['factura'] = $facturaNumero;
        }

        $refFecha = trim((string) ($row['ref_fecha'] ?? ''));
        if ($refFecha !== '' && strlen($refFecha) > 10) {
            $refFecha = substr($refFecha, 0, 10);
        }
        if ($refFecha !== '') {
            if ($entry['ref_fecha'] === '' || $refFecha < $entry['ref_fecha']) {
                $entry['ref_fecha'] = $refFecha;
            }
            if ($entry['fecha_documento'] === null || $refFecha < $entry['fecha_documento']) {
                $entry['fecha_documento'] = $refFecha;
            }
        }

        $destId = isset($row['destinatario_id']) ? (int) $row['destinatario_id'] : 0;
        if ($destId > 0) {
            $destNombre = trim((string) ($row['destinatario_nombre'] ?? ''));
            $entry['_dest_ids'][$destId] = $destNombre !== '' ? $destNombre : ('Destinatario #' . $destId);
        }

        $clienteId = isset($row['cliente_id']) ? (int) $row['cliente_id'] : 0;
        if ($clienteId > 0) {
            $clienteNombre = trim((string) ($row['cliente_nombre'] ?? ''));
            $entry['_cliente_ids'][$clienteId] = $clienteNombre !== '' ? $clienteNombre : ('Cliente #' . $clienteId);
        }

        $docTipo = trim((string) ($row['factura_doc_tipo'] ?? 'FACTURA'));
        if ($docTipo !== '') {
            $entry['doc_fuentes'][$docTipo] = true;
        }

        $entry['_doc_count']++;
    }
}

if (!function_exists('facturacion_apply_rend_row')) {
    function facturacion_apply_rend_row(array &$entry, array $row): void
    {
        if (isset($row['rend_condicion'])) {
            $cond = trim((string) $row['rend_condicion']);
            if ($cond !== '') {
                $entry['condicion'] = $cond;
            }
        }

        if (isset($row['rend_monto']) && $row['rend_monto'] !== null) {
            $monto = (float) $row['rend_monto'];
            $entry['monto'] = ($entry['monto'] ?? 0.0) + $monto;
        }

        $fechaFact = facturacion_normalize_date_str($row['rend_fecha_factura'] ?? null);
        if ($fechaFact !== null) {
            if ($entry['fecha_factura'] === null || $fechaFact < $entry['fecha_factura']) {
                $entry['fecha_factura'] = $fechaFact;
            }
            if ($entry['ref_fecha'] === '' || $fechaFact < $entry['ref_fecha']) {
                $entry['ref_fecha'] = $fechaFact;
            }
        }

        $fechaRend = facturacion_normalize_datetime_str($row['rend_fecha_rendicion'] ?? null);
        if ($fechaRend !== null) {
            if ($entry['fecha_rendicion'] === null || $fechaRend > $entry['fecha_rendicion']) {
                $entry['fecha_rendicion'] = $fechaRend;
            }
            if ($entry['ref_fecha'] === '' && $fechaRend !== null) {
                $entry['ref_fecha'] = substr($fechaRend, 0, 10);
            }
        }
    }
}

if (!function_exists('facturacion_finalize_entry')) {
    function facturacion_finalize_entry(array &$entry): void
    {
        $clientes = [];
        foreach ($entry['_cliente_ids'] as $id => $label) {
            $clientes[] = ['id' => (int) $id, 'label' => $label];
        }
        usort($clientes, static function (array $a, array $b): int {
            return strcmp($a['label'], $b['label']);
        });
        $entry['clientes'] = $clientes;
        $entry['clientes_label'] = $clientes ? implode(' · ', array_map(static function (array $c): string {
            return $c['label'];
        }, $clientes)) : 'Sin cliente';
        unset($entry['_cliente_ids']);

        $destinatarios = [];
        foreach ($entry['_dest_ids'] as $id => $label) {
            $destinatarios[] = ['id' => (int) $id, 'label' => $label];
        }
        usort($destinatarios, static function (array $a, array $b): int {
            return strcmp($a['label'], $b['label']);
        });
        $entry['destinatarios'] = $destinatarios;
        $entry['destinatarios_label'] = $destinatarios ? implode(' · ', array_map(static function (array $d): string {
            return $d['label'];
        }, $destinatarios)) : 'Sin destinatario';
        unset($entry['_dest_ids']);

        if ($entry['monto'] !== null) {
            $entry['monto'] = (float) $entry['monto'];
        }

        if ($entry['fecha_factura'] === null && $entry['fecha_documento'] !== null) {
            $entry['fecha_factura'] = $entry['fecha_documento'];
        }

        if ($entry['ref_fecha'] === '') {
            $candidates = [];
            if ($entry['fecha_factura'] !== null) {
                $candidates[] = $entry['fecha_factura'];
            }
            if ($entry['fecha_documento'] !== null) {
                $candidates[] = $entry['fecha_documento'];
            }
            if ($entry['fecha_rendicion'] !== null) {
                $candidates[] = substr($entry['fecha_rendicion'], 0, 10);
            }
            if ($entry['_creado_at']) {
                $candidates[] = substr((string) $entry['_creado_at'], 0, 10);
            }
            foreach ($candidates as $candidate) {
                if ($candidate !== null && $candidate !== '') {
                    $entry['ref_fecha'] = $candidate;
                    break;
                }
            }
        }

        if ($entry['fecha_rendicion'] !== null) {
            $entry['estado'] = 'RENDIDA';
        }

        $refDate = $entry['fecha_factura'] ?? $entry['fecha_documento'] ?? $entry['ref_fecha'];
        if ($refDate !== null && $refDate !== '') {
            if ($entry['fecha_rendicion'] !== null) {
                $entry['dias_a_rendir'] = facturacion_diff_days($refDate, substr($entry['fecha_rendicion'], 0, 10));
            } else {
                $entry['dias_pendientes'] = facturacion_diff_days($refDate, date('Y-m-d'));
            }
        }

        if ($entry['estado'] === 'RENDIDA') {
            $entry['importe_rendido'] = $entry['monto'] !== null ? (float) $entry['monto'] : 0.0;
            $entry['importe_pendiente'] = 0.0;
        } else {
            $entry['importe_rendido'] = 0.0;
            $entry['importe_pendiente'] = $entry['monto'] !== null ? (float) $entry['monto'] : null;
        }

        $entry['doc_fuentes'] = array_keys($entry['doc_fuentes']);
        $entry['doc_count'] = $entry['_doc_count'];
        unset($entry['_doc_count'], $entry['_creado_at']);
    }
}

if (!function_exists('facturacion_filter_entries')) {
    function facturacion_filter_entries(array $entries, array $filters): array
    {
        $desde = $filters['fecha_desde'];
        $hasta = $filters['fecha_hasta'];
        $deposito = $filters['deposito_id'] !== '' ? (int) $filters['deposito_id'] : null;
        $chofer = $filters['chofer_id'] !== '' ? (int) $filters['chofer_id'] : null;
        $cliente = $filters['cliente_id'] !== '' ? (int) $filters['cliente_id'] : null;
        $condicion = $filters['condicion'] !== '' ? strtoupper($filters['condicion']) : '';
        $soloPendientes = (bool) $filters['solo_pendientes'];
        $soloRendidas = (bool) $filters['solo_rendidas'];

        $result = [];
        foreach ($entries as $entry) {
            $ref = $entry['ref_fecha'] ?? $entry['fecha_factura'] ?? $entry['fecha_documento'] ?? null;
            if ($ref === null || $ref === '') {
                $ref = $desde;
            }
            if ($ref < $desde || $ref > $hasta) {
                continue;
            }

            if ($deposito !== null && (int) ($entry['deposito_id'] ?? 0) !== $deposito) {
                continue;
            }

            if ($chofer !== null && (int) ($entry['chofer_id'] ?? 0) !== $chofer) {
                continue;
            }

            if ($condicion !== '' && strtoupper((string) ($entry['condicion'] ?? '')) !== $condicion) {
                continue;
            }

            if ($soloPendientes && $entry['estado'] !== 'PENDIENTE') {
                continue;
            }

            if ($soloRendidas && $entry['estado'] !== 'RENDIDA') {
                continue;
            }

            if ($cliente !== null) {
                $belongs = false;
                foreach ($entry['clientes'] as $cli) {
                    if ((int) $cli['id'] === $cliente) {
                        $belongs = true;
                        break;
                    }
                }
                if (!$belongs) {
                    continue;
                }
            }

            $result[] = $entry;
        }

        return $result;
    }
}

if (!function_exists('facturacion_build_summary')) {
    function facturacion_build_summary(array $rows, array $filters): array
    {
        $total = count($rows);
        $rendidas = 0;
        $pendientes = 0;
        $montoTotal = 0.0;
        $montoPendiente = 0.0;
        $diasRendidos = [];
        $diasPendientes = [];

        foreach ($rows as $row) {
            $monto = (float) ($row['monto'] ?? 0.0);
            $montoTotal += $monto;
            if ($row['estado'] === 'RENDIDA') {
                $rendidas++;
                if ($row['dias_a_rendir'] !== null) {
                    $diasRendidos[] = (float) $row['dias_a_rendir'];
                }
            } else {
                $pendientes++;
                $pend = (float) ($row['importe_pendiente'] ?? 0.0);
                $montoPendiente += $pend;
                if ($row['dias_pendientes'] !== null) {
                    $diasPendientes[] = (float) $row['dias_pendientes'];
                }
            }
        }

        $promedioRend = $diasRendidos ? round(array_sum($diasRendidos) / max(count($diasRendidos), 1), 1) : null;
        $maxPend = $diasPendientes ? max($diasPendientes) : null;
        $pendMayor7 = 0;
        $pendMayor15 = 0;
        foreach ($diasPendientes as $dias) {
            if ($dias >= 7) {
                $pendMayor7++;
            }
            if ($dias >= 15) {
                $pendMayor15++;
            }
        }

        return [
            'total_facturas' => $total,
            'total_rendidas' => $rendidas,
            'total_pendientes' => $pendientes,
            'importe_total' => round($montoTotal, 2),
            'importe_rendido' => round($montoTotal - $montoPendiente, 2),
            'importe_pendiente' => round($montoPendiente, 2),
            'avg_dias_rendicion' => $promedioRend,
            'max_dias_pendiente' => $maxPend,
            'pendientes_mayor_7' => $pendMayor7,
            'pendientes_mayor_15' => $pendMayor15,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('facturacion_build_aggregates')) {
    function facturacion_build_aggregates(array $rows): array
    {
        $clientes = [];
        $depositos = [];
        $choferes = [];
        $condiciones = [];

        foreach ($rows as $row) {
            $monto = (float) ($row['monto'] ?? 0.0);
            $pend = ($row['estado'] === 'PENDIENTE') ? (float) ($row['importe_pendiente'] ?? 0.0) : 0.0;

            if (!empty($row['clientes'])) {
                foreach ($row['clientes'] as $cli) {
                    $key = (string) ($cli['id'] ?? 0);
                    if (!isset($clientes[$key])) {
                        $clientes[$key] = [
                            'id' => (int) ($cli['id'] ?? 0),
                            'label' => $cli['label'] ?? 'Sin cliente',
                            'facturas' => 0,
                            'rendidas' => 0,
                            'pendientes' => 0,
                            'monto_total' => 0.0,
                            'monto_pendiente' => 0.0,
                        ];
                    }
                    $clientes[$key]['facturas']++;
                    if ($row['estado'] === 'RENDIDA') {
                        $clientes[$key]['rendidas']++;
                    } else {
                        $clientes[$key]['pendientes']++;
                        $clientes[$key]['monto_pendiente'] += $pend;
                    }
                    $clientes[$key]['monto_total'] += $monto;
                }
            } else {
                $key = '0';
                if (!isset($clientes[$key])) {
                    $clientes[$key] = [
                        'id' => 0,
                        'label' => 'Sin cliente',
                        'facturas' => 0,
                        'rendidas' => 0,
                        'pendientes' => 0,
                        'monto_total' => 0.0,
                        'monto_pendiente' => 0.0,
                    ];
                }
                $clientes[$key]['facturas']++;
                if ($row['estado'] === 'RENDIDA') {
                    $clientes[$key]['rendidas']++;
                } else {
                    $clientes[$key]['pendientes']++;
                    $clientes[$key]['monto_pendiente'] += $pend;
                }
                $clientes[$key]['monto_total'] += $monto;
            }

            $depKey = (string) ($row['deposito_id'] ?? 0);
            if (!isset($depositos[$depKey])) {
                $depositos[$depKey] = [
                    'id' => (int) ($row['deposito_id'] ?? 0),
                    'label' => $row['deposito_label'] ?? 'Sin depósito',
                    'facturas' => 0,
                    'pendientes' => 0,
                    'monto_total' => 0.0,
                ];
            }
            $depositos[$depKey]['facturas']++;
            if ($row['estado'] === 'PENDIENTE') {
                $depositos[$depKey]['pendientes']++;
            }
            $depositos[$depKey]['monto_total'] += $monto;

            $choferKey = (string) ($row['chofer_id'] ?? 0);
            if (!isset($choferes[$choferKey])) {
                $choferes[$choferKey] = [
                    'id' => (int) ($row['chofer_id'] ?? 0),
                    'label' => $row['chofer_nombre'] ?? 'Sin chofer',
                    'facturas' => 0,
                    'pendientes' => 0,
                ];
            }
            $choferes[$choferKey]['facturas']++;
            if ($row['estado'] === 'PENDIENTE') {
                $choferes[$choferKey]['pendientes']++;
            }

            $condKey = strtoupper((string) ($row['condicion'] ?? 'SIN CONDICION'));
            if (!isset($condiciones[$condKey])) {
                $condiciones[$condKey] = [
                    'condicion' => $row['condicion'] ?? 'Sin condición',
                    'facturas' => 0,
                    'pendientes' => 0,
                ];
            }
            $condiciones[$condKey]['facturas']++;
            if ($row['estado'] === 'PENDIENTE') {
                $condiciones[$condKey]['pendientes']++;
            }
        }

        $clienteList = array_values($clientes);
        usort($clienteList, static function (array $a, array $b): int {
            $cmp = ($b['monto_total'] ?? 0) <=> ($a['monto_total'] ?? 0);
            if ($cmp !== 0) {
                return $cmp;
            }
            return $b['facturas'] <=> $a['facturas'];
        });
        $clienteList = array_slice($clienteList, 0, 10);

        $depositoList = array_values($depositos);
        usort($depositoList, static function (array $a, array $b): int {
            $cmp = $b['facturas'] <=> $a['facturas'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp($a['label'], $b['label']);
        });

        $choferList = array_values($choferes);
        usort($choferList, static function (array $a, array $b): int {
            $cmp = $b['facturas'] <=> $a['facturas'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp($a['label'], $b['label']);
        });

        $condList = array_values($condiciones);
        usort($condList, static function (array $a, array $b): int {
            return $b['facturas'] <=> $a['facturas'];
        });

        return [
            'clientes' => $clienteList,
            'depositos' => $depositoList,
            'choferes' => $choferList,
            'condiciones' => $condList,
        ];
    }
}

if (!function_exists('facturacion_normalize_filters')) {
    function facturacion_normalize_filters(array $input): array
    {
        $defaultHasta = date('Y-m-d');
        $defaultDesde = date('Y-m-d', strtotime('-29 days'));

        $desde = facturacion_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = facturacion_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);
        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'deposito_id' => facturacion_sanitize_numeric($input['deposito_id'] ?? null),
            'chofer_id' => facturacion_sanitize_numeric($input['chofer_id'] ?? null),
            'cliente_id' => facturacion_sanitize_numeric($input['cliente_id'] ?? null),
            'condicion' => facturacion_sanitize_string($input['condicion'] ?? ''),
            'solo_pendientes' => facturacion_sanitize_bool($input['solo_pendientes'] ?? null),
            'solo_rendidas' => facturacion_sanitize_bool($input['solo_rendidas'] ?? null),
        ];
    }
}

if (!function_exists('facturacion_sanitize_date')) {
    function facturacion_sanitize_date($value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('facturacion_sanitize_numeric')) {
    function facturacion_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $int = (int) $value;
        return $int > 0 ? (string) $int : '';
    }
}

if (!function_exists('facturacion_sanitize_string')) {
    function facturacion_sanitize_string($value): string
    {
        $value = trim((string) $value);
        return $value;
    }
}

if (!function_exists('facturacion_sanitize_bool')) {
    function facturacion_sanitize_bool($value): bool
    {
        if (is_bool($value)) {
            return $value;
        }
        if (is_numeric($value)) {
            return (int) $value === 1;
        }
        $value = strtolower(trim((string) $value));
        return in_array($value, ['1', 'true', 'yes', 'on', 'si'], true);
    }
}

if (!function_exists('facturacion_normalize_factura_key')) {
    function facturacion_normalize_factura_key(?string $value): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        return strtoupper($value);
    }
}

if (!function_exists('facturacion_normalize_date_str')) {
    function facturacion_normalize_date_str($value): ?string
    {
        if ($value === null) {
            return null;
        }
        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }
        if (strlen($value) >= 10) {
            $candidate = substr($value, 0, 10);
            if (preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $candidate)) {
                return $candidate;
            }
        }
        return null;
    }
}

if (!function_exists('facturacion_normalize_datetime_str')) {
    function facturacion_normalize_datetime_str($value): ?string
    {
        if ($value === null) {
            return null;
        }
        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }
        $value = str_replace('T', ' ', $value);
        if (strlen($value) === 16) {
            $value .= ':00';
        }
        if (preg_match('/^20\d{2}-[01]\d-[0-3]\d [0-2]\d:[0-5]\d:[0-5]\d$/', $value)) {
            return $value;
        }
        return null;
    }
}

if (!function_exists('facturacion_diff_days')) {
    function facturacion_diff_days(?string $from, ?string $to): ?float
    {
        if ($from === null || $to === null) {
            return null;
        }
        $fromTs = strtotime($from);
        $toTs = strtotime($to);
        if ($fromTs === false || $toTs === false) {
            return null;
        }
        return round(($toTs - $fromTs) / 86400, 1);
    }
}

if (!function_exists('facturacion_table_exists')) {
    function facturacion_table_exists(PDO $pdo, string $table): bool
    {
        static $cache = [];
        $key = spl_object_id($pdo) . ':' . strtolower($table);
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
        $stmt->execute([$table]);
        $cache[$key] = ((int) $stmt->fetchColumn()) > 0;
        return $cache[$key];
    }
}

if (!function_exists('facturacion_column_exists')) {
    function facturacion_column_exists(PDO $pdo, string $table, string $column): bool
    {
        static $cache = [];
        $key = spl_object_id($pdo) . ':' . strtolower($table) . ':' . strtolower($column);
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
        $stmt->execute([$table, $column]);
        $cache[$key] = ((int) $stmt->fetchColumn()) > 0;
        return $cache[$key];
    }
}
