<?php

declare(strict_types=1);

if (!function_exists('ingresos_fetch_combos')) {
    /**
     * Obtiene catálogos necesarios para los filtros del reporte de ingresos.
     */
    function ingresos_fetch_combos(PDO $pdo): array
    {
        $depositos = $pdo->query('SELECT id, code, nombre FROM wh_deposito ORDER BY nombre LIMIT 200')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $clientes = $pdo->query('SELECT id, razon_social FROM para_clientes WHERE deleted_at IS NULL ORDER BY razon_social LIMIT 400')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $moviles = $pdo->query('SELECT id, chapa FROM para_moviles WHERE activo = 1 ORDER BY chapa')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $estados = $pdo->query('SELECT DISTINCT estado FROM pl_packinglist ORDER BY estado')
            ->fetchAll(PDO::FETCH_ASSOC) ?: [];

        return [
            'depositos' => array_map(static function (array $row): array {
                $nombre = trim((string) ($row['nombre'] ?? ''));
                $code = trim((string) ($row['code'] ?? ''));
                if ($nombre === '' && $code !== '') {
                    $nombre = $code;
                }
                if ($nombre === '') {
                    $nombre = 'Sin depósito';
                }
                return [
                    'id' => (int) $row['id'],
                    'nombre' => $nombre,
                    'code' => $code !== '' ? $code : null,
                ];
            }, $depositos),
            'clientes' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'nombre' => trim((string) ($row['razon_social'] ?? '')) ?: 'Sin cliente',
                ];
            }, $clientes),
            'moviles' => array_map(static function (array $row): array {
                return [
                    'id' => (int) $row['id'],
                    'label' => trim((string) ($row['chapa'] ?? '')) ?: ('ID ' . (int) $row['id']),
                ];
            }, $moviles),
            'estados' => array_values(array_filter(array_map(static function (array $row): ?array {
                $code = trim((string) ($row['estado'] ?? ''));
                if ($code === '') {
                    return null;
                }
                return [
                    'code' => $code,
                    'nombre' => ingresos_format_estado_label($code),
                ];
            }, $estados))),
        ];
    }
}

if (!function_exists('ingresos_fetch_data')) {
    /**
     * Devuelve los datos del reporte de ingresos, aplicando filtros y límite.
     */
    function ingresos_fetch_data(PDO $pdo, array $rawFilters, int $limit = 1000): array
    {
        $filters = ingresos_normalize_filters($rawFilters);
        $limit = max(1, min($limit, 2000));

        $where = [];
        $params = [];

        $where[] = 'DATE(ing.fecha_ingreso) >= :fecha_desde';
        $where[] = 'DATE(ing.fecha_ingreso) <= :fecha_hasta';
        $params[':fecha_desde'] = $filters['fecha_desde'];
        $params[':fecha_hasta'] = $filters['fecha_hasta'];

        if ($filters['deposito_id'] !== '') {
            $where[] = 'ing.deposito_id = :deposito_id';
            $params[':deposito_id'] = (int) $filters['deposito_id'];
        }

        if ($filters['cliente_id'] !== '') {
            $where[] = 'pl.cliente_id = :cliente_id';
            $params[':cliente_id'] = (int) $filters['cliente_id'];
        }

        if ($filters['estado'] !== '') {
            $where[] = 'pl.estado = :estado_code';
            $params[':estado_code'] = $filters['estado'];
        }

        if ($filters['movil_id'] !== '') {
            $where[] = 'ing.movil_id = :movil_id';
            $params[':movil_id'] = (int) $filters['movil_id'];
        }

        if ($filters['codigo'] !== '') {
            $where[] = 'pl.codigo LIKE :codigo';
            $params[':codigo'] = '%' . $filters['codigo'] . '%';
        }

        if ($filters['doc_numero'] !== '') {
            $where[] = 'ing.doc_numero LIKE :doc_numero';
            $params[':doc_numero'] = '%' . $filters['doc_numero'] . '%';
        }

        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        $sql = '
            SELECT
                ing.id,
                ing.packinglist_id,
                ing.fecha_ingreso,
                ing.llegada_at,
                ing.descarga_inicio_at,
                ing.descarga_fin_at,
                ing.doc_tipo,
                ing.doc_numero,
                ing.doc_fecha,
                ing.observacion,
                ing.operarios_cant,
                ing.movil_id,
                ing.chofer_id,
                ing.deposito_id,
                ing.created_at,
                ing.updated_at,
                mov.chapa AS movil_chapa,
                cho.nombre AS chofer_nombre,
                dep.code AS deposito_code,
                dep.nombre AS deposito_nombre,
                pl.codigo AS packinglist_codigo,
                pl.fecha AS packinglist_fecha,
                pl.estado AS estado_code,
                pl.cliente_id,
                pl.cliente_ref,
                cli.razon_social AS cliente_nombre,
                agg.pallets,
                agg.items,
                agg.uv_cajas,
                agg.uc_unidades
            FROM pl_ingreso ing
            JOIN pl_packinglist pl ON pl.id = ing.packinglist_id
            LEFT JOIN para_clientes cli ON cli.id = pl.cliente_id
            LEFT JOIN wh_deposito dep ON dep.id = ing.deposito_id
            LEFT JOIN para_moviles mov ON mov.id = ing.movil_id
            LEFT JOIN para_choferes cho ON cho.id = ing.chofer_id
            LEFT JOIN (
                SELECT
                    packinglist_id,
                    COUNT(*) AS items,
                    COUNT(DISTINCT pallet_id) AS pallets,
                    SUM(uv_cajas) AS uv_cajas,
                    SUM(uc_unidades) AS uc_unidades
                FROM pl_rcv_link
                GROUP BY packinglist_id
            ) agg ON agg.packinglist_id = ing.packinglist_id
            ' . $whereSql . '
            ORDER BY ing.fecha_ingreso DESC, ing.id DESC
            LIMIT :limit_plus
        ';

        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit_plus', $limit + 1, PDO::PARAM_INT);
        $stmt->execute();

        $rawRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $truncated = false;
        if (count($rawRows) > $limit) {
            $rawRows = array_slice($rawRows, 0, $limit);
            $truncated = true;
        }

        $rows = array_map('ingresos_format_row', $rawRows);
        $summary = ingresos_build_summary($rows, $filters);
        $aggregates = ingresos_build_aggregates($rows);

        return [
            'filters' => $filters,
            'rows' => $rows,
            'summary' => $summary,
            'aggregates' => $aggregates,
            'limit' => $limit,
            'truncated' => $truncated,
        ];
    }
}

if (!function_exists('ingresos_build_summary')) {
    /**
     * Totales generales para cabeceras y tarjetas de resumen.
     */
    function ingresos_build_summary(array $rows, array $filters): array
    {
        if (!$rows) {
            return [
                'total_ingresos' => 0,
                'total_clientes' => 0,
                'total_depositos' => 0,
                'total_moviles' => 0,
                'total_pallets' => 0,
                'total_uv' => 0,
                'total_uc' => 0,
                'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
            ];
        }

        $clientes = [];
        $depositos = [];
        $moviles = [];
        $pallets = 0;
        $uv = 0;
        $uc = 0;

        foreach ($rows as $row) {
            if (!empty($row['cliente_id'])) {
                $clientes[(int) $row['cliente_id']] = true;
            }
            if (!empty($row['deposito_id'])) {
                $depositos[(int) $row['deposito_id']] = true;
            }
            if (!empty($row['movil_id'])) {
                $moviles[(int) $row['movil_id']] = true;
            }
            $pallets += (int) ($row['pallets'] ?? 0);
            $uv += (int) ($row['uv_cajas'] ?? 0);
            $uc += (int) ($row['uc_unidades'] ?? 0);
        }

        return [
            'total_ingresos' => count($rows),
            'total_clientes' => count($clientes),
            'total_depositos' => count($depositos),
            'total_moviles' => count($moviles),
            'total_pallets' => $pallets,
            'total_uv' => $uv,
            'total_uc' => $uc,
            'range_label' => $filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'],
        ];
    }
}

if (!function_exists('ingresos_build_aggregates')) {
    /**
     * Construye rankings y estadísticas por cliente, depósito y estado.
     */
    function ingresos_build_aggregates(array $rows): array
    {
        if (!$rows) {
            return [
                'clientes' => [],
                'depositos' => [],
                'estados' => [],
            ];
        }

        $clientes = [];
        $depositos = [];
        $estados = [];

        foreach ($rows as $row) {
            $clienteKey = (string) ($row['cliente_id'] ?? 0);
            if (!isset($clientes[$clienteKey])) {
                $clientes[$clienteKey] = [
                    'key' => $clienteKey,
                    'label' => $row['cliente_nombre'] ?? ($row['cliente_ref'] ?? 'Sin cliente'),
                    'rows' => 0,
                    'pallets' => 0,
                    'uv' => 0,
                    'uc' => 0,
                ];
            }
            $clientes[$clienteKey]['rows']++;
            $clientes[$clienteKey]['pallets'] += (int) ($row['pallets'] ?? 0);
            $clientes[$clienteKey]['uv'] += (int) ($row['uv_cajas'] ?? 0);
            $clientes[$clienteKey]['uc'] += (int) ($row['uc_unidades'] ?? 0);

            $depositoKey = (string) ($row['deposito_id'] ?? 0);
            if (!isset($depositos[$depositoKey])) {
                $depositos[$depositoKey] = [
                    'key' => $depositoKey,
                    'label' => $row['deposito_nombre'] ?? ($row['deposito_code'] ?? 'Sin depósito'),
                    'rows' => 0,
                    'pallets' => 0,
                    'uv' => 0,
                    'uc' => 0,
                ];
            }
            $depositos[$depositoKey]['rows']++;
            $depositos[$depositoKey]['pallets'] += (int) ($row['pallets'] ?? 0);
            $depositos[$depositoKey]['uv'] += (int) ($row['uv_cajas'] ?? 0);
            $depositos[$depositoKey]['uc'] += (int) ($row['uc_unidades'] ?? 0);

            $estadoCode = (string) ($row['estado_code'] ?? '');
            if ($estadoCode === '') {
                $estadoCode = 'SIN_ESTADO';
            }
            if (!isset($estados[$estadoCode])) {
                $estados[$estadoCode] = [
                    'code' => $estadoCode === 'SIN_ESTADO' ? null : $estadoCode,
                    'label' => $estadoCode === 'SIN_ESTADO' ? 'Sin estado' : ingresos_format_estado_label($estadoCode),
                    'rows' => 0,
                    'pallets' => 0,
                    'uv' => 0,
                    'uc' => 0,
                ];
            }
            $estados[$estadoCode]['rows']++;
            $estados[$estadoCode]['pallets'] += (int) ($row['pallets'] ?? 0);
            $estados[$estadoCode]['uv'] += (int) ($row['uv_cajas'] ?? 0);
            $estados[$estadoCode]['uc'] += (int) ($row['uc_unidades'] ?? 0);
        }

        $clientesList = array_values($clientes);
        usort($clientesList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            $cmp = $b['uc'] <=> $a['uc'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });
        $clientesList = array_slice($clientesList, 0, 10);

        $depositosList = array_values($depositos);
        usort($depositosList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            $cmp = $b['uc'] <=> $a['uc'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });
        $depositosList = array_slice($depositosList, 0, 10);

        $estadosList = array_values($estados);
        usort($estadosList, static function (array $a, array $b): int {
            $cmp = $b['rows'] <=> $a['rows'];
            if ($cmp !== 0) {
                return $cmp;
            }
            return strcmp((string) $a['label'], (string) $b['label']);
        });

        return [
            'clientes' => $clientesList,
            'depositos' => $depositosList,
            'estados' => $estadosList,
        ];
    }
}

if (!function_exists('ingresos_format_row')) {
    /**
     * Normaliza la fila cruda del SELECT para consumo del front.
     */
    function ingresos_format_row(array $row): array
    {
        $fechaIngreso = $row['fecha_ingreso'] ?? null;
        $fecha = $fechaIngreso ? substr((string) $fechaIngreso, 0, 10) : null;

        $llegada = $row['llegada_at'] ?? null;
        $llegadaHora = $llegada ? substr((string) $llegada, 0, 5) : null;
        $descargaInicio = $row['descarga_inicio_at'] ?? null;
        $descargaFin = $row['descarga_fin_at'] ?? null;

        $docTipo = trim((string) ($row['doc_tipo'] ?? ''));
        $docNumero = trim((string) ($row['doc_numero'] ?? ''));
        $docReferencia = '-';
        if ($docTipo !== '' && $docNumero !== '') {
            $docReferencia = $docTipo . ' ' . $docNumero;
        } elseif ($docNumero !== '') {
            $docReferencia = $docNumero;
        } elseif ($docTipo !== '') {
            $docReferencia = $docTipo;
        }

        $clienteNombre = trim((string) ($row['cliente_nombre'] ?? ''));
        if ($clienteNombre === '' && !empty($row['cliente_ref'])) {
            $clienteNombre = (string) $row['cliente_ref'];
        }
        if ($clienteNombre === '') {
            $clienteNombre = 'Sin cliente';
        }

        $depositoNombre = trim((string) ($row['deposito_nombre'] ?? ''));
        if ($depositoNombre === '' && !empty($row['deposito_code'])) {
            $depositoNombre = (string) $row['deposito_code'];
        }
        if ($depositoNombre === '') {
            $depositoNombre = 'Sin depósito';
        }

        $estadoCode = trim((string) ($row['estado_code'] ?? ''));
        $estadoNombre = $estadoCode !== '' ? ingresos_format_estado_label($estadoCode) : 'Sin estado';

        $observacion = $row['observacion'] ?? ($row['obervacion'] ?? null);
        if ($observacion !== null) {
            $observacion = trim((string) $observacion);
            if ($observacion === '') {
                $observacion = null;
            }
        }

        return [
            'ingreso_id' => (int) ($row['id'] ?? 0),
            'packinglist_id' => (int) ($row['packinglist_id'] ?? 0),
            'packinglist_codigo' => $row['packinglist_codigo'] ?? '',
            'fecha_ingreso' => $fecha,
            'hora_llegada' => $llegadaHora,
            'descarga_inicio' => $descargaInicio !== null ? substr((string) $descargaInicio, 0, 5) : null,
            'descarga_fin' => $descargaFin !== null ? substr((string) $descargaFin, 0, 5) : null,
            'cliente_id' => isset($row['cliente_id']) ? (int) $row['cliente_id'] : null,
            'cliente_nombre' => $clienteNombre,
            'cliente_ref' => $row['cliente_ref'] ?? null,
            'deposito_id' => isset($row['deposito_id']) ? (int) $row['deposito_id'] : null,
            'deposito_nombre' => $depositoNombre,
            'deposito_code' => $row['deposito_code'] ?? null,
            'estado_code' => $estadoCode !== '' ? $estadoCode : null,
            'estado_nombre' => $estadoNombre,
            'pallets' => (int) ($row['pallets'] ?? 0),
            'uv_cajas' => (int) ($row['uv_cajas'] ?? 0),
            'uc_unidades' => (int) ($row['uc_unidades'] ?? 0),
            'items' => (int) ($row['items'] ?? 0),
            'operarios_cant' => isset($row['operarios_cant']) ? (int) $row['operarios_cant'] : null,
            'movil_id' => isset($row['movil_id']) ? (int) $row['movil_id'] : null,
            'movil_chapa' => $row['movil_chapa'] ?? null,
            'chofer_id' => isset($row['chofer_id']) ? (int) $row['chofer_id'] : null,
            'chofer_nombre' => $row['chofer_nombre'] ?? null,
            'doc_tipo' => $docTipo !== '' ? $docTipo : null,
            'doc_numero' => $docNumero !== '' ? $docNumero : null,
            'doc_referencia' => $docReferencia,
            'doc_fecha' => $row['doc_fecha'] ?? null,
            'observacion' => $observacion,
            'created_at' => $row['created_at'] ?? null,
            'updated_at' => $row['updated_at'] ?? null,
        ];
    }
}

if (!function_exists('ingresos_normalize_filters')) {
    /**
     * Normaliza los filtros de entrada, aplicando defaults seguros.
     */
    function ingresos_normalize_filters(array $input): array
    {
        $defaultDesde = date('Y-m-d', strtotime('-30 days'));
        $defaultHasta = date('Y-m-d');

        $desde = ingresos_sanitize_date($input['fecha_desde'] ?? null, $defaultDesde);
        $hasta = ingresos_sanitize_date($input['fecha_hasta'] ?? null, $defaultHasta);

        if ($desde > $hasta) {
            [$desde, $hasta] = [$hasta, $desde];
        }

        return [
            'fecha_desde' => $desde,
            'fecha_hasta' => $hasta,
            'deposito_id' => ingresos_sanitize_numeric($input['deposito_id'] ?? null),
            'cliente_id' => ingresos_sanitize_numeric($input['cliente_id'] ?? null),
            'movil_id' => ingresos_sanitize_numeric($input['movil_id'] ?? null),
            'estado' => ingresos_sanitize_estado($input['estado'] ?? null),
            'codigo' => ingresos_sanitize_string($input['codigo'] ?? '', 80),
            'doc_numero' => ingresos_sanitize_string($input['doc_numero'] ?? '', 64),
        ];
    }
}

if (!function_exists('ingresos_sanitize_date')) {
    function ingresos_sanitize_date(?string $value, string $fallback): string
    {
        $value = trim((string) $value);
        if ($value !== '' && preg_match('/^20\d{2}-[01]\d-[0-3]\d$/', $value)) {
            return $value;
        }
        return $fallback;
    }
}

if (!function_exists('ingresos_sanitize_numeric')) {
    function ingresos_sanitize_numeric($value): string
    {
        if ($value === null || $value === '') {
            return '';
        }
        if (!is_numeric($value)) {
            return '';
        }
        $intVal = (int) $value;
        return $intVal > 0 ? (string) $intVal : '';
    }
}

if (!function_exists('ingresos_sanitize_string')) {
    function ingresos_sanitize_string(?string $value, int $maxLength): string
    {
        $value = trim((string) $value);
        if ($value === '') {
            return '';
        }
        if (strlen($value) > $maxLength) {
            $value = substr($value, 0, $maxLength);
        }
        return $value;
    }
}

if (!function_exists('ingresos_sanitize_estado')) {
    function ingresos_sanitize_estado(?string $value): string
    {
        $value = strtoupper(trim((string) $value));
        if ($value === '') {
            return '';
        }
        if (!preg_match('/^[A-Z_]{3,40}$/', $value)) {
            return '';
        }
        return $value;
    }
}

if (!function_exists('ingresos_format_estado_label')) {
    function ingresos_format_estado_label(string $code): string
    {
        static $map = [
            'CREADO' => 'Creado',
            'IMPORTADO' => 'Importado',
            'EN_DESCARGA' => 'En descarga',
            'DESCARGADO' => 'Descargado',
            'VALIDADO' => 'Validado',
            'CERRADO' => 'Cerrado',
        ];
        $code = strtoupper(trim($code));
        if ($code === '') {
            return 'Sin estado';
        }
        if (isset($map[$code])) {
            return $map[$code];
        }
        $label = str_replace('_', ' ', strtolower($code));
        return ucwords($label);
    }
}
